// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#include "MtgIncl.h"
#include "MtgLattice.h"
#include "MtgDtGenerator.h"
#include "MtgEvGenerator.h"
#include "MtgModel.h"
#include "MtgParser.h"
#include "MtgPortfolio.h"
#include "MtgScenario.h"
#include "MtgSignature.h"
#include "MtgSlotLayer.h"

MTG_BEGIN_NAMESPACE


//
//   i n i t
//

void tLattice::init()

{
    m_pModel = 0;
    m_pPortfolio = 0;
    m_pDtGen = 0;
    m_pTime = 0;
    reset();
}


//
//   p a r s e S h a p e
//

tRetCode tLattice::parseShape( tParser& Parser )

{
    MTG_ASSERT( Parser.curToken() == xTokShape ||
                Parser.curToken() == xTokBox ||
                Parser.curToken() == xTokTree );

    tRetCode nRet;

    if( Parser.curToken() == xTokShape ) {
        if( ( nRet = Parser.readToken() ) != OK )
            return nRet;
        if( Parser.curToken() != xTokBox &&
            Parser.curToken() != xTokTree ) {
            return Parser.setError( "box or tree shape expected" );
        }
    }

    switch( Parser.curToken() ) {
        case xTokBox :
            m_bIsBox = true;
            break;

        case xTokTree :
            m_bIsBox = false;
            break;

        default :
            throw tException( INTERNAL_ERROR );
    }

    if( ( nRet = Parser.readToken() ) != OK )
        return nRet;

    if( Parser.curToken() == xTokNumber ) {
        if( m_bIsTrimmed )
            return Parser.setError( "trimming already defined" );
        if( ( nRet = Parser.scanPosDouble( m_gTrimDev ) ) != OK )
            return nRet;
        m_bIsTrimmed = true;
    }

    return OK;
}


//
//   p a r s e M e t h o d
//

tRetCode tLattice::parseMethod( tParser& Parser )

{
    MTG_ASSERT( Parser.curToken() == xTokMethod ||
                Parser.curToken() == xTokExplicit ||
                Parser.curToken() == xTokImplicit );

    tRetCode nRet;

    if( Parser.curToken() == xTokMethod ) {
        if( ( nRet = Parser.readToken() ) != OK )
            return nRet;
        if( Parser.curToken() != xTokExplicit &&
            Parser.curToken() != xTokImplicit ) {
            return Parser.setError( "explicit or implicit method expected" );
        }
    }

    switch( Parser.curToken() ) {
        case xTokExplicit :
            m_nMethod = xExplicit;
            break;

        case xTokImplicit :
            m_nMethod = xImplicit;
            break;

        default :
            throw tException( INTERNAL_ERROR );
    }

    return Parser.readToken();
}


//
//   p a r s e T i m e S t e p
//

tRetCode tLattice::parseTimeStep( tParser& Parser )

{
    MTG_ASSERT( Parser.curToken() == xTokTimeStep );
    MTG_ASSERT( m_pDtGen == 0 );

    tRetCode nRet;
    double dt;

    if( ( nRet = Parser.readToken() ) != OK ||
        ( nRet = Parser.scanPosDouble( dt ) ) != OK ) {
        return nRet;
    }

    m_pDtGen = new tLinDtGenerator( dt, dt, 1 );
    return OK;
}


//
//   p a r s e P a r a m
//

tRetCode tLattice::parseParam( tParser& Parser, tParseInfoStub& Info )

{
    tRetCode nRet;
    tObject* pObj;
    tModel* pModel;
    tPortfolio* pPf;

    tParseInfo& I = static_cast<tParseInfo&>( Info );

    switch( Parser.curToken() ) {
        case xTokModel :
            if( m_pModel != 0 )
                return Parser.setError( "model already defined" );
            if( ( nRet = Parser.readToken() ) != OK )
                return nRet;
            if( Parser.curToken() != xTokId )
                return Parser.setError( "model id expected" );
            if( ( pObj = Parser.findObject() ) == 0 )
                return Parser.setError( "model object does not exist" );
            if( ( pModel = dynamic_cast<tModel*>( pObj ) ) == 0 ) 
                return Parser.setError( "model object expected" );
            setObjRefToZeroOrSo( m_pModel, pModel );
            if( ( nRet = Parser.readToken() ) != OK )
                return nRet;
            break;

        case xTokPortfolio :
            if( m_pPortfolio != 0 )
                return Parser.setError( "portfolio already defined" );
            if( ( nRet = Parser.readToken() ) != OK )
                return nRet;
            if( Parser.curToken() != xTokId )
                return Parser.setError( "portfolio id expected" );
            if( ( pObj = Parser.findObject() ) == 0 )
                return Parser.setError( "portfolio object does not exist" );
            if( ( pPf = dynamic_cast<tPortfolio*>( pObj ) ) == 0 ) 
                return Parser.setError( "portfolio object expected" );
            setObjRefToZeroOrSo( m_pPortfolio, pPf );
            if( ( nRet = Parser.readToken() ) != OK )
                return nRet;
            break;

        case xTokMaturity :
            if( m_nMaturity > 0 )
                return Parser.setError( "maturity already defined" );
            if( ( nRet = Parser.readToken() ) != OK ||
                ( nRet = Parser.scanInteger( m_nMaturity, 1 ) ) != OK ) {
                return nRet;
            }
            break;

        case xTokTrimDev :
            if( m_bIsTrimmed )
                return Parser.setError( "trimming already defined" );
            if( ( nRet = Parser.readToken() ) != OK ||
                ( nRet = Parser.scanPosDouble( m_gTrimDev ) ) != OK ) {
                return nRet;
            }
            m_bIsTrimmed = true;
            break;

        case xTokShape :
        case xTokBox :
        case xTokTree :
            if( I.m_bShape )
                return Parser.setError( "shape already defined" );
            if( ( nRet = parseShape( Parser ) ) != OK )
                return nRet;
            I.m_bShape = true;
            break;

        case xTokMethod :
        case xTokExplicit :
        case xTokImplicit :
            if( I.m_bMethod )
                return Parser.setError( "method already defined" );
            if( ( nRet = parseMethod( Parser ) ) != OK )
                return nRet;
            I.m_bMethod = true;
            break;

        case xTokTimeStep :
            if( I.m_bTimeStep )
                return Parser.setError( "timestep already defined" );
            if( ( nRet = parseTimeStep( Parser ) ) != OK )
                return nRet;
            I.m_bTimeStep = true;
            break;

        default :
            if( ( nRet = super::parseParam( Parser, Info ) ) != OK )
                return nRet;
    }

    return OK;
}


//
//   p a r s e P o s t f i x
//

tRetCode tLattice::parsePostfix( tParser& Parser, tParseInfoStub& Info )

{
    if( m_pPortfolio == 0 && m_nMaturity <= 0 )
        return Parser.setError( MISSING_MATURITY );
    return super::parsePostfix( Parser, Info );
}


//
//   t L a t t i c e
//

tLattice::tLattice()

{
    init();
}


//
//   t L a t t i c e
//

tLattice::tLattice( const tLattice& Lattice )

{
    init();
    copyFrom( Lattice );
}


//
//   ~ t L a t t i c e
//

tLattice::~tLattice()

{
    reset();
}


//
//   r e s e t
//

void tLattice::reset()

{
    setObjRefToZeroOrSo( m_pModel );
    setObjRefToZeroOrSo( m_pPortfolio );

    if( m_pDtGen != 0 ) {
        delete m_pDtGen;
        m_pDtGen = 0;
    }

    m_Bounds.reset();

    if( m_pTime != 0 ) {
        delete m_pTime;
        m_pTime = 0;
    }

    for( int i = 0; i < m_Space.numOfElems(); ++i )
        delete m_Space[i];
    m_Space.reset();

    m_nMaturity = -1;
    m_bIsBox = true;
    m_bIsTrimmed = false;
    m_gTrimDev = 3.5;
    m_nMethod = xExplicit;
    //m_nMethod = xImplicit;

    setProfile();

    touch();
}


//
//   o p e r a t o r =
//

tLattice& tLattice::operator=( const tLattice& Lattice )

{
    if( &Lattice != this )
        copyFrom( Lattice );
    return *this;
}


//
//   c o p y F r o m
//

void tLattice::copyFrom( const tLattice& Lattice )

{
    if( &Lattice == this )
        return;

    reset();

    setObjRefToZeroOrSo( m_pModel, Lattice.m_pModel );
    setObjRefToZeroOrSo( m_pPortfolio, Lattice.m_pPortfolio );

    if( Lattice.m_pDtGen != 0 )
        m_pDtGen = Lattice.m_pDtGen->clone();
    if( Lattice.m_pTime != 0 )
        m_pTime = Lattice.m_pTime->clone();

    m_Bounds = Lattice.m_Bounds;

    m_Space.numOfElems( Lattice.m_Space.numOfElems() );
    for( int i = 0; i < m_Space.numOfElems(); ++i )
        m_Space[i] = Lattice.m_Space[i]->clone();

    m_nMaturity = Lattice.m_nMaturity;
    m_bIsBox = Lattice.m_bIsBox;
    m_bIsTrimmed = Lattice.m_bIsTrimmed;
    m_gTrimDev = Lattice.m_gTrimDev;
    m_nMethod = Lattice.m_nMethod;

    tWithProfile::copyFrom( Lattice );
    super::copyFrom( Lattice );
}


//
//   c l o n e
//

tObject* tLattice::clone() const

{
    return new tLattice( *this );
}


//
//   f i n a l i z e
//

tRetCode tLattice::finalize( const tScenario* pScenario )

{
    MTG_ASSERT( m_pModel != 0 );
    MTG_ASSERT( m_pPortfolio != 0 || m_nMaturity > 0 );

    tRetCode nRet;
    double gMaxDt;
    tEvGenerator* pEvGen;

    if( m_pPortfolio != 0 ) {
        int n = m_pPortfolio->maturity();
        if( n > m_nMaturity )
            m_nMaturity = n;
    }

    if( pScenario != 0 )
        pEvGen = pScenario->getEvGenerator( m_nMaturity );
    else
        pEvGen = 0;

    if( isFinalized() && pEvGen == 0 )
        return OK;

    for( int i = 0; i < m_Space.numOfElems(); ++i )
        delete m_Space[i];
    m_Space.reset();

    if( m_pTime != 0 ) {
        delete m_pTime;
        m_pTime = 0;
    }

    if( m_pDtGen != 0 )
        gMaxDt = m_pDtGen->maxDt( m_nMaturity );
    else
        gMaxDt = 1;  // Default: 1 day timestep.

    if( ( nRet = m_pModel->createSpaceAxis( m_nMethod, gMaxDt, m_Space,
                    m_pPortfolio ) ) != OK ) {
        if( pEvGen != 0 )
            delete pEvGen;
        return nRet;
    }

    m_pTime = new tTimeAxis;
    m_pTime->setSpaceAxis( &m_Space );

    if( m_pPortfolio != 0 )
        m_pTime->setPortfolio( m_pPortfolio );
    if( m_nMaturity > 0 )
        m_pTime->setMaturity( m_nMaturity );
    if( m_pDtGen != 0 )
        m_pTime->setDtGenerator( m_pDtGen );
    if( pEvGen != 0 )
        m_pTime->setEvGenerator( pEvGen );

    for( MTG_FOR_INIT( int ) i = 0; i < m_Space.numOfElems(); ++i ) {
        m_Space[i]->setTimeAxis( m_pTime );
        m_Space[i]->setBox( m_bIsBox );
        m_Space[i]->setTrimming( m_bIsTrimmed );
        m_Space[i]->setTrimDev( m_gTrimDev );
        m_Space[i]->finalize();
    }

    m_pTime->finalize();

    m_Bounds.reset();
    for( MTG_FOR_INIT( int ) i = 0; i < m_Space.numOfElems(); ++i )
        m_Space[i]->append( m_Bounds );

    if( pEvGen != 0 )
        delete pEvGen;
    return super::finalize();
}


//
//   f i n a l i z e
//

tRetCode tLattice::finalize()

{
    return finalize( 0 );
}


//
//   b e f o r e R u n
//

tRetCode tLattice::beforeRun()

{
    MTG_ASSERT( isFinalized() );

    if( hasProfile() ) {
        try { 
            profile().putLattice( m_Bounds, *m_pTime, m_Space );
        }
        catch( tException e ) {
            return e.ret();
        }
    }

    return OK;
}


//
//   c r e a t e O F S o l v e r
//

tOFSolver* tLattice::createOFSolver( int nFactor )

{
    return m_Space[nFactor]->createOFSolver();
}


//
//   c r e a t e A u x L a y e r
//

tRetCode tLattice::createAuxLayer( tPortfolio* pPf,
    tSlotLayer*& pLayer ) const

{
    MTG_ASSERT( isFinalized() );
    MTG_ASSERT( pPf != 0 );

    tRetCode nRet;
    tArrayBounds Bounds;

    pLayer = new tSlotLayer;

    pLayer->setPortfolio( pPf->m_aAuxByBirth );
    pLayer->setBounds( m_Bounds );
    tWithProfile::copyTo( pLayer );

    if( ( nRet = pLayer->finalize() ) != OK ) {
        delete pLayer;
        return nRet;
    }

    return OK;
}


//
//   c r e a t e N o n A u x L a y e r
//

tRetCode tLattice::createNonAuxLayer( tPortfolio* pPf,
    const tSignature* pSig, tSlotLayer*& pLayer ) const

{
    MTG_ASSERT( isFinalized() );
    MTG_ASSERT( pPf != 0 );

    tRetCode nRet;

    pLayer = new tSlotLayer;

    pLayer->setPortfolio( pPf->m_Sorted );
    pLayer->setSignature( pSig );
    pLayer->setBounds( m_Bounds );

    tWithProfile::copyTo( pLayer );

    if( ( nRet = pLayer->finalize() ) != OK ) {
        delete pLayer;
        return nRet;
    }

    return OK;
}


//
//   c r e a t e N o n A u x L a y e r
//

tRetCode tLattice::createNonAuxLayer( tPortfolio* pPf,
    int nTag, tSlotLayer*& pLayer ) const

{
    tSignature Sig( pPf->m_Sorted.numOfElems() );

    for( int i = 0; i < pPf->m_Sorted.numOfElems(); ++i )
        Sig.on( i );
    Sig.setTag( nTag );
    return createNonAuxLayer( pPf, &Sig, pLayer );
}


//
//   p a r s e
//

tRetCode tLattice::parse( tParser& Parser, tSystem& System, tObject*& pObj )

{
    tRetCode nRet;
    tLattice* pLattice;
    tParseInfo Info;

    Info.m_bShape = false;
    Info.m_bMethod = false;
    Info.m_bTimeStep = false;

    if( ( nRet = Parser.scanBeginOfObj() ) != OK )
        return nRet;

    pLattice = new tLattice;
    pLattice->setSystem( System );

    if( ( nRet = pLattice->super::parse( Parser, &Info ) ) != OK ) {
        delete pLattice;
        return nRet;
    }
    pObj = pLattice;

    return OK;
}

MTG_END_NAMESPACE
