// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#if ! defined(_MTG_LATTICE_)
#define _MTG_LATTICE_

#include "MtgArrayBounds.h"
#include "MtgHeap.h"
#include "MtgObject.h"
#include "MtgProfile.h"
#include "MtgSpaceAxis.h"
#include "MtgTimeAxis.h"

MTG_BEGIN_NAMESPACE
   
class tDtGenerator;
class tModel;
class tOFSolver;
class tParser;
class tPortfolio;
class tScenario;
class tSignature;
class tSlotLayer;


//
//   t L a t t i c e
//
    
class tLattice : public tObject, public tWithProfile {

    typedef tObject super;

    struct tParseInfo : super::tParseInfoStub {
        bool m_bShape;
        bool m_bMethod;
        bool m_bTimeStep;
    };

    tModel* m_pModel;          // reference
    tPortfolio* m_pPortfolio;  // reference

    int m_nMaturity;

    tDtGenerator* m_pDtGen;    // owned

    bool m_bIsBox;
    bool m_bIsTrimmed;
    double m_gTrimDev;

    tFDMethod m_nMethod;

    tArrayBounds m_Bounds;

    tHeap<tSpaceAxis*> m_Space;
    tTimeAxis* m_pTime;

    void init();

    tRetCode parseShape( tParser& Parser );
    tRetCode parseMethod( tParser& Parser );
    tRetCode parseTimeStep( tParser& Parser );

    tRetCode parseParam( tParser& Parser, tParseInfoStub& Info );
    tRetCode parsePostfix( tParser& Parser, tParseInfoStub& Info );

public:

    tLattice();
    tLattice( const tLattice& Lattice );

    ~tLattice();

    void reset();

    tLattice& operator=( const tLattice& Lattice );

    void copyFrom( const tLattice& Lattice );

    tObject* clone() const;

    tRetCode finalize( const tScenario* pScenario );
    tRetCode finalize();

    tRetCode beforeRun();

    tModel* model() const {
        return m_pModel;
    }

    int dimension() const {
        return m_Space.numOfElems();
    }

    tFDMethod method() const {
        return m_nMethod;
    }

    bool isExplicit() const {
        return m_nMethod == xExplicit;
    }

    bool isImplicit() const { 
        return m_nMethod == xImplicit;
    }

    int numOfSlices() const {
        return m_pTime->numOfSlices();
    }

    int day( int nSlice ) const {
        return m_pTime->day( nSlice );
    }

    double fractionOfDay( int nSlice ) const {
        return m_pTime->fractionOfDay( nSlice );
    }

    double absDuration( int nSlice ) const {
        return m_pTime->absDuration( nSlice );
    }

    double relDuration( int nSlice ) const {
        return m_pTime->relDuration( nSlice );
    }

    double factor( int nFactor, int nLevel ) const {
        return m_Space[nFactor]->factor( nLevel );
    }

        // Solver in one direction (OF = one factor):

    tOFSolver* createOFSolver( int nFactor );

        // Directional derivatives:

    double calcDelta( int nFactor, int nLevel, double gVU, double gVM,
        double gVD ) const {
        return m_Space[nFactor]->calcDelta( nLevel, gVU, gVM, gVD );
    }

    double calcGamma( int nFactor, int nLevel, double gVU, double gVM,
        double gVD ) const {
        return m_Space[nFactor]->calcGamma( nLevel, gVU, gVM, gVD );
    }

    int numOfUpLevels( int nFactor, int nSlice ) const {
        return m_Space[nFactor]->numOfUpLevels( nSlice );
    }

    int numOfDownLevels( int nFactor, int nSlice ) const {
        return m_Space[nFactor]->numOfDownLevels( nSlice );
    }

    tRetCode createAuxLayer( tPortfolio* pPf, tSlotLayer*& pLayer ) const;
    tRetCode createNonAuxLayer( tPortfolio* pPf, const tSignature* pSig,
        tSlotLayer*& pLayer ) const;
    tRetCode createNonAuxLayer( tPortfolio* pPf, int nTag,
        tSlotLayer*& pLayer ) const;

    static tRetCode parse( tParser& Parser, tSystem& System, tObject*& pObj );
};

MTG_END_NAMESPACE

#endif
