// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#include "MtgIncl.h"

#define _MTG_LOOKUP_BODY_
#include "MtgLookup.h"
#undef _MTG_LOOKUP_BODY_

MTG_BEGIN_NAMESPACE


//
//   i n i t
//

template <class T>
void tLookup<T>::init()

{
    m_nHashSize = 0;
    reset();
}


//
//   g r o w H a s h
//

template <class T>
void tLookup<T>::growHash()

{
    int k, h, nNewSize, nOldSize;
    tEntry *aNew, *aOld;

    if( m_nNumOfEntries < ( m_nHashSize - 1 ) / 2 )
        return;

    if( m_nHashSize == 0 )
        nNewSize = NextPrime( 3 );
    else
        nNewSize = NextPrime( 2 * m_nHashSize + 1 );

    aNew = new tEntry[nNewSize];
    for( k = 0; k < nNewSize; ++k )
        aNew[k].m_sId = 0;

    nOldSize = m_nHashSize;
    aOld = m_aHash;

    m_nHashSize = nNewSize;
    m_aHash = aNew;

    if( nOldSize > 0 ) {
        for( k = 0; k < nOldSize; ++k ) {
            if( aOld[k].m_sId != 0 ) {
                h = find( aOld[k].m_sId );

                MTG_ASSERT( m_aHash[h].m_sId == 0 );

                m_aHash[h].m_sId = aOld[k].m_sId;
                m_aHash[h].m_Data = aOld[k].m_Data;
            }
        }
        delete[] aOld;
    }
}


//
//   h a s h
//

template <class T>
int tLookup<T>::hash( const char *sId ) const

{
    unsigned h, k;

    h = 0;
    k = 1;

    while( *sId ) {
        h += *sId++ * k++;
    }

    return (int) ( h % m_nHashSize );
}


//
//   f i n d
//

template <class T>
int tLookup<T>::find( const char *sId ) const

{
    int h, h1;

    h = hash( sId );
    h1 = 1;

    while( m_aHash[h].m_sId != 0 && strcmp( m_aHash[h].m_sId, sId ) != 0 ) {
            // note that we cannot have a table overflow, since
            // there are more than twice as many hash entries as
            // filled entries, and quadratic hashing covers
            // at least half of the hash table:
        MTG_ASSERT( h1 < m_nHashSize );

        h = h + h1;
        if( h >= m_nHashSize )
            h -= m_nHashSize;
        h1 += 2;
    }

    return h;
}


//
//   t L o o k u p
//

template <class T>
tLookup<T>::tLookup()

{
    init();
}


//
//   t L o o k u p
//

template <class T>
tLookup<T>::tLookup( const tLookup<T>& Lookup )

{
    init();
    copyFrom( Lookup );
}


//
//   ~ t L o o k u p
//

template <class T>
tLookup<T>::~tLookup()

{
    clear();
}


//
//   r e s e t
//

template <class T>
void tLookup<T>::reset()

{
    for( int k = 0; k < m_nHashSize; ++k ) {
        if( m_aHash[k].m_sId != 0 ) {
            delete m_aHash[k].m_sId;
            m_aHash[k].m_sId = 0;
        }
    }
    m_nNumOfEntries = 0;
}


//
//   c l e a r
//

template <class T>
void tLookup<T>::clear()

{
    reset();
    if( m_nHashSize > 0 ) {
        delete[] m_aHash;
        m_nHashSize = 0;
    }
}


//
//   o p e r a t o r =
//

template <class T>
tLookup<T>& tLookup<T>::operator=( const tLookup<T>& Lookup )

{
    if( &Lookup != this )
        copyFrom( Lookup );
    return *this;
}


//
//   c o p y F r o m
//

template <class T>
void tLookup<T>::copyFrom( const tLookup<T>& Lookup )

{
    if( &Lookup == this )
        return;

    if( Lookup.m_nHashSize == 0 ) {
        clear();
    }
    else {
        tEntry* aNew = new tEntry[Lookup.m_nHashSize];
           
        for( int k = 0; k < Lookup.m_nHashSize; ++k ) {
            if( Lookup.m_aHash[k].m_sId == 0 ) {
                aNew[k].m_sId = 0;
            }
            else {
                aNew[k].m_sId = StrCopy( Lookup.m_aHash[k].m_sId );
                aNew[k].m_Data = Lookup.m_aHash[k].m_Data;
            }
        }
   
        clear();
        m_aHash = aNew;
        m_nHashSize = Lookup.m_nHashSize;
        m_nNumOfEntries = Lookup.m_nNumOfEntries;
    }
}


//
//   i n s e r t
//

template <class T>
tRetCode tLookup<T>::insert( const char* sId, T& Data )

{
    MTG_ASSERT( sId != 0 );

    growHash();
    int nPos = find( sId );

    if( m_aHash[nPos].m_sId != 0 )
        return ID_REDEFINITION;

    m_aHash[nPos].m_sId = StrCopy( sId ); 
    m_aHash[nPos].m_Data = Data;
    ++m_nNumOfEntries;

    return OK;
}


//
//   r e p l a c e
//

template <class T>
tRetCode tLookup<T>::replace( const char* sId, T& Data )

{
    MTG_ASSERT( sId != 0 );

    growHash();
    int nPos = find( sId );

    if( m_aHash[nPos].m_sId != 0 ) {
        m_aHash[nPos].m_Data = Data;
    }
    else {
        m_aHash[nPos].m_sId = StrCopy( sId ); 
        m_aHash[nPos].m_Data = Data;
        ++m_nNumOfEntries;
    }

    return OK;
}


//
//   r e t r i e v e
//

template <class T>
tRetCode tLookup<T>::retrieve( const char* sId, T& Data ) const

{
    int h;

    if( m_nNumOfEntries == 0 )
        return NOT_FOUND;

    h = find( sId );
    if( m_aHash[h].m_sId == 0 )
        return NOT_FOUND;

    Data = m_aHash[h].m_Data;
    return OK;
}

#if defined(_TEST)

#if defined(_WIN32)
    #include <conio.h>
#endif


//
//   m a i n
//

void main( int argc, char *argv[] )

{
    bool bGo;
    tRetCode nRet;
    int cCmd, nData;
    char sId[32];
    tLookup<int> Lookup, Lookup1;

    printf( "\nTest tLookup\n\n" );

    bGo = true;
    while( bGo ) {
        printf( "<C>lear <I>nsert <R>etrieve E<x>it: " );
        cCmd = getche();
        printf( "\n" );

        nRet = OK;
        switch( cCmd ) {
        case 'c' :
        case 'C' :
            Lookup.clear();
            break;

        case 'i' :
        case 'I' :
            printf( "  Id Data: " );
            scanf( "%s%d", sId, &nData );
            nRet = Lookup.insert( sId, nData );  
            break;

        case 'r' :
        case 'R' :
            Lookup1 = Lookup;
            printf( "  Id: " );
            scanf( "%s", sId );
            nRet = Lookup1.retrieve( sId, nData );
            if( nRet == OK )
                printf( "  Data=%d\n", nData );
            break;

        case 'x' :
        case 'X' :
            bGo = false;
            break;
        }

        if( nRet != OK )
            printf( "  Error=%d\n", nRet );
    }
}

#endif

MTG_END_NAMESPACE
