// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#include "MtgIncl.h"
#include "MtgMCEngine.h"
#include "MtgClaim.h"
#include "MtgMCOptInstance.h"
#include "MtgModel.h"
#include "MtgOptimizer.h"
#include "MtgPathSpace.h"
#include "MtgPortfolio.h"
#include "MtgRandom.h"
#include "MtgSamplePath.h"

#include <float.h>

MTG_BEGIN_NAMESPACE


//
//   p r i c e
//

tRetCode tMCEngine::price()

{
    MTG_ASSERT( m_pEvolution != 0 );

    tSamplePath Path;
    tRandom Random;
    tRetCode nRet;

    tPortfolio& Pf = portfolio();

    m_pEvolution->setPath( Path );
    m_gDirtyTotal = 0;

    try {
        for( int i = 0; i < m_pPathSpace->size(); ++i ) {
            double gWeight;

            m_pPathSpace->generatePath( Path, Random, i, gWeight );

            if( ( nRet = preparePath() ) != OK )
                return nRet;

                // Evaluate all instruments on this path:

            for( int k = 0; k < Pf.numOfClaims(); ++k ) {
                tClaim& Claim = Pf.claim( k );
                double gMultiplier = multiplier( Claim.index() );

                    // First, the final payoff; this, we're
                    // discounting ourselves; secondly, ALL
                    // cashflows; these are discounted by
                    // the function.

                double gValue = gWeight * ( 
                    presentValue( settlement(), Claim.maturityDate() ) *
                        Claim.payoff( *this ) +
                    Claim.cashflow( *this ) );

                m_Gradient[Claim.index()] += gValue;
                m_gDirtyTotal += gMultiplier * gValue;
            }

                // Evaluate all curves on this path:

            if( ( nRet = curveContainer().evaluate( gWeight ) ) != OK )
                return nRet;
        }
    }
    catch( tException e ) {
        return e.ret();
    }

    m_gCleanTotal = m_gDirtyTotal;
    dirty2Clean( m_gCleanTotal );

    return OK;
}


//
//   o p t i m i z e 1
//

tRetCode tMCEngine::optimize1( tSamplePath& Path, tRandom& Random )

{
    MTG_ASSERT( m_pEvolution != 0 );

        // first, fill m_Value, the matrix that holds
        // unit values for each instrument and path or 
        // model parameter combination.

    tRetCode nRet;

    int nSlot;
    double gWeight;

    tPortfolio& Pf = portfolio();
    int n = Pf.numOfPricedClaims();

    if( optimizer().isVerbose() )
        optimizer().verbose() << "Computing payoff matrix" << endl;

    try {
        int nPath = 0;

        for( int nInst = 0; nInst < m_nNumOfInstantiations; ++nInst ) {
            m_pEvolution->setInstantiation( nInst );

            if( m_nNumOfInstantiations > 1 ) {
                nSlot = nInst;
                for( int k = 0; k <= n; ++k )
                    m_Value[nSlot][k] = 0;
            }

            for( int i = 0; i < m_pPathSpace->size(); ++i, ++nPath ) {
                if( optimizer().isVerbose() && nPath % 500 == 0 ) {
                    optimizer().verbose() << "\r... " 
                        << nPath / m_nNumOfInstantiations << " paths";
                }

                if( m_nNumOfInstantiations > 1 ) {
                    m_pPathSpace->generatePath( Path, Random, nPath, gWeight );
                }
                else {
                    m_pPathSpace->generatePath( Path, Random, nPath );
                    nSlot = i;
                }

                if( ( nRet = preparePath() ) != OK )
                    return nRet;

                    // evaluate all priced instruments:

                for( int k = 0; k < n; ++k ) {
                    tClaim& Claim = Pf.pricedClaim( k );
                    double gMultiplier = multiplier( Claim.index() );

                        // first, the final payoff; this, we're
                        // discounting ourselves

                    double gValue = Claim.payoff( *this ) *
                        presentValue( settlement(), Claim.maturityDate() );                        
    
                        // second, ALL cashflows; these are discounted
                        // by the function

                    gValue += Claim.cashflow( *this );

                    if( m_nNumOfInstantiations > 1 ) 
                        m_Value[nSlot][k] += gWeight * gValue;
                    else
                        m_Value[nSlot][k] = gValue;
                }

                    // now evaluate all non-priced instruments
                    // (these are multiplied with their multipliers
                    // right away, as these don't change):

                if( m_nNumOfInstantiations == 1 ) 
                    m_Value[nSlot][n] = 0;
                    
                for( MTG_FOR_INIT( int ) k = 0;
                        k < Pf.numOfNotPricedClaims(); ++k ) {

                    tClaim& Claim = Pf.notPricedClaim( k );
                    double gMultiplier = multiplier( Claim.index() );

                    double gValue = gMultiplier * (                    
                        presentValue( settlement(), Claim.maturityDate() ) *
                            Claim.payoff( *this ) +
                        Claim.cashflow( *this ) );

                    if( m_nNumOfInstantiations > 1 ) 
                        m_Value[nSlot][n] += gWeight * gValue;
                    else
                        m_Value[nSlot][n] += gValue;
                }
            }
        }

        if( optimizer().isVerbose() ) {
            optimizer().verbose()
                << "\r    " << m_pPathSpace->size()
                << " paths" << endl;
        }
    }
    catch( tException e ) {
        if( optimizer().isVerbose() )
            optimizer().verbose() << endl;
        return e.ret();
    }

    return OK;
}

    
//
//   o p t i m i z e 2
//
    
tRetCode tMCEngine::optimize2( tSamplePath& Path, tRandom& Random )

{
        // now create a Monte Carlo optimizer instance:

    tMCOptInstance* pOpt;
    tRetCode nRet;

    tPortfolio& Pf = portfolio();
    int n = Pf.numOfPricedClaims();

    if( ( nRet = optimizer().createInstance( this, pOpt ) ) != OK )
        return nRet;

    tMCOptInstance& Opt = *pOpt;

    if( ( nRet = Opt.prepare( m_Value, Pf ) ) != OK )
        return nRet;

        // correct multipliers one more time:

    Opt.injectMultipliers( multiplier() );

        // ...and use it:

    if( ( nRet = Opt.run( m_gDirtyTotal ) ) != OK )
        return nRet;

        // store lambdas and gradient back (notice that
        // the elements of the gradient do not make sense
        // for non-priced instruments)

    Opt.extractMultipliers( multiplier() );
    Opt.extractGradient( m_Gradient );

        // evaluate all the curves

    if( curveContainer().hasCurves() ) {
        tCurveContainer& Container = curveContainer();
        const tHeap<double>& Weight = Opt.weight();

        if( optimizer().isVerbose() )
            optimizer().verbose() << "Creating curves" << endl;

            // recreate all paths

        int nPath = 0;

        for( int nInst = 0; nInst < m_nNumOfInstantiations; ++nInst ) {
            if( m_nNumOfInstantiations > 1 && Weight[nInst] < DBL_EPSILON ) { 
                    // simply recreate the paths to keep the
                    // random number generator up-to-date
                    
                for( int i = 0; i < m_pPathSpace->size(); ++i, ++nPath ) {
                    if( optimizer().isVerbose() && nPath % 500 == 0 ) {
                        optimizer().verbose() << "\r... " 
                            << nPath / m_nNumOfInstantiations << " paths";
                    }
                    m_pPathSpace->generatePath( Path, Random, nPath );
                }
            }
            else {
                m_pEvolution->setInstantiation( nInst );

                for( int i = 0; i < m_pPathSpace->size(); ++i, ++nPath ) {
                    if( optimizer().isVerbose() && nPath % 500 == 0 ) {
                        optimizer().verbose() << "\r... " 
                            << nPath / m_nNumOfInstantiations << " paths";
                    }

                    if( m_nNumOfInstantiations > 1 ) {
                        double gWeight;

                        m_pPathSpace->generatePath( Path, Random,
                            nPath, gWeight );

                        if( ( nRet = preparePath() ) != OK ||
                            ( nRet = Container.evaluate( gWeight ) ) != OK ) {
                            return nRet;
                        }   
                    }
                    else {
                        m_pPathSpace->generatePath( Path, Random, nPath );
                        if( ( nRet = preparePath() ) != OK ||
                            ( nRet = Container.evaluate(
                                Weight[i] ) ) != OK ) {
                            return nRet;
                        }
                    }
                }

                if( m_nNumOfInstantiations > 1 ) {
                    if( ( nRet =
                            Container.postProcess( Weight[nInst] ) ) != OK ) {
                        return nRet;
                    }
                }
                else {
                    if( ( nRet = Container.postProcess() ) != OK )
                        return nRet;
                }
            }
        }

        if( optimizer().isVerbose() ) {
            optimizer().verbose()
                << "\r    " << m_pPathSpace->size()
                << " paths" << endl;
        }
    }

        // Save results to disk, if the optimizer supports it.
        // (The lambdas are saved by the optimizer itself.)

    if( m_nNumOfInstantiations > 1 ) {
        if( ( nRet = optimizer().savePathSpace( *m_pPathSpace ) ) != OK ||
            ( nRet = model().saveWeights( optimizer().outWeightFile(),
                    pOpt->weight() ) ) != OK ) {
                return nRet;
        }
    }
    else {
        if( ( nRet = optimizer().savePathSpace(
                *m_pPathSpace, pOpt->weight() ) ) != OK ||
            ( nRet = optimizer().saveWeights( pOpt->weight() ) ) != OK ) {
                return nRet;
        }
    }

    delete pOpt;

    m_gCleanTotal = m_gDirtyTotal;
    dirty2Clean( m_gCleanTotal );

    return OK;
}


//
//   o p t i m i z e
//

tRetCode tMCEngine::optimize()

{
    MTG_ASSERT( m_pEvolution != 0 );

    tSamplePath Path;
    tRandom Random;
    tRetCode nRet;

    m_pEvolution->setPath( Path );

    if( ( nRet = optimize1( Path, Random ) ) != OK ||
        ( nRet = optimize2( Path, Random ) ) != OK ) {
        return nRet;
    }

    return OK;
}


//
//   g e t C l a i m
//

void tMCEngine::getClaim( int nIndex, int nTag, double& gUnitValue )

{
    throw tException( NOT_AVAILABLE );
}


//
//   b e f o r e R u n
//

tRetCode tMCEngine::beforeRun()

{
    MTG_ASSERT( m_pEvolution == 0 );

    tRetCode nRet;

    if( ( nRet = super::beforeRun() ) != OK )
        return nRet;

    if( ! m_pPathSpace->modelMatches( model() ) )
        return OBJECT_MISMATCH;

    if( ! m_pPathSpace->isCovered( settlement() ) ||
        ! m_pPathSpace->isCovered( portfolio().maturityDate() ) ) {
        return OUT_OF_RANGE;
    }

    if( ( nRet = model().createEvolution( *m_pPathSpace,
                    m_pEvolution ) ) != OK ) {
        return nRet;
    }

    m_nNumOfInstantiations = model().numOfInstantiations();
    MTG_ASSERT( m_nNumOfInstantiations >= 1 );

    m_Gradient.numOfElems( multiplier().numOfElems() );
    m_Gradient.fill( 0 );

    if( hasOptimizer() ) {
            // Stronger test:
        if( m_pPathSpace->base() != settlement() )
            return SETTLEMENT_CONFLICT;

            // The first elements in m_Value are indexed with
            // 0, ..., k - 1, where k is the number of instruments
            // in the portfolio that are priced. There is always
            // a last column for the instruments which are not
            // priced, although this column may always be 0.

        m_Value.reset( portfolio().numOfPricedClaims() + 1 );

        if( m_nNumOfInstantiations > 1 )
            m_Value.numOfRows( m_nNumOfInstantiations );
        else
            m_Value.numOfRows( m_pPathSpace->size() );
    }

    return OK;
}


//
//   a f t e r R u n
//

tRetCode tMCEngine::afterRun()

{
    if( m_pEvolution != 0 ) {
        delete m_pEvolution;
        m_pEvolution = 0;
    }

    return super::afterRun();
}    


//
//   p r e p a r e P a t h
//

tRetCode tMCEngine::preparePath()

{
    return OK;
}


//
//   t M C E n g i n e
//

tMCEngine::tMCEngine()

{
    m_pPathSpace = 0;
    m_pEvolution = 0;
}


//
//   ~ t M C E n g i n e
//

tMCEngine::~tMCEngine()

{
    if( m_pEvolution != 0 )
        delete m_pEvolution;
}


//
//   s e t P a t h S p a c e
//

void tMCEngine::setPathSpace( tPathSpace* pPathSpace )

{
    MTG_ASSERT( pPathSpace != 0 );
    m_pPathSpace = pPathSpace;
}


//
//   r u n
//

tRetCode tMCEngine::run()

{
    MTG_ASSERT( m_pPathSpace != 0 );
    MTG_ASSERT( &m_pPathSpace->model() == &model() );

    tRetCode nRet;

    if( ( nRet = beforeRun() ) != OK )
        return nRet;

    if( hasOptimizer() ) {
        if( ( nRet = optimize() ) != OK )
            return nRet;
    }
    else {
        if( ( nRet = price() ) != OK )
            return nRet;
    }

    return afterRun();
}


//
//   g r a d i e n t
//

void tMCEngine::gradient( tHeap<double>& Gradient ) const

{
    Gradient = m_Gradient;
}

MTG_END_NAMESPACE

