// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#if ! defined(_MTG_MCENGINE_)
#define _MTG_MCENGINE_

#include "MtgEngine.h"
#include "MtgHeap2.h"

MTG_BEGIN_NAMESPACE

class tModel;
class tPathSpace;
class tPortfolio;
class tRandom;
class tSamplePath;


//
//   t M S E n g i n e
//

class tMCEngine : public tEngine {

    typedef tEngine super;

        // evolution stub for path transfer between objects

public:

    class tEvolutionStub {

        const tSamplePath* m_pPath;

    protected:

        const tSamplePath& path() const {
            return *m_pPath;
        }

    public:

        virtual ~tEvolutionStub() {
        }

        virtual void setPath( const tSamplePath& Path ) {
            m_pPath = &Path;
        }

        virtual void setInstantiation( int nInstantiation ) = 0;
    };

private:

    tPathSpace* m_pPathSpace;       // referenced
    tEvolutionStub* m_pEvolution;   // owned

        // The m_Value array represents the values of the
        // individual instruments that are priced, and the
        // cumulative value of all instruments that are
        // not priced in the last column (the latter weighed
        // with their multipliers). It is only used for
        // optimization, since pricing can be done without
        // storing all this information.

    tHeap2<double> m_Value;

        // The gradient, on the other hand, is indexed with
        // the instrument index, and ALL instruments have
        // an entry. For instruments that are not priced,
        // the entry is simply set to zero. The gradient
        // is also computed in pricing mode.

    tHeap<double> m_Gradient;

        // The total is computed at the end and is just the
        // total value over the (modified) multipliers.

    double m_gDirtyTotal;
    double m_gCleanTotal;

        // If the number of model instantiations is 1, the
        // weighing is done for the individual paths; if it
        // is more than one, the weighing is done on a
        // macro-level, for each combination of model
        // parameters.

    int m_nNumOfInstantiations;

    tRetCode price();

    tRetCode optimize1( tSamplePath& Path, tRandom& Random );
    tRetCode optimize2( tSamplePath& Path, tRandom& Random );
    tRetCode optimize();

protected:

    tEvolutionStub& evolution() const {
        return *m_pEvolution;
    }

    void getClaim( int nIndex, int nTag, double& gUnitValue );

        // If these functions are overridden, kepp in mind that
        // beforeRun() has to be called FIRST in the overridden
        // version, and afterRun() LAST.

    tRetCode beforeRun();
    tRetCode afterRun();

        // Subclasses get a chance to prepare themselves before
        // instruments that get evaluated for one path. Preparation
        // could include evolution of the initial forward rate
        // curve, for instance.

    virtual tRetCode preparePath();
    
public:

    tMCEngine();

    virtual ~tMCEngine();

    void setPathSpace( tPathSpace* pPathSpace );
    void setPathSpace( tPathSpace& PathSpace ) {
        setPathSpace( &PathSpace );
    }

    tPathSpace& pathSpace() const {
        MTG_ASSERT( m_pPathSpace != 0 );
        return *m_pPathSpace;
    }

    tRetCode run();

    double cleanTotal() const {
        return m_gCleanTotal;
    }

    double dirtyTotal() const {
        return m_gDirtyTotal;
    }

    double total() const {
        return dirtyTotal();
    }

        // Notice that the gradient corresponds always to
        // the "dirty" total.

    void gradient( tHeap<double>& Gradient ) const;
};

MTG_END_NAMESPACE

#endif
