// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#include "MtgIncl.h"

#define _MTG_MAP_BODY_
#include "MtgMap.h"
#undef _MTG_MAP_BODY_

MTG_BEGIN_NAMESPACE


//
//   i n i t
//

template<class T, int nol, int ff>
void tMap<T,nol,ff>::init()

{
    m_aFoldingFactor[0] = 1;
    for( int k = 1; k < nol; ++k )
        m_aFoldingFactor[k] = m_aFoldingFactor[k - 1] * ff;
    reset();
}


//
//   t M a p
//

template<class T, int nol, int ff>
tMap<T,nol,ff>::tMap()

{
    init();
}


//
//   t M a p
//

template<class T, int nol, int ff>
tMap<T,nol,ff>::tMap( const tMap<T,nol,ff>& Map )

{
    init();
    copyFrom( Map );
}


//
//   ~ t M a p
//

template<class T, int nol, int ff>
tMap<T,nol,ff>::~tMap()

{
}


//
//   r e s e t
//

template<class T, int nol, int ff>
void tMap<T,nol,ff>::reset()

{
    m_nNumOfElems = 0;
    for( int k = 0; k < nol; ++k )
        m_aLayer[k].reset();
    m_aData.reset();
}


//
//   c l e a r
//

template<class T, int nol, int ff>
void tMap<T,nol,ff>::clear()

{
    m_nNumOfElems = 0;
    for( int k = 0; k < nol; ++k )
        m_aLayer[k].clear();
    m_aData.clear();
}


//
//   o p e r a t o r =
//

template<class T, int nol,int ff>
tMap<T,nol,ff>& tMap<T,nol,ff>::operator=( const tMap<T,nol,ff>& Map )

{
    if( &Map != this )
        copyFrom( Map );
    return *this;
}


//
//   c o p y F r o m
//

template<class T, int nol, int ff>
void tMap<T,nol,ff>::copyFrom( const tMap<T,nol,ff>& Map )

{
    if( &Map == this )
        return;

    reset();
    for( int k = 0; k < nol; ++k )
        m_aLayer[k].copyFrom( Map.m_aLayer[k] );

    m_aData.copyFrom( Map.m_aData );
    m_nNumOfElems = Map.m_nNumOfElems;
}


//
//   m i g r a t e T o
//

template<class T, int nol, int ff>
void tMap<T,nol,ff>::migrateTo( tMap<T,nol,ff>& Map )

{
    if( &Map == this )
        return;

    Map.reset();
    for( int k = 0; k < nol; ++k )
        m_aLayer[k].migrateTo( Map.m_aLayer[k] );

    m_aData.migrateTo( Map.m_aData );
    Map.m_nNumOfElems = m_nNumOfElems;

    m_nNumOfElems = 0;
}


//
//   a p p e n d
//

template<class T, int nol, int ff>
void tMap<T,nol,ff>::append( int nToPos, T& Data )

{
    MTG_ASSERT( nToPos >= m_nNumOfElems );

    int k, n1, n2, nRelPos, nOfs, nSlot, nLayer, nLastLayer,
        nData, nFactor, nNumOfBlocks;
    bool bEqual;

    nData = m_aData.numOfElems() - 1;
    if( m_nNumOfElems > 0 &&
            memcmp( &m_aData[nData], &Data, sizeof(T) ) == 0 ) {
        bEqual = true;
    }
    else {
        ++m_aData;
        ++nData;
        m_aData[nData] = Data;
        bEqual = false;
    }

    nLastLayer = nol - 1;
    nLayer = nLastLayer;
    nRelPos = m_nNumOfElems;
    nOfs = 0;

    while( true ) {
        nSlot = nRelPos / m_aFoldingFactor[nLayer] - nOfs;
        nRelPos %= m_aFoldingFactor[nLayer];

        if( nLayer < nol - 1 ) {
            if( nRelPos == 0 )
                m_aLayer[nLayer][nSlot] = nData;

            k = nSlot;
            while( ++k % m_aFoldingFactor[1] > 0 )
                m_aLayer[nLayer][k] = nData;
        }

        if( nRelPos == 0 )
            break;
       
        nOfs = m_aLayer[nLayer][nSlot];
        if( nOfs < 0 ) {
            ++nOfs;
            --nLayer;
        }
        else {
            if( bEqual ) {
                break;
            }
            else {
                --nLayer;

                n1 = m_aLayer[nLayer].numOfElems();
                n2 = n1 + m_aFoldingFactor[1];
                m_aLayer[nLayer].numOfElems( n2 );
                for( k = n1; k < n2; ++k )
                    m_aLayer[nLayer][k] = nOfs;

                nOfs = -n1;
                m_aLayer[nLayer + 1][nSlot] = nOfs - 1;
            }
        }
    }

    nRelPos = m_nNumOfElems;
    nFactor = m_aFoldingFactor[nLastLayer];

    if( nRelPos % nFactor != 0 )
        nRelPos = ( nRelPos / nFactor + 1 ) * nFactor;

    if( nToPos >= nRelPos ) {
        nNumOfBlocks = m_aLayer[nLastLayer].numOfElems() +
            ( nToPos - nRelPos ) / nFactor + 1;
        k = m_aLayer[nLastLayer].numOfElems();
        m_aLayer[nLastLayer].numOfElems( nNumOfBlocks );
        while( k < nNumOfBlocks )
            m_aLayer[nLastLayer][k++] = nData;
    }

    m_nNumOfElems = nToPos + 1;
}


//
//   o p e r a t o r [ ]
//

template<class T, int nol, int ff>
T& tMap<T,nol,ff>::operator[]( int nPos ) const

{
    MTG_ASSERT( nPos >= 0 && nPos < m_nNumOfElems );

    int nOfs, nSlot, nLayer;

    nLayer = nol - 1;
    nSlot = nPos / m_aFoldingFactor[nLayer];

    while( ( nOfs = m_aLayer[nLayer][nSlot] ) < 0 ) {
        nPos %= m_aFoldingFactor[nLayer];
        nSlot = nPos / m_aFoldingFactor[--nLayer] - nOfs - 1;
    }

    return m_aData[nOfs];
}


#if defined(_TEST)

//
//   m a i n
//

void main( int argc, char *argv[] )

{
    tMap<int, 3, 5> Map;

    Map.reset();
    for( int k = 0; k < 70; ++k ) {
        int nData = k / 5;
        Map.append( k, nData );
    }

    printf( "%d elements\n", Map.numOfElems() );
    for( MTG_FOR_INIT( int ) k = 0; k < Map.numOfElems(); ++k ) {
        printf( "%d ", Map[k] );
    }
    printf( "\n" );
    getchar();
}

#endif

MTG_END_NAMESPACE
