// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#if ! defined(_MTG_MAP_)
#define _MTG_MAP_

#include "MtgHeap.h"

MTG_BEGIN_NAMESPACE

// In MTG, arrays often consist of large contiguous sections of
// identical data. For instance, the forward volatility has
// to be accessible per day, which would lead to 30*365=10950
// array entries, the size of each of which would be at least
// 24 Byte (band + prior). 256 KB just for that purpose sound
// a little too wasteful (they have to be initialized, too),
// since the forward vol is pretty stable over time.
//
// A map is a special data-structure that stores only the
// unique data and superimposes a tree-like access-structure
// that still allows fast access time for each day. The structure
// works like a quadtree in one dimension. The maximum depth
// of the tree (m_nNumOfLayers) can be specified (there's a limit,
// though), and the folding factor that determines the magnification
// from one level to the next is also variable. For instance, a
// folding factor of 10 and 3 access levels (magnifications of
// 1, 10 and 100 at each level, respectively) can reduce the
// amount of space needed for the forward vol curve by over
// 90%, assuming that the number of sample points is in the single
// digits.
//
// Maps can only be created by appending values. Modifications are
// not allowed. Given the applications within MTG, this is a
// reasonable restriction.
//
// Maps are meant as a container for elementary C-style objects
// that don't need constructor and destructor care. Comparison
// is simply done with the library function memcmp().


//
//   t M a p
//

template<class T, int numOfLayers = 3, int foldingFactor = 8>
class tMap {

        // The number of elements accessible in the map. To the outside, the
        // map behaves like a plain tHeap. m_nNumOfElems is the number that
        // would be reported by the heap as the number of elements.

    int m_nNumOfElems;

    tHeap<int> m_aLayer[numOfLayers];
    int m_aFoldingFactor[numOfLayers];

    tHeap<T> m_aData;

    void init();

public:

    tMap();
    tMap( const tMap& Map );

    ~tMap();

    void reset();
    void clear();

    tMap& operator=( const tMap& Map );

    void copyFrom( const tMap& Map );
    void migrateTo( tMap& Map );

        // It is only possible to append data. The following function
        // appends nToPos - m_nNumOfElems + 1 copies of Data at the
        // end of the map. (Of course, Data is copied only once. And
        // if the current last element of the map is equal to Data,
        // it is completely ignored.)

    void append( int nToPos, T& Data );

    T& operator[]( int nPos ) const;

    operator int() const {
        return m_nNumOfElems;
    }

    int numOfElems() const {
        return m_nNumOfElems;
    }
};

MTG_END_NAMESPACE

#if ! defined(_MTG_MAP_BODY_)
    #define _MTG_MAP_HEADER_
    #if defined(_WIN32)
        #include "MtgMap.cpp"
    #else   
        #include "MtgMap.C"
    #endif
    #undef _MTG_MAP_HEADER_
#endif

#endif
