// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#include "MtgIncl.h"
#include "MtgMinimizer.h"

#include <string.h>
#include <float.h>

extern "C" {

#if defined(min)
#undef min
#endif

#if defined(max)
#undef max
#endif

#include "f2c.h"

int lbfgsb_( integer *n, integer *m, doublereal *x,
    doublereal *l, doublereal *u, integer *nbd, doublereal *f,
    doublereal *g, doublereal *factr, doublereal *wa, integer *iwa,
    char *task, integer *iprint, integer *isbmin, char *csave,
    logical *lsave, integer *isave, doublereal *dsave,
    ftnlen task_len, ftnlen csave_len );
}

MTG_BEGIN_NAMESPACE


//
//   c o p y F r o m
//

void tMinimizer::copyFrom( const tMinimizer& Minimizer )

{
    if( &Minimizer == this )
        return;

    m_gTolerance = Minimizer.m_gTolerance;
    m_bVerbose = Minimizer.m_bVerbose;
    m_gResult = Minimizer.m_gResult;
}


//
//   t M i n i m i z e r
//

tMinimizer::tMinimizer()

{
    m_gTolerance = 1e-6;
    m_gResult = 0;

#if defined(_DEBUG)
    m_bVerbose = false;
#else
    m_bVerbose = false;
#endif
}


//
//   t M i n i m i z e r
//

tMinimizer::tMinimizer( const tMinimizer& Minimizer )

{
    m_gTolerance = Minimizer.m_gTolerance;
    m_bVerbose = Minimizer.m_bVerbose;
    m_gResult = Minimizer.m_gResult;
}


//
//   ~ t M i n i m i z e r
//

tMinimizer::~tMinimizer()

{
}


//
//   s e t T o l e r a n c e
//

void tMinimizer::setTolerance( double gTolerance )

{
    MTG_ASSERT( gTolerance > 0 );
    m_gTolerance = gTolerance;
}


//
//   s e t V e r b o s e
//

void tMinimizer::setVerbose( bool bVerbose )

{
    m_bVerbose = bVerbose;
}


//
//   m i n i m i z e
//

void tMinimizer::minimize( const tHeap<double>& LoBound,
    const tHeap<double>& HiBound, tHeap<double>& Lambda,
    tHeap<double>& Gradient )

{
    MTG_ASSERT( Lambda.numOfElems() > 0 );
    MTG_ASSERT( LoBound.numOfElems() >= Lambda.numOfElems() );
    MTG_ASSERT( HiBound.numOfElems() >= Lambda.numOfElems() );

    Gradient.numOfElems( Lambda.numOfElems() );

    tHeap<double> Lo = LoBound;
    tHeap<double> Hi = HiBound;

        // make sure lambdas are in the box

    for( int i = 0; i < Lambda.numOfElems(); ++i ) {
        if( Lambda[i] < Lo[i] )
            Lambda[i] = Lo[i];
        else
        if( Lambda[i] > Hi[i] )
            Lambda[i] = Hi[i];
    }

    char task[61], csave[61];
    logical lsave[4];
    integer isave[44];
    doublereal dsave[29];

    integer n = Lambda.numOfElems();
    integer m = 5;
    integer nmax = 1024;
    integer mmax = 17;

    integer iprint = m_bVerbose ? 99 : -1;
    integer isbmin = 1;  // Subspace min. method
                         // BFGS uses factr * machineEpsilon
    double factr = m_gTolerance / DBL_EPSILON; 

    integer* nbd = new integer[n];
    integer* iwa = new integer[3 * nmax];
    double* wa = new double[2 * mmax * nmax + 
                        4 * nmax + 12 * mmax * mmax + 12 * mmax];

    for( MTG_FOR_INIT( int ) i = 0; i < n; ++i )
	    nbd[i] = 2;

    strcpy( task, "START" );
    memset( &task[5], ' ', sizeof(task) - 6 );
    task[sizeof(task) - 1] = 0;
    
    bool bGo = true;

    while( bGo ) {
	    lbfgsb_( &n, &m, &Lambda[0], &Lo[0], &Hi[0], nbd,
            &m_gResult, &Gradient[0], &factr, wa, iwa, task,
            &iprint, &isbmin, csave, lsave, isave, dsave,
            sizeof(task) - 1, sizeof(csave) - 1 );
	
	    if( strncmp( task, "FG", 2 ) == 0 ) {
                // Call the callback function:
	        eval( Lambda, m_gResult, Gradient );
            // MTG_TRACE( "Optimization: %lg\n", m_gResult );
	    }
	    else 
        if( strncmp( task, "NEW_X", 5 ) != 0 ) {
	        bGo = false;
        }
    }

    delete nbd;
    delete wa;
    delete iwa;
}


//
//   m i n i m i z e
//

void tMinimizer::minimize( double gLoBound, double gHiBound,
    tHeap<double>& Lambda, tHeap<double>& Gradient )

{
    MTG_ASSERT( Lambda.numOfElems() > 0 );

    tHeap<double> Lo;
    tHeap<double> Hi;

    Lo.numOfElems( Lambda.numOfElems() );
    Lo.fill( gLoBound );

    Hi.numOfElems( Lambda.numOfElems() );
    Hi.fill( gHiBound );

    minimize( Lo, Hi, Lambda, Gradient );
}


//
//   m i n i m i z e
//

void tMinimizer::minimize( const tHeap<double>& LoBound,
    const tHeap<double>& HiBound, tHeap<double>& Lambda )

{
    tHeap<double> Gradient;
    minimize( LoBound, HiBound, Lambda, Gradient );
}


//
//   m i n i m i z e
//

void tMinimizer::minimize( double gLoBound, double gHiBound,
    tHeap<double>& Lambda )

{
    tHeap<double> Gradient;
    minimize( gLoBound, gHiBound, Lambda );
}

MTG_END_NAMESPACE
