// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#if ! defined(_MTG_NUMERICAL_EXPR_)
#define _MTG_NUMERICAL_EXPR_

#include "MtgExpression.h"
#include "MtgLookup.h"

MTG_BEGIN_NAMESPACE

// Expressions compute the new value from the local environment at
// the current node. They can also act as guards in if-then-else-endif
// statements. There is no boolean type; the C convention is used instead.
//
// Expressions can contain the following keywords which access
// local parameters of the current node of the lattice:
//
//    time                  the current time in days since day 0
//    day                   corresponds to floor( time )
//    fraction_of_day       corresponds to time - floor( time )
//                          this holds: time == day + fractionofday
//    "factor"              the value of one of the underlying factors
//                          by its name (stock price, interest rate)
//    value                 the current value of the instrument
//    <name>.value          the current value of the instrument whose
//                          name is <name>
//
// The arithmetic operations +,-,*,/ are supported. Furthermore
// supported are the unary functions floor, ceil, and the binary
// functions min, max. The boolean operations ==, !=, >, >=,
// <, <=, &&, ||, ! are supported as they are in C. Precedences
// follow C, too. There is also the ternary .. ? .. : .. operator,
// or the equivalent if..then..else..endif construct.

class tDateExpr;
class tEngine;
class tFactor;
class tPortfolio;
class tParser;
class tSeqEvGenerator;
    

//
//   t N u m e r i c a l E x p r
//
    
class tNumericalExpr : public tExpression {

public:

        // These are the valid expression types. They correspond to
        // the keywords, functions and operators as described above
        // in the introduction. xNumber stands for any real number.

    enum tType {
        xNumber,
        xUnitValue,
        xTime, xDay, xFractionOfDay, xFactor,
        xNeg, xPlus, xMinus, xTimes, xOver,
        xMin, xMax, xFloor, xCeil,
        xEq, xNe, xGt, xLt, xGe, xLe, xNot, xAnd, xOr,
        xIfThen, xIfThenElse,
        xAccrualFactor, xDiscountFactor,
        xFutureValue, xPresentValue
    };

    static const int ALLOW_UNITVALUE;
    static const int ALLOW_FACTOR;

private:

    tType m_nType;

        // For xUnitValue and xMultiplier expressions, references to other
        // instruments are possible. Ultimately, these instruments are
        // referenced by their numerical id in m_nId. Before these ids
        // are known, however, the string that is used in the script
        // is stored in m_sId, waiting to be examined by the function
        // resolve() (see below). Note that the string m_sId is owned by
        // the expression node.

        // For xUnitValue and xMultiplier nodes that refer to the current
        // instrument, m_nId == -1 and m_sId == 0.

    char *m_sId;
    int m_nId;

        // xNumber nodes store their value here.

    double m_gValue;

        // xFactor expressions store the reference here, and, after
        // resolution, the factor id in m_nId;

    tFactor* m_pFactor;

        // Function and operation nodes store the pointers to their
        // subexpressions here. Unused pointers must be set to 0.

    tExpression *m_Child[3];

        // Some helper functions.

    void init();
    void cleanup();

    void copyFrom( tNumericalExpr& Expr );

        // This constructor creates a xUnitValue, xMultiplier, xTime,
        // xDay or xFractionOfDay node.

    void setType( tType nType );

        // This constructor creates a xUnitValue node
        // that refers to instrument sId. Note that sId must point
        // to a dynamically allocated string that is subsequently
        // owned by the expression. That means, sId must not be deleted
        // outside the expression. Deallocation is handled by the
        // expression destructor.

    void setType( tType nType, char *sId );

        // This constructor creates a xUnitValue  that refers to
        // instrument nId. (Here, the numerical id is known.)

    void setType( tType nType, int nId );

        // This constructor creates a xNumber node.

    void setType( tType nType, double gValue );

        // This constructor creates a factor node; the reference
        // counter of factor (it's a tObject!) is incremented *here*.

    void setType( tType nType, tFactor* pFactor );

        // This constructor creates a xNeg, xFloor, xCeil or xNot node.

    void setType( tType nType, tNumericalExpr *pChild );

        // This constructor creates a xPlus, xMinus, xTimes, xOver,
        // xMin, xMax, xEq, xNe, xGt, xLt, xGe, xLe, xAnd or xOr node.

    void setType( tType nType, tNumericalExpr *pChild1, 
        tNumericalExpr *pChild2 );

        // This constructor creates a xIfThenElse node (ternary operator).

    void setType( tType nType, tNumericalExpr *pChild1, 
        tNumericalExpr *pChild2, tNumericalExpr *pChild3 );

        // This constructor creates a xFutureValue or xPresentValue
        // node.

    void setType( tType nType, tDateExpr *pChild1, tDateExpr *pChild2 );

        // This constructor creates a xAccrualFactor or xDiscountFactor
        // node.

    void setType( tType nType, tDateExpr *pChild1,
        tDateExpr *pChild2, tDateExpr *pChild3 );

    tRetCode parseEnv( tParser& Parser, int nFlags, bool& bGo );
    tRetCode parseId( tParser& Parser, int nFlags );
    tRetCode parseMinMax( tParser& Parser, int nFlags );
    tRetCode parseIfThenElse( tParser& Parser, int nFlags );
    tRetCode parseRateExpr( tParser& Parser, int nFlags );

        // Sorry, but this factor is the grammar-factor:

    tRetCode parseFactor( tParser& Parser, int nFlags );
    tRetCode parseTerm( tParser& Parser, int nFlags,
        int nLevel = INT_MAX );

    tRetCode parse1( tParser& Parser, int nFlags );
    tRetCode parse1( tParser& Parser );

public:

    tNumericalExpr();
    tNumericalExpr( tNumericalExpr& Expr );

    ~tNumericalExpr();

    tExpression* clone();

    tNumericalExpr& operator=( tNumericalExpr& Expr );

    tRetCode resolve( int nThisId, tLookup<int>& Claim,
        tLookup<int>& Factor, tHeap<tFactor*>& FactorPtr,
        tPortfolio& Pf, int nFlags );

    tRetCode resolve( int nThisId, tLookup<int>& Claim,
        tLookup<int>& Factor, tHeap<tFactor*>& FactorPtr,
        tPortfolio& Pf );

    void getEvents( tSeqEvGenerator& EvGen ) const;
    void getBarriers( const tFactor* pFactor, tHeap<double>& Barrier ) const;
    void getConstants( tHeap<double>& Constant ) const;

        // If the expression is just a number, sometimes a more
        // efficient approach can be made.

    bool isNumber( double& gValue ) const;

        // The next functions throw an exception if something goes
        // wrong. bValid is used to determine whether the missing
        // else branch of any if..then..endif clause has been
        // triggered.

    double apply( tEngine& Engine ) {
        bool bValid;
        return apply( Engine, 0, bValid );
    }

    double apply( tEngine& Engine, bool& bValid ) {
        return apply( Engine, 0, bValid );
    }

    double apply( tEngine& Engine, double gUnitValue ) {
        bool bValid;
        return apply( Engine, gUnitValue, bValid );
    }

    double apply( tEngine& Engine, double gUnitValue, bool& bValid );

        // Note that ALLOW_UNITVALUE only covers the unit value
        // of the _current_ instrument. Unit values of other
        // instruments (auxiliaries) may be queried without
        // restriction.

    tRetCode parse( tParser& Parser, int nFlags );
    tRetCode parse( tParser& Parser );

    friend class tExPolicyExpr;
    friend class tDateExpr;
};

MTG_END_NAMESPACE

#endif
