// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#include "MtgIncl.h"
#include "MtgOFEngine.h"
#include "MtgClaim.h"
#include "MtgScenario.h"

MTG_BEGIN_NAMESPACE


//
//   t O F S p e c i f i c
//

tOFEngine::tOFSpecific::tOFSpecific( tOFEngine& Engine )
    : tSpecific( Engine )

{
    int n = Engine.multiplier().numOfElems();

    m_NumOfUpLevels.numOfElems( n );
    m_NumOfDownLevels.numOfElems( n );
    m_PayoffIterator.numOfElems( n );
    m_CashflowIterator.numOfElems( n );
}


//
//   i n i t
//

void tOFEngine::init()

{
    m_pSolver = 0;
}


//
//   m o n i t o r
//

void tOFEngine::monitor( tSlotLayer::tIter& I, bool& bHasAlternatives )

{
    tClaim& Claim = I.claim();

    MTG_ASSERT( Claim.isMonitored() && Claim.maturity() > day() );

    int nIndex = I.index();
    int nPos = I.xlat();
    double gMultiplier = multiplier( nIndex );

    double gCurrent = curRow()[nPos];
    double gPayoff;

        // Three possibilities: don't, may and force exercise.
        // Default is don't exercise.

    tExPolicy nExPolicy = Claim.monitor( *this, gCurrent );

    if( nExPolicy != xDontExercise ) {
            // Find intrinsic value.
        gPayoff = Claim.exercisePayoff( *this );
        temp2Row()[nPos] = gPayoff;
    }

    if( nExPolicy == xMayExercise ) {
            // Give it another shot.
        if( accuracy() != xMedium ) {
            scenario().refineExPolicy( *this, m_pLayer->tag(),
                nIndex, gCurrent, gPayoff, gMultiplier, nExPolicy );
        }
        else {
            tExPolicy nControlExPolicy;

                // Here, we expect nExPolicy to become
                // either xForceExercise or xDontExercise.

            nControlExPolicy = xMayExercise;
            scenario().refineExPolicy( *this, m_pLayer->tag(),
                nIndex, gCurrent, gPayoff, gMultiplier,
                nControlExPolicy, nExPolicy );

            MTG_ASSERT( nExPolicy != xMayExercise );
            MTG_ASSERT( nControlExPolicy == nExPolicy ||
                nControlExPolicy == xMayExercise );

                // nControlExPolicy contains the "uncertain"
                // policy, while nExPolicy contains a guess.

            switch( nControlExPolicy ) {
                case xMayExercise :
                    MTG_ASSERT( ! isLinear() );
                    m_JumpBag.controlSigBag( m_nLevel ).
                        mayExercise( underControl( nIndex ), nIndex );
                    bHasAlternatives = true;
                    break;

                case xForceExercise :
                    if( ! isLinear() ) {
                        m_JumpBag.controlSigBag( m_nLevel ).
                            forceExercise( underControl( nIndex ), nIndex );
                    }
                    bHasAlternatives = true;
                    break;

                default :
                    break;
            }
        }
    }

    switch( nExPolicy ) {
        case xMayExercise :
            MTG_ASSERT( ! isLinear() );
             m_JumpBag.approxSigBag( m_nLevel ).
                mayExercise( underControl( nIndex ), nIndex );
            bHasAlternatives = true;
            break;

        case xForceExercise :
            if( isLinear() ) {
                    // Correct entries. Treat early exercise
                    // premium as cashflow.
                m_pLayer->replCurValueAndTotal( m_nLevel,
                    nIndex, gPayoff, gMultiplier );
            }
            else {
                m_JumpBag.approxSigBag( m_nLevel ).
                    forceExercise( underControl( nIndex ), nIndex );
            }
            bHasAlternatives = true;
            break;

        default :
            break;
    }

    if( nExPolicy != xDontExercise && hasProfile() ) {
        profile().putLevel( m_nLevel );
        profile().putExPolicy( nIndex, nExPolicy, gPayoff );
    }
}


//
//   c o n s i d e r
//

void tOFEngine::consider( const tSigBag& SigBag, double& gTotal )

{
    gTotal = 0;

        // Add the payoff of all expiring instruments. Use the index
        // number for access.

    double* Payoff = temp2Row();

    for( int j = SigBag.altNumOfExIndexes() - 1; j >= 0; --j ) {
        int nIndex = SigBag.altExIndex( j );
        gTotal += multiplier( nIndex ) * Payoff[xlat( nIndex )];
    }
}


//
//   c o n s i d e r
//

void tOFEngine::consider( const tSigBag& SigBag, double& gTotal,
    tSlotLayer*& pSource )

{
    getNonAuxLayer( SigBag.altSignature(), pSource );

    gTotal = pSource->curTotal( m_nLevel );

    double* Payoff = temp2Row();

        // Now add the payoff of all expiring instruments.
    for( int j = SigBag.altNumOfExIndexes() - 1; j >= 0; --j ) {
        int nIndex = SigBag.altExIndex( j );
        gTotal += multiplier( nIndex ) * Payoff[xlat( nIndex )];
    }
}


//
//   c o n s i d e r
//

void tOFEngine::consider( const tSigBag& SigBag, double& gTotal,
    tSlotLayer*& pSource, bool& bFromPrep )

{
    tMode nMode;

    getNonAuxLayer( SigBag.altSignature(), pSource, nMode );

    if( nMode == xSingleton ) {
        bFromPrep = false;
        gTotal = pSource->curTotal( m_nLevel );
    }
    else {
        gTotal = pSource->prepTotal( m_nLevel );
    }

    double* Payoff = temp2Row();

        // Now add the payoff of all expiring instruments.
    for( int j = SigBag.altNumOfExIndexes() - 1; j >= 0; --j ) {
        int nIndex = SigBag.altExIndex( j );
        gTotal += multiplier( nIndex ) * Payoff[xlat( nIndex )];
    }
}


//
//   r e t r i e v e
//

void tOFEngine::retrieve( const tSigBag& SigBag, double gTotal,
    tSlotLayer* pSource, bool bFromPrep )

{
    MTG_ASSERT( pSource != m_pLayer );

    double* To;
    
    To = curRow();
    curTotal() = gTotal;

    if( pSource != 0 ) {
        double* From = bFromPrep ?
            pSource->prepRow( m_nLevel ) : pSource->curRow( m_nLevel );

            // First copy the instruments which are alive in
            // both layers.
        for( tSlotLayer::tIter I( *pSource ); I; ++I )
            To[m_pLayer->xlatIndex( I.index() )] = From[I.xlat()];
    }

    double* Payoff = temp2Row();

            // Now insert the payoff of all expiring instruments.
    for( int j = SigBag.altNumOfExIndexes() - 1; j >= 0; --j ) {
        int nPos = xlat( SigBag.altExIndex( j ) );
        To[nPos] = Payoff[nPos];
    }
}


//
//   f i n d A l t e r n a t i v e
//

void tOFEngine::findAlternative( tSigBag& SigBag )

{
    MTG_ASSERT( ! isLinear() && SigBag.hasAlternatives() );

    tScenario& S = scenario();

    int n1 = SigBag.numOfAlternatives( 0 );
    int n2 = SigBag.numOfAlternatives( 1 );

    int nSelJ = -1;
    double gTotalJ = 0;
    tSlotLayer* pSelJ = 0;
    bool bFromPrepJ = true;

        // Iterate over instruments not under our control, keeping
        // all other instruments alive.

    for( int j = -1; j < n2; ++j ) {
        tSlotLayer *pSel1;
        double gTotal1;
        bool bFromPrep1 = true;

        if( SigBag.considerAlternative( -1, j ) ) {
            pSel1 = m_pLayer;
            gTotal1 = curTotal();
        }
        else {
            if( SigBag.altSignature().isEmpty() ) {
                pSel1 = 0;
                consider( SigBag, gTotal1 );
            }
            else {
                consider( SigBag, gTotal1, pSel1, bFromPrep1 );
            }
        }
        if( j < 0 ||
            S.endureOver( m_pLayer->tag(), gTotal1, gTotalJ ) ) {
            nSelJ = j;
            pSelJ = pSel1;
            gTotalJ = gTotal1;
            bFromPrepJ = bFromPrep1;
        }
    }

    int nSelK = -1;
    double gTotalK = gTotalJ;
    tSlotLayer* pSelK = pSelJ;

        // Iterate over instruments that we can control.

    for( int k = 0; k < n1; ++k ) {
        tSlotLayer *pSel1;
        double gTotal1;

        SigBag.considerAlternative( k, -1 );
        if( SigBag.altSignature().isEmpty() ) {
            pSel1 = 0;
            consider( SigBag, gTotal1 );
        }
        else {
            consider( SigBag, gTotal1, pSel1 );
        }
        if( S.chooseOver( m_pLayer->tag(), gTotal1, gTotalK ) ) {
            nSelK = k;
            pSelK = pSel1;
            gTotalK = gTotal1;
        }
    }

    if( pSelK != m_pLayer ) {
        if( hasProfile() )
            profile().putAlternative( SigBag.numOfAlternatives(), pSelK );

        if( m_pLayer->withChain() ) {
            if( pSelK == 0 )
                m_pLayer->chain( m_nLevel ) = 0;
            else
                m_pLayer->chain( m_nLevel ) = pSelK->chain( m_nLevel );
        }

        if( pSelK == pSelJ ) {
                // Copy prep/current row into current row.                    
            SigBag.selectAlternative( -1, nSelJ );
            retrieve( SigBag, gTotalJ, pSelJ, bFromPrepJ );
        }
        else {  // Copy current row into current row.                
            SigBag.selectAlternative( nSelK, -1 );
            retrieve( SigBag, gTotalK, pSelK, false );
        }
    }
    else {      // No alternative selected.        
        SigBag.selectAlternative( -1, -1 );
        if( hasProfile() )
            profile().putAlternative( SigBag.numOfAlternatives() );
        if( m_pLayer->withChain() )
            m_pLayer->chain( m_nLevel ) = m_pLayer;
    }
}


//
//   r e f r e s h A l t e r n a t i v e
//

void tOFEngine::refreshAlternative( tSigBag& SigBag, bool bUnderControl )

{
    MTG_ASSERT( ! isLinear() && SigBag.altNumOfExIndexes() > 0 );

        // Alternative is still selected!

    double gTotal;
    bool bRetrieve;

        // Use the following heuristic: if the alternative that has been
        // selected previously is still dominant under the scenario,
        // redo the alternative; otherwise, undo the alternative.

    tScenario& S = scenario();
    SigBag.considerAlternative();

    if( SigBag.altSignature().isEmpty() ) {
        MTG_ASSERT( ! m_pLayer->withChain() ||
                    m_pLayer->chain( m_nLevel ) == 0 );

        consider( SigBag, gTotal );

            // gTotal contains the total payoff; curTotal() contains the
            // corrected SOR value.

        if( bUnderControl )
            bRetrieve = S.chooseOver( m_pLayer->tag(), gTotal, curTotal() );
        else
            bRetrieve = S.endureOver( m_pLayer->tag(), gTotal, curTotal() );

        if( bRetrieve ) {
                // redo alternative
            retrieve( SigBag, gTotal, 0, false );
        }
    }
    else {
        tSlotLayer* pSel;

        if( SigBag.alternative0() == -1 ) {
            bool bFromPrep = true;

            consider( SigBag, gTotal, pSel, bFromPrep );

            if( bUnderControl )
                bRetrieve = S.chooseOver( m_pLayer->tag(), gTotal, curTotal() );
            else
                bRetrieve = S.endureOver( m_pLayer->tag(), gTotal, curTotal() );

            if( bRetrieve )
                retrieve( SigBag, gTotal, pSel, bFromPrep );
        }
        else {
            MTG_ASSERT( bUnderControl );
            MTG_ASSERT( SigBag.alternative1() == -1 );

            consider( SigBag, gTotal, pSel );
            bRetrieve = S.chooseOver( m_pLayer->tag(), gTotal, curTotal() );

            if( bRetrieve )
                retrieve( SigBag, gTotal, pSel, false );
        }

        MTG_ASSERT( ! m_pLayer->withChain() ||
                    m_pLayer->chain( m_nLevel ) == pSel->chain( m_nLevel ) );
    }

    if( ! bRetrieve ) {
            // undo alternative
        //MTG_TRACE( "Engine: undoing alternative\n" );
    }
}


//
//   r e f i n e
//

void tOFEngine::refine( tSigBag& SigBag, const tHeap<int>& Pos,
    const tSignature*& pAltSig, bool bFirst )

{
    MTG_ASSERT( accuracy() == xMedium && ! isLinear() );
    MTG_ASSERT( SigBag.hasAlternatives() );

    m_nLevel = pos()[0] = Pos[0];

    if( hasProfile() )
        profile().putLevel( m_nLevel );

    findAlternative( SigBag );

    const tSlotLayer* p = m_pLayer->chain( m_nLevel );

    if( p == 0 )
        pAltSig = 0;
    else
        pAltSig = p->signature();
}


//
//   d o B a r r i e r s
//

void tOFEngine::doBarriers( int& nAdjDown, int& nAdjUp )

{
    tLattice& L = lattice();

    int nDown = L.numOfDownLevels( 0, slice() );
    int nUp = L.numOfUpLevels( 0, slice() );

    nAdjDown = nDown;
    nAdjUp = nUp;

    for( tSlotLayer::tIter I( *m_pLayer ); I; ++I ) {
        tClaim& Claim = I.claim();
        int nIndex = I.index();

        MTG_ASSERT( Claim.maturity() >= day() );

            // All claims are included, including those that
            // mature today (although for those, we don't
            // store the payoff at this point).

        double gMultiplier = multiplier( nIndex );

        double gUpBarrier;
        bool bHasUpBarrier = Claim.upBarrier( *this, gUpBarrier );

        double gDownBarrier;
        bool bHasDownBarrier = Claim.downBarrier( *this, gDownBarrier );

        m_NumOfUpLevels[nIndex] = nUp;
        m_NumOfDownLevels[nIndex] = nDown;

            // These two are helpful in keeping track where
            // payoff and cashflow have been determined, since
            // there might be an interruption.

        m_pSpecific->m_PayoffIterator[nIndex] = -nDown;
        m_pSpecific->m_CashflowIterator[nIndex] = -nDown;

        if( ! bHasUpBarrier && ! bHasDownBarrier )
            continue;

        bool bNewUp = false;

        for( pos()[0] = -nDown; ( m_nLevel = pos()[0] ) <= nUp; ++pos()[0] ) {
            bool bOut = false;

            if( bHasUpBarrier && dominant() >= gUpBarrier ) {
                    // Note that this can be negative:
                if( ! bNewUp ) {
                    m_NumOfUpLevels[nIndex] = m_nLevel - 1;
                    bNewUp = true;
                }
                bOut = true;
            }
            else
            if( bHasDownBarrier && dominant() <= gDownBarrier ) {
                m_NumOfDownLevels[nIndex] = -m_nLevel - 1;
                bOut = true;
            }

            if( bOut && Claim.maturity() > day() ) {
                double gPayoff = Claim.knockoutPayoff( *this );

                if( hasProfile() ) {
                    profile().putLevel( m_nLevel );
                    profile().putExPolicy( nIndex, xForceExercise,
                        gPayoff );
                }
    
                m_pLayer->setCurValueAndTotal( m_nLevel, nIndex,
                    gPayoff, gMultiplier );

                if( ! isLinear() ) {
                    m_JumpBag.approxSigBag( m_nLevel ).
                        forceExercise( underControl( nIndex ), nIndex );

                        // the control sigbag is not necessary in this case
                }
            }
        }

        if( Claim.maturity() > day() ) {
            if( m_NumOfUpLevels[nIndex] < nAdjUp )
                nAdjUp = m_NumOfUpLevels[nIndex];
            if( m_NumOfDownLevels[nIndex] < nAdjDown )
                nAdjDown = m_NumOfDownLevels[nIndex];
        }
    }
}


//
//   d o B o u n d a r y
//

void tOFEngine::doBoundary( int nAdjDown, int nAdjUp )

{
    if( isLinear() )
        return;

    tLattice& L = lattice();

    int nDown = L.numOfDownLevels( 0, slice() );
    int nUp = L.numOfUpLevels( 0, slice() );

        // Under non-linearity, we have to fill in the
        // claims that don't knock out, from a different
        // layer (under linearity, we'll do a rollback
        // in another place).

    pos()[0] = -nDown;
    while( pos()[0] <= nUp ) {
        if( pos()[0] == -nAdjDown && nAdjUp + 1 > -nAdjDown ) {
            pos()[0] = nAdjUp + 1;
            continue;
        }

        m_nLevel = pos()[0];

        tSigBag& Approx = m_JumpBag.approxSigBag( m_nLevel );
        MTG_ASSERT( Approx.hasAlternatives() );

        if( Approx.considerAlternative( -1, -1 ) )
            throw INTERNAL_ERROR;

        tSlotLayer* pSource = 0;

            // We only have to copy the instruments which are not
            // exercised; the payoff for the others has already
            // been filled in!

        if( ! Approx.altSignature().isEmpty() ) {
            getNonAuxLayer( Approx.altSignature(), pSource );

            double* To = curRow();
            double* From = pSource->curRow( m_nLevel );

            for( tSlotLayer::tIter I( *pSource ); I; ++I )
                To[m_pLayer->xlatIndex( I.index() )] = From[I.xlat()];
            curTotal() += pSource->curTotal( m_nLevel );
        }

        if( hasProfile() ) {
            profile().putLevel( m_nLevel );
            profile().putAlternative( 1, pSource );
        }

        ++pos()[0];
    }    
}


//
//   d o R o l l b a c k
//

void tOFEngine::doRollback( int nAdjDown, int nAdjUp )

{
    if( isFinal() )
        return;

    tLattice& L = lattice();

    int nOrgDown = L.numOfDownLevels( 0, slice() );
    int nOrgUp = L.numOfUpLevels( 0, slice() );

    if( isLinear() ) {
            // Under linearity, we have to go over the
            // entire width of the lattice.
        nAdjDown = nOrgDown;
        nAdjUp = nOrgUp;
    }

    m_pSolver->prepare( *this, *m_pLayer, nAdjUp, nAdjDown,
        L.numOfUpLevels( 0, slice() + 1 ),
        L.numOfDownLevels( 0, slice() + 1 ) );

    for( tSlotLayer::tIter I( *m_pLayer ); I; ++I ) {
        tClaim& Claim = I.claim();
        int nIndex = I.index();

        if( Claim.maturity() > day() ) {
            int nDown = nAdjDown;
            int nUp = nAdjUp;

            if( nDown > m_NumOfDownLevels[nIndex] )
                nDown = m_NumOfDownLevels[nIndex];
            if( nUp > m_NumOfUpLevels[nIndex] )
                nUp = m_NumOfUpLevels[nIndex];

            tOFSolver::tBoundary nDownBoundary = 
                nDown < nOrgDown ?
                    tOFSolver::xGivenValue : tOFSolver::xZeroGamma;

            tOFSolver::tBoundary nUpBoundary = 
                nUp < nOrgUp ?
                    tOFSolver::xGivenValue : tOFSolver::xZeroGamma;

            m_pSolver->prepare( nIndex, nUp, nUpBoundary,
                nDown, nDownBoundary );
        }
    }

    m_pSolver->solve();
}


//
//   d o M o n i t o r
//

void tOFEngine::doMonitor( int nAdjDown, int nAdjUp, bool& bHasAlternatives )

{
    bHasAlternatives = false;

    for( tSlotLayer::tIter I( *m_pLayer ); I; ++I ) {
        tClaim& Claim = I.claim();

            // It makes sense to loop over the claims, then
            // over the lattice. Monitored claims are much
            // more rare than non-monitored claims.

        if( Claim.isMonitored() && Claim.maturity() > day() ) {

            int nIndex = I.index();
            int nDown, nUp;

                // Monitor only where the claim does not knock out.

            if( isLinear() ) {
                nDown = m_NumOfDownLevels[nIndex];
                nUp = m_NumOfUpLevels[nIndex];
            }
            else {
                nDown = nAdjDown;
                nUp = nAdjUp;
            }

            for( pos()[0] = -nDown;
                    ( m_nLevel = pos()[0] ) <= nUp; ++pos()[0] ) {
                monitor( I, bHasAlternatives );
            }
        }
    }

        // The prep row later will contain the min respectively the
        // max over all combinations that are not under the investors
        // control, for non-singletons.
}


//
//   d o A l t e r n a t i v e s
//

void tOFEngine::doAlternatives( int nAdjDown, int nAdjUp,
    bool bHasAlternatives )

{
    if( ! bHasAlternatives && ! hasProfile() )
        return;

    if( isLinear() ) {
            // Nothing to do except updating the profile.
        if( hasProfile() ) {
            int nDown = lattice().numOfDownLevels( 0, slice() );
            int nUp = lattice().numOfUpLevels( 0, slice() );

            for( int k = -nDown; k <= nUp; ++k ) {
                profile().putLevel( k );
                profile().putAlternative( 1 );
            }
        }
        return;
    }

    if( -nAdjDown > nAdjUp )
        return;

    if( accuracy() == xMedium ) {
        if( hasProfile() ) {
            for( int k = -nAdjDown; k <= nAdjUp; ++k ) {
                if( ! m_JumpBag.controlSigBag( k ).hasAlternatives() ) {
                    profile().putLevel( k );
                    profile().putAlternative( 1 );
                }
            }
        }

        tRefine T( underControl(), *this );
        m_JumpBag.shrink( -nAdjDown, nAdjUp );
        m_JumpBag.refine( T, false );
    }
    else {
        for( pos()[0] = -nAdjDown;
                ( m_nLevel = pos()[0] ) <= nAdjUp; ++pos()[0] ) {
            tSigBag& SigBag = m_JumpBag.approxSigBag( m_nLevel );

            if( hasProfile() )
                profile().putLevel( m_nLevel );

            if( SigBag.hasAlternatives() ) {
                findAlternative( SigBag );
            }
            else    
            if( hasProfile() ) {
                profile().putAlternative( 1 );
            }
        }
    }
}


//
//   b e g i n I n c r e m e n t
//

void tOFEngine::beginIncrement( int nAdjDown, int nAdjUp,
    bool bUC, bool bNotUC )

{
    MTG_ASSERT( lattice().isImplicit() );

        // Note that nAdjDown and nAdjUp may be different
        // from the corresponding parameters in the preceding
        // functions, as the implicit solver may decide to
        // extrapolate the outer nodes, rather than to 
        // compute them directly.
}


//
//   d o I n c r e m e n t
//
    
void tOFEngine::doIncrement( const tHeap<int>& Pos, bool bUC, bool bNotUC )

{
    MTG_ASSERT( lattice().isImplicit() );

    bool bHasAlternatives = false;
    m_nLevel = pos()[0] = Pos[0];

    if( hasProfile() )
        profile().putLevel( m_nLevel );

    if( isLinear() ) {
        for( tSlotLayer::tIter I( *m_pLayer ); I; ++I ) {
            tClaim& Claim = I.claim();
            int nIndex = I.index();

            if( Claim.isMonitored() && Claim.maturity() > day() && 
                    m_nLevel >= -m_NumOfDownLevels[nIndex] &&
                    m_nLevel <= m_NumOfUpLevels[nIndex] ) {
                monitor( I, bHasAlternatives );
            }
        }
    }
    else {
        MTG_ASSERT( bUC != bNotUC );

        tSigBag& SigBag = m_JumpBag.approxSigBag( m_nLevel );

        if( SigBag.hasAlternatives() && SigBag.altNumOfExIndexes() > 0 ) {
            refreshAlternative( SigBag, bUC );
        }
    }
}


//
//   e n d I n c r e m e n t
//

void tOFEngine::endIncrement( int nAdjDown, int nAdjUp )

{
    MTG_ASSERT( lattice().isImplicit() );
}


//
//   d o P a y o f f
//

void tOFEngine::doPayoff()

{
    int nUp = lattice().numOfUpLevels( 0, slice() );

    for( tSlotLayer::tIter I( *m_pLayer ); I; ++I ) {
        tClaim& Claim = I.claim();

        if( Claim.maturity() == day() ) {
            int nIndex = I.index();
            double gMultiplier = multiplier( nIndex );
            double gPayoff;

                // Don't compute payoff twice!

            for( m_nLevel = m_pSpecific->m_PayoffIterator[nIndex];
                    m_nLevel <= nUp; ++m_nLevel ) {
                pos()[0] = m_nLevel;

                    // Distinguish between knockout and regular payoff:                            

                if( m_nLevel < -m_NumOfDownLevels[nIndex] ||
                    m_nLevel > m_NumOfUpLevels[nIndex] ) {
                    gPayoff = Claim.knockoutPayoff( *this );
                    if( hasProfile() ) {
                        profile().putLevel( m_nLevel );
                        profile().putExPolicy( nIndex, xForceExercise,
                            gPayoff );
                    }
                }
                else {
                    gPayoff = Claim.payoff( *this );
                }

                m_pLayer->setCurValueAndTotal( m_nLevel, nIndex,
                    gPayoff, gMultiplier );

                m_pSpecific->m_PayoffIterator[nIndex] = m_nLevel + 1;
            }
        }
    }
}


//
//   d o C a s h f l o w
//

void tOFEngine::doCashflow()

{
    if( ! atDawn() )
        return;

    for( tSlotLayer::tIter I( *m_pLayer ); I; ++I ) {
        tClaim& Claim = I.claim();

        if( Claim.hasCashflow( *this ) ) {
            int nIndex = I.index();
            int nPos = I.xlat();
            double gMultiplier = multiplier( nIndex );

            int nDown, nUp;
            double gCashflow;

                // Cash flows only where the claim hasn't knocked out.

            nDown = m_NumOfDownLevels[nIndex];
            nUp = m_NumOfUpLevels[nIndex];

                // Don't compute and add cashflow payoff twice:

            if( nDown > -m_pSpecific->m_CashflowIterator[nIndex] )
                nDown = -m_pSpecific->m_CashflowIterator[nIndex];

            for( m_nLevel = -nDown; m_nLevel <= nUp; ++m_nLevel ) {
                pos()[0] = m_nLevel;
                gCashflow = Claim.cashflow( *this );
                curRow()[nPos] += gCashflow;
                curTotal() += gCashflow * gMultiplier;
                m_pSpecific->m_CashflowIterator[nIndex] = m_nLevel + 1;
            }
        }
    }
}


//
//   d o P r o f i l e
//

void tOFEngine::doProfile()

{
    if( hasProfile() ) {
        tLattice& L = lattice();

        int nDown = L.numOfDownLevels( 0, slice() );
        int nUp = L.numOfUpLevels( 0, slice() );

        for( int k = -nDown; k <= nUp; ++k ) {
            profile().putLevel( k );
            profile().putTotal( m_pLayer->curTotal( k ) );
        }
    }
}


//
//   c r e a t e S o l v e r
//

tRetCode tOFEngine::createSolver( tOFSolver*& pSolver )

{
    pSolver = lattice().createOFSolver( 0 );
    return OK;
}



//
//   b e f o r e T a s k 1
//

void tOFEngine::beforeTask1( tSlotLayer& Layer, tSpecific*& pSpecific )

{
    MTG_ASSERT( lattice().dimension() == 1 && Layer.dimension() == 1 );
    MTG_ASSERT( Layer.signature() != 0 );

    m_pLayer = &Layer;

    tLattice& L = lattice();

    int nDown = L.numOfDownLevels( 0, slice() );
    int nUp = L.numOfUpLevels( 0, slice() );

    if( ! isLinear() ) {
        m_JumpBag.setCoord( 0, -nDown, nUp, accuracy() == xMedium );
        m_JumpBag.setRefSignature( m_pLayer->signature() );
    }

    m_pLayer->selfChain();

    if( isFinal() )
        m_pLayer->zero();
    else
        m_pLayer->curClearTotal();

    if( pSpecific == 0 ) {
        m_pSpecific = new tOFSpecific( *this );
        pSpecific = m_pSpecific;
    }
    else {
        m_pSpecific = static_cast<tOFSpecific*>( pSpecific );
    }
}


//
//   a f t e r T a s k 1
//

void tOFEngine::afterTask1()

{
    m_pSpecific->m_NumOfUpLevels = m_NumOfUpLevels;
    m_pSpecific->m_NumOfDownLevels = m_NumOfDownLevels;
}


//
//   b e f o r e T a s k 2
//

void tOFEngine::beforeTask2( tSlotLayer& Layer, tSpecific*& pSpecific )

{
    MTG_ASSERT( pSpecific != 0 );

    m_pLayer = &Layer;
    m_pSpecific = static_cast<tOFSpecific*>( pSpecific );

    m_NumOfUpLevels = m_pSpecific->m_NumOfUpLevels;
    m_NumOfDownLevels = m_pSpecific->m_NumOfDownLevels;
}


//
//   a f t e r T a s k 2
//

void tOFEngine::afterTask2()

{
}


//
//   m o d i f y N o n A u x L a y e r
//

void tOFEngine::modifyNonAuxLayer( tSlotLayer& Layer ) const

{
    Layer.setWithPrep( ! Layer.signature()->isFull() );

    if( accuracy() == xMedium || lattice().isImplicit() )
        Layer.setWithChain( true );

    if( Layer.hasMonitoredClaims() )
        Layer.setWithTemp2( true );
}


//
//   d o A u x T a s k 1
//

void tOFEngine::doAuxTask1( tSlotLayer& Layer, tSpecific*& pSpecific )

{
    throw tException( NOT_IMPLEMENTED );
}


//
//   d o A u x T a s k 2
//

void tOFEngine::doAuxTask2( tSlotLayer& Layer, tSpecific*& pSpecific )

{
    throw tException( NOT_IMPLEMENTED );
}


//
//   d o N o n A u x T a s k 1 
//

void tOFEngine::doNonAuxTask1( tSlotLayer& Layer, tSpecific*& pSpecific )

{
    beforeTask1( Layer, pSpecific );

    int nDown = lattice().numOfDownLevels( 0, slice() );
    int nUp = lattice().numOfUpLevels( 0, slice() );

    if( preprocess() ) {
        for( pos()[0] = -nDown - 1; ( m_nLevel = pos()[0] ) <= nUp + 1;
                ++pos()[0] ) {
            nonAuxPreprocess();
        }
    }

    tRetCode nRet;
    int nAdjDown, nAdjUp;
    bool bHasAlternatives;

    doBarriers( nAdjDown, nAdjUp );
    doBoundary( nAdjDown, nAdjUp );
    doRollback( nAdjDown, nAdjUp );

    if( isSingleton() ||
            ( ! m_pLayer->withPrep() && ! lattice().isImplicit() ) ) {
        if( m_pLayer->withPrep() )  
            m_pLayer->copyCurToPrep();

        doMonitor( nAdjDown, nAdjUp, bHasAlternatives );
        doAlternatives( nAdjDown, nAdjUp, bHasAlternatives );

        if( bHasAlternatives ) {
            tIncrement T( *this, true, true );
            if( ( nRet = m_pSolver->refine( T ) ) != OK )
                throw tException( nRet );
        }
    }
    else {
        doMonitor( nAdjDown, nAdjUp, bHasAlternatives );

            // First, look at the instruments not under our control.

        m_JumpBag.hideMayExAlternatives( true );
        doAlternatives( nAdjDown, nAdjUp, bHasAlternatives );

        if( bHasAlternatives ) {
            tIncrement T( *this, false, true );
            if( ( nRet = m_pSolver->refine( T ) ) != OK )
                throw tException( nRet );
        }

            // That's going into the prep layer.

        m_pLayer->copyCurToPrep();

            // Then, look at the instruments under our control.

        m_JumpBag.showMayExAlternatives( true );
        m_JumpBag.hideMayExAlternatives( false );
        doAlternatives( nAdjDown, nAdjUp, bHasAlternatives );

        if( bHasAlternatives ) {
            tIncrement T( *this, true, false );
            if( ( nRet = m_pSolver->refine( T ) ) != OK )
                throw tException( nRet );
        }
    }

    afterTask1();
}


//
//   d o N o n A u x T a s k 2
//

void tOFEngine::doNonAuxTask2( tSlotLayer& Layer, tSpecific*& pSpecific )

{
    beforeTask2( Layer, pSpecific );

    int nDown = lattice().numOfDownLevels( 0, slice() );
    int nUp = lattice().numOfUpLevels( 0, slice() );

    doPayoff();
    doCashflow();

    if( postprocess() ) {
        for( pos()[0] = -nDown; ( m_nLevel = pos()[0] ) <= nUp;
                ++pos()[0] ) {
            if( hasProfile() )
                profile().putLevel( m_nLevel );
            nonAuxPostprocess();
        }
    }

    afterTask2();
    doProfile();
}


//
//   b e f o r e R u n
//

tRetCode tOFEngine::beforeRun()

{
    tRetCode nRet;

    if( ( nRet = super::beforeRun() ) != OK )
        return nRet;

    if( m_pSolver != 0 ) {
        delete m_pSolver;
        m_pSolver = 0;
    }
    
    if( ( nRet = createSolver( m_pSolver ) ) != OK )
        return nRet;

    m_NumOfUpLevels.numOfElems( multiplier().numOfElems() );
    m_NumOfDownLevels.numOfElems( multiplier().numOfElems() );

    return OK;
}


//
//   a f t e r R u n
//

tRetCode tOFEngine::afterRun()

{
    if( m_pSolver != 0 ) {
        delete m_pSolver;
        m_pSolver = 0;
    }

    return super::afterRun();
}


//
//   t O F E n g i n e
//

tOFEngine::tOFEngine()

{
    init();
}


//
//   ~ t O F E n g i n e
//

tOFEngine::~tOFEngine()

{
    if( m_pSolver != 0 )
        delete m_pSolver;
}

MTG_END_NAMESPACE
