// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#if ! defined(_MTG_OFENGINE_)
#define _MTG_OFENGINE_

#include "MtgFDEngine.h"
#include "MtgJumpBag.h"
#include "MtgOFSolver.h"

MTG_BEGIN_NAMESPACE


//
//   t O F E n g i n e
//

class tOFEngine : public tFDEngine {

    typedef tFDEngine super;

public:

    struct tOFSpecific : public super::tSpecific {
        tHeap<int> m_NumOfUpLevels;     // per instrument
        tHeap<int> m_NumOfDownLevels;   // per instrument
        tHeap<int> m_PayoffIterator;    // per instrument
        tHeap<int> m_CashflowIterator;  // per instrument

        tOFSpecific( tOFEngine& Engine );
    };

private:

        // OF for one-factor

    class tRefine : public tJumpBag::tRefine {
            tOFEngine& m_Engine;

        public:
            tRefine( const tHeap<bool>& UnderControl, tOFEngine& Engine ) 
                : tJumpBag::tRefine( UnderControl ), m_Engine( Engine ) {
            }

            void refine( tSigBag& Approx, const tHeap<int>& Pos,
                    const tSignature*& pAltSig, bool bFirst ) {
                m_Engine.refine( Approx, Pos, pAltSig, bFirst );
            }
    };

    class tIncrement : public tOFSolver::tIncrement {
        tOFEngine& m_Engine;
        bool m_bUC;         // under control
        bool m_bNotUC;      // not under control

        public:
            tIncrement( tOFEngine& Engine, bool bUC, bool bNotUC ) 
                : m_Engine( Engine ), m_bUC( bUC ), m_bNotUC( bNotUC ) {
            }

            void beginIncrement( int nAdjDown, int nAdjUp ) {
                m_Engine.beginIncrement( nAdjDown, nAdjUp, m_bUC, m_bNotUC );
            }

            void doIncrement( const tHeap<int>& Pos ) {
                m_Engine.doIncrement( Pos, m_bUC, m_bNotUC );
            }

            void endIncrement( int nAdjDown, int nAdjUp ) {
                m_Engine.endIncrement( nAdjDown, nAdjUp );
            }
    };

    tJumpBag m_JumpBag;

        // The following object is allocated in beforeRun()
        // by calling createSolver().

    tOFSolver* m_pSolver;

    void init();

        // monitor() is used by both doMonitor(), the main monitoring
        // function, and incrMonitor(), the incremental monitoring
        // function for implicit methods.

    void monitor( tSlotLayer::tIter& I, bool& bHasAlternatives );

    void consider( const tSigBag& SigBag, double& gTotal );
    void consider( const tSigBag& SigBag, double& gTotal,
        tSlotLayer*& pSource );
    void consider( const tSigBag& SigBag, double& gTotal,
        tSlotLayer*& pSource, bool& bFromPrep );
    void retrieve( const tSigBag& SigBag, double gTotal,
        tSlotLayer* pSource, bool bFromPrep );

    void findAlternative( tSigBag& SigBag );
    void refreshAlternative( tSigBag& SigBag, bool bUnderControl );

    void refine( tSigBag& SigBag, const tHeap<int>& Pos,
        const tSignature*& pAltSig, bool bFirst );

        // While the above are helper functions, these are the
        // central functions that distribute the work:

    void doBarriers( int& nAdjDown, int& nAdjUp );
    void doBoundary( int nAdjDown, int nAdjUp );
    void doRollback( int nAdjDown, int nAdjUp );

    void doMonitor( int nAdjDown, int nAdjUp, bool& bHasAlternatives );
    void doAlternatives( int nAdjDown, int nAdjUp, bool bHasAlternatives );

        // Incremental monitoring functions for the current level,
        // needed for implicit methods:

    void beginIncrement( int nAdjDown, int nAdjUp, bool bUC, bool bNotUC );
    void doIncrement( const tHeap<int>& Pos, bool bUC, bool bNotUC );
    void endIncrement( int nAdjDown, int nAdjUp );

        // These two have to be careful about idempotency,
        // because layer rotation no longer comes to the
        // rescue:

    void doPayoff();
    void doCashflow();

    void doProfile();

    friend struct tOFSpecific;
    friend class tOFSolver;
    friend class tRefine;
    friend class tIncrement;

protected:

    tSlotLayer* m_pLayer;
    tOFSpecific* m_pSpecific;
        
    int m_nLevel;

    tHeap<int> m_NumOfUpLevels;         // copied from specific structure
    tHeap<int> m_NumOfDownLevels;

    const tOFSolver& solver() {         // allow only const use of
        return *m_pSolver;              // solver, like computation
    }                                   // of gamma, for instance 

        // Called from tOFSolver:

    virtual const tOFSolver::tProcessParamsStub& getProcessParams(
        int& nLevel, int nNumOfUpLevels ) = 0;

    virtual tRetCode createSolver( tOFSolver*& pSolver );

    virtual void beforeTask1( tSlotLayer& Layer, tSpecific*& pSpecific );
    virtual void afterTask1();

    virtual void beforeTask2( tSlotLayer& Layer, tSpecific*& pSpecific );
    virtual void afterTask2();

    void modifyNonAuxLayer( tSlotLayer& Layer ) const;

    void doAuxTask1( tSlotLayer& Layer, tSpecific*& pSpecific );
    void doAuxTask2( tSlotLayer& Layer, tSpecific*& pSpecific );

    void doNonAuxTask1( tSlotLayer& Layer, tSpecific*& pSpecific );
    void doNonAuxTask2( tSlotLayer& Layer, tSpecific*& pSpecific );

    tRetCode beforeRun();
    tRetCode afterRun();

        // xlat() translates from the claim index to
        // the absolute slot position.

    int xlat( int nIndex ) {
        return m_pLayer->xlatIndex( nIndex );
    }

    double* curRow() {
        return m_pLayer->curRow( m_nLevel );
    }

    double* prepRow() {
        return m_pLayer->prepRow( m_nLevel );
    }

    double* lastRow() {
        return m_pLayer->lastRow( m_nLevel );
    }

    double* temp2Row() {
        return m_pLayer->temp2Row( m_nLevel );
    }

    double& curTotal() {
        return m_pLayer->curTotal( m_nLevel );
    }

    double& prepTotal() {
        return m_pLayer->prepTotal( m_nLevel );
    }

    double& lastTotal() {
        return m_pLayer->lastTotal( m_nLevel );
    }

public:

    tOFEngine();
    ~tOFEngine();
};

MTG_END_NAMESPACE

#endif
