// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#include "MtgIncl.h"
#include "MtgOFImplicit.h"
#include "MtgOFEngine.h"

#include <float.h>

MTG_BEGIN_NAMESPACE


//
//   v a r i a b l e s
//

const double tOFImplicit::m_gEps =
    DBL_EPSILON;

const double tOFImplicit::m_gSafe1 =
    4 * DBL_MIN;

    // this is m_gSafe1 / DBL_EPSILON:
const double tOFImplicit::m_gSafe2 =
    ( 4 * DBL_MIN ) / DBL_EPSILON;

const int tOFImplicit::m_nMaxRefineCount = 10;


//
//   i n i t
//

void tOFImplicit::init()

{
    m_gOmega = 1.0;
    m_gOmegaStep = 0.05; 
    m_gErrorThreshold = 1e-8;
    m_nEstIterCount = 0;
    m_nMaxIterCount = 250;
}


//
//   e x t r a p o l a t e
//

void tOFImplicit::extrapolate( int nSlot, bool bInitial )

{
    tSlotLayer& L = layer();
    tSlot& S = slot( nSlot );

        // Check whether extrapolation is necessary. Use the
        // prepared weights.

    if( S.m_nDownBoundary == xZeroGamma ) {
        double gNewValue = 
            S.m_DownZGWeights.m_gBoundary * 
                L.curValue( -S.m_nNumOfDownLevels, S.m_nIndex ) +
            S.m_DownZGWeights.m_gInner * 
                L.curValue( -S.m_nNumOfDownLevels + 1, S.m_nIndex );

        if( bInitial ) {
            L.setCurValueAndTotal( -S.m_nNumOfDownLevels - 1,
                S.m_nIndex, gNewValue, S.m_gMultiplier );
        }
        else {
            L.replCurValueAndTotal( -S.m_nNumOfDownLevels - 1,
                S.m_nIndex, gNewValue, S.m_gMultiplier );
        }
    }

    if( S.m_nUpBoundary == xZeroGamma ) {
        double gNewValue = 
            S.m_UpZGWeights.m_gBoundary * 
                L.curValue( S.m_nNumOfUpLevels, S.m_nIndex ) +
            S.m_UpZGWeights.m_gInner * 
                L.curValue( S.m_nNumOfUpLevels - 1, S.m_nIndex );

        if( bInitial ) {
            L.setCurValueAndTotal( S.m_nNumOfUpLevels + 1,
                S.m_nIndex, gNewValue, S.m_gMultiplier );
        }
        else {
            L.replCurValueAndTotal( S.m_nNumOfUpLevels + 1,
                S.m_nIndex, gNewValue, S.m_gMultiplier );
        }
    }
}


//
//   s a v e C u r V a l u e s
//

void tOFImplicit::saveCurValues()

{
    tSlotLayer& L = layer();

    for( int i = 0; i < numOfSlots(); ++i ) {
        tSlot& S = slot( i );

        int nUp = S.m_nNumOfUpLevels;
        int nDown = S.m_nNumOfDownLevels;
   
        if( S.m_nUpBoundary == xZeroGamma ) 
            ++nUp;
        if( S.m_nDownBoundary == xZeroGamma )
            ++nDown;

        for( int nLevel = -nDown; nLevel <= nUp; ++nLevel )
            temp3( nLevel, i ) = L.curValue( nLevel, S.m_nIndex );
    }
}


//
//   c a l c E r r o r
//

double tOFImplicit::calcError()

{
    tSlotLayer& L = layer();
    double gError = 0;

    for( int i = 0; i < numOfSlots(); ++i ) {
        tSlot& S = slot( i );
        double e = 0;

        int nUp = S.m_nNumOfUpLevels;
        int nDown = S.m_nNumOfDownLevels;
   
        if( S.m_nUpBoundary == xZeroGamma ) 
            ++nUp;
        if( S.m_nDownBoundary == xZeroGamma )
            ++nDown;

        for( int nLevel = -nDown; nLevel <= nUp; ++nLevel ) {
            double d = temp3( nLevel, i ) - L.curValue( nLevel, S.m_nIndex );
            e += d * d;
        }

        if( e > gError )
            gError = e;
    }

    return gError;
}

                
//
//   t O F I m p l i c i t
//

tOFImplicit::tOFImplicit()

{
    init();
}


//
//   c o p y F r o m
//

void tOFImplicit::copyFrom( const tOFImplicit& Implicit )

{
    if( this == &Implicit )
        return;

    m_gOmega = Implicit.m_gOmega;
    m_gOmegaStep = Implicit.m_gOmegaStep; 
    m_gErrorThreshold = Implicit.m_gErrorThreshold;
    m_nEstIterCount = Implicit.m_nEstIterCount;
    m_nMaxIterCount = Implicit.m_nMaxIterCount;

    m_D = Implicit.m_D;
    m_DL = Implicit.m_DL;
    m_DU = Implicit.m_DU;
    m_DU2 = Implicit.m_DU2;
    m_B = Implicit.m_B;
    m_R = Implicit.m_R;
    m_Pivot = Implicit.m_Pivot; 

    m_Last = Implicit.m_Last;
    m_Temp3 = Implicit.m_Temp3;

    super::copyFrom( Implicit );
}


//
//   s o l v e
//

void tOFImplicit::solve( int nSlot )

{
    // The real tridiagonal system
    //
    //  | B0 C0               | | V0 |   | Z0 |
    //  | A1 B1 C1            | | V1 |   | Z1 |
    //  |    A2 B2 C2         | |    |   |    |
    //  |                     | |    | = |    |
    //  |                     | |    |   |    |
    //  |              C[n-1] | |    |   |    |
    //  |           An Bn     | | Vn |   | Zn |
    //
    // is solved with the robust LU solver found in the
    // LAPACK package, with partial pivoting. The code is directly
    // ported to C++ from the routines DGTTRF and DGTTRFS.
    //
    // The A's are the m_gCurD components of tRBWeights.
    // The B's are the m_gCurM components of tRBWeights.
    // The C's are the m_gCurU components of tRBWeights.
    // Z is the column of m_Last corresponding to nIndex.
    //
    // For more information, see http://gams.nist.gov.

    tSlot& S = slot( nSlot );
    tSlotLayer& L = layer();

    int n = S.m_nNumOfUpLevels + S.m_nNumOfDownLevels + 1;

    for( int i = 0; i < n; ++i )
        m_Pivot[i] = 0;

        // LU decomposition. This part always succeeds,
        // even for singular matrices.

    m_D[0] = S.m_DownRBWeights.m_gCurM;
    if( n > 1 )
        m_DU[0] = S.m_DownRBWeights.m_gCurU;

    for( MTG_FOR_INIT( int ) i = 0; i < n - 1; ++i ) {
        int nLevel = -S.m_nNumOfDownLevels + i;
        tRBWeights& W = ( nLevel + 1 == S.m_nNumOfUpLevels ) ?
            S.m_UpRBWeights : RBWeights( nLevel + 1 );

            // DL(I) == W.m_gCurD
            // D(I+1) == W.m_gCurM
        
        if( W.m_gCurD == 0 ) {
                // no elimination required

            if( i < n - 2 ) {
                m_DU[i + 1] = W.m_gCurU;
                m_DU2[i] = 0;
            }
            m_D[i + 1] = W.m_gCurM;
        }
        else {

#if defined(_MTG_IMPLICIT_PIVOTING)

                // Do as if the largest element in the
                // row was normalized to 1.

            double pa1 = fabs( m_D[i] );
            double pa2 = fabs( m_DU[i] );
            double pa = pa1 > pa2 ? 1 : pa1 / pa2;

            double pb1 = fabs( W.m_gCurD );
            double pb2 = fabs( W.m_gCurM );
            double pb3 = fabs( W.m_gCurU );

            if( pb2 < pb3 )
                pb2 = pb3;

            double pb = pb1 > pb2 ? 1 : pb1 / pb2;
#else
            double pa = fabs( m_D[i] );
            double pb = fabs( W.m_gCurD );
#endif

            if( pa > 2 * pb ) {
                    // no row interchange, eliminate DL(I)

                double f = W.m_gCurD / m_D[i];

                m_DL[i] = f;
                m_D[i + 1] = W.m_gCurM - f * m_DU[i];

                if( i < n - 2 ) {
                    m_DU[i + 1] = W.m_gCurU;
                    m_DU2[i] = 0;
                }
            }
            else {
                    // row interchange, eliminate DL(I)
    
                double f = m_D[i] / W.m_gCurD;

                m_D[i] = W.m_gCurD;
                m_DL[i] = f;

                double g = m_DU[i];

                m_DU[i] = W.m_gCurM;
                m_D[i + 1] = g - f * W.m_gCurM;

                if( i < n - 2 ) {
                    m_DU2[i] = W.m_gCurU;
                    m_DU[i + 1] = -f * W.m_gCurU;
                }

                m_Pivot[i] = 1;
            }
        }
    }

        // Backsubstitution. This part may fail due
        // to division by zero.

    m_B[0] = last( -S.m_nNumOfDownLevels, nSlot );

    for( MTG_FOR_INIT( int ) i = 0; i < n - 1; ++i ) {
        int nLevel = -S.m_nNumOfDownLevels + i;

        if( m_Pivot[i] == 0 ) {
            m_B[i + 1] = last( nLevel + 1, nSlot ) - m_DL[i] * m_B[i];
        }
        else {
            double f = m_B[i];
            m_B[i] = last( nLevel + 1, nSlot );
            m_B[i + 1] = f - m_DL[i] * m_B[i];
        }
    }

    if( m_D[n - 1] == 0 ) {
        printf( "A: n=%d\n", n );
        throw tException( DIVISION_BY_ZERO );
    }

    m_B[n - 1] /= m_D[n - 1];

    if( n > 1 ) {
        if( m_D[n - 2] == 0 )
            throw tException( DIVISION_BY_ZERO );
        m_B[n - 2] = ( m_B[n - 2] - m_DU[n - 2] * m_B[n - 1] ) / m_D[n - 2];
    }

    for( MTG_FOR_INIT( int ) i = n - 3; i >= 0; --i ) {
        if( m_D[i] == 0 )
            throw tException( DIVISION_BY_ZERO );
        m_B[i] = ( m_B[i] - m_DU[i] * m_B[i + 1] -
                    m_DU2[i] * m_B[i + 2] ) / m_D[i];
    }

        // Refine?

    bool bRefine = ( n > 1 );
    double gLastErr = 3;
    int nRefineCount = 0;

    while( bRefine ) {
        MTG_ASSERT( -S.m_nNumOfDownLevels < S.m_nNumOfUpLevels );

            // Compute residual r = m_Last - Matrix * m_B for improvement.
            // Also compute s = abs(Matrix) * abs(m_B) + abs(m_Last).

        double l, s, d, m, u, e, gErr;

        l = last( -S.m_nNumOfDownLevels, nSlot );
        m = S.m_DownRBWeights.m_gCurM * m_B[0];
        u = S.m_DownRBWeights.m_gCurU * m_B[1];

        m_R[0] = l - ( m + u );
        s = fabs( l ) + fabs( m ) + fabs( u );

        if( s > m_gSafe2 )
            gErr = fabs( m_R[0] ) / s;
        else
            gErr = ( fabs( m_R[0] ) + m_gSafe1 ) / ( s + m_gSafe1 );

        l = last( S.m_nNumOfUpLevels, nSlot );
        d = S.m_UpRBWeights.m_gCurD * m_B[n - 2];
        m = S.m_UpRBWeights.m_gCurM * m_B[n - 1];

        m_R[n - 1] = l - ( d + m );
        s = fabs( l ) + fabs( d ) + fabs( m );

        if( s > m_gSafe2 )
            e = fabs( m_R[n - 1] ) / s;
        else
            e = ( fabs( m_R[n - 1] ) + m_gSafe1 ) / ( s + m_gSafe1 );
        if( e > gErr )
            gErr = e;

        for( MTG_FOR_INIT( int ) i = 1; i < n - 1; ++i ) {
            int nLevel = -S.m_nNumOfDownLevels + i;
            tRBWeights& W = RBWeights( nLevel );

            l = last( nLevel, nSlot );
            d = W.m_gCurD * m_B[i - 1];
            m = W.m_gCurM * m_B[i];
            u = W.m_gCurU * m_B[i + 1];

            m_R[i] = l - ( d + m + u );
            s = fabs( l ) + fabs( d ) + fabs( m ) + fabs( u );

            if( s > m_gSafe2 )
                e = fabs( m_R[i] ) / s;
            else
                e = ( fabs( m_R[i] ) + m_gSafe1 ) / ( s + m_gSafe1 );
            if( e > gErr )
                gErr = e;
        }

        //if( nRefineCount > 0 )
        //    MTG_TRACE( "[%d] gErr = %lg\n", nRefineCount, gErr );

        if( gErr > m_gEps &&
            2 * gErr <= gLastErr &&
            ++nRefineCount <= m_nMaxRefineCount ) {
            gLastErr = gErr;

                // Repeat backsubstitution, this time with m_R
                // instead of m_B. Note that we also need to
                // do the pivoting, as m_R is in the original
                // order.

            for( int i = 0; i < n - 1; ++i ) {
                int nLevel = -S.m_nNumOfDownLevels + i;

                if( m_Pivot[i] == 0 ) {
                    m_R[i + 1] -= m_DL[i] * m_R[i];
                }
                else {
                    double f = m_R[i];
                    m_R[i] = m_R[i + 1];
                    m_R[i + 1] = f - m_DL[i] * m_R[i];
                }
            }

            m_R[n - 1] /= m_D[n - 1];
            m_R[n - 2] =
                ( m_R[n - 2] - m_DU[n - 2] * m_R[n - 1] ) / m_D[n - 2];

            for( MTG_FOR_INIT( int ) i = n - 3; i >= 0; --i ) {
                m_R[i] = ( m_R[i] - m_DU[i] * m_R[i + 1] -
                            m_DU2[i] * m_R[i + 2] ) / m_D[i];
            }

                // Do the correction!

            for( MTG_FOR_INIT( int ) i = 0; i < n; ++i ) {
                m_B[i] += m_R[i];
            }
        }
        else {
            bRefine = false;
        }
    }

        // Store values back.

    for( MTG_FOR_INIT( int ) i = 0; i < n; ++i ) {
        L.setCurValueAndTotal( -S.m_nNumOfDownLevels + i,
            S.m_nIndex, m_B[i], S.m_gMultiplier );
    }

    extrapolate( nSlot, true );
}


//
//   t O F I m p l i c i t
//

tOFImplicit::tOFImplicit( int nRootLevel, int nNumOfUpLevels,
    int nNumOfDownLevels )
    : super( nRootLevel, nNumOfUpLevels, nNumOfDownLevels )

{
    init();

    int n = rootLevel() + nNumOfUpLevels + 1;

    m_D.numOfElems( n );
    if( n > 1 ) {
        m_DL.numOfElems( n - 1 );
        m_DU.numOfElems( n - 1 );
        if( n > 2 )
            m_DU2.numOfElems( n - 2 );
    }
    m_B.numOfElems( n );
    m_R.numOfElems( n );
    m_Pivot.numOfElems( n );
}


//
//   t O F I m p l i c i t
//

tOFImplicit::tOFImplicit( const tOFImplicit& Implicit )

{
    init();
    copyFrom( Implicit );
}


//
//   ~ t O F I m p l i c i t
//

tOFImplicit::~tOFImplicit()

{
}


//
//   p r e p a r e
//

void tOFImplicit::prepare( tOFEngine& Engine, tSlotLayer& Layer,
     int nNumOfUpLevels, int nNumOfDownLevels,
     int nLastNumOfUpLevels, int nLastNumOfDownLevels )

{
    super::prepare( Engine, Layer, nNumOfUpLevels, nNumOfDownLevels,
        nLastNumOfUpLevels, nLastNumOfDownLevels );

        // Make sure we have space for each instrument, and
        // for each level.

    int n = rootLevel() + nNumOfUpLevels + 1;
    int m = layer().rowSize() - 1;    // number of slots

    if( m_Last.numOfCols() < m )
        m_Last.reset( m );
    if( m_Last.numOfRows() < n )
        m_Last.numOfRows( n );

    if( m_Temp3.numOfCols() < m )
        m_Temp3.reset( m );
    if( m_Temp3.numOfRows() < n )
        m_Temp3.numOfRows( n );
}


//
//   p r e p a r e
//

void tOFImplicit::prepare( int nIndex,
    int nNumOfUpLevels, tBoundary nUpBoundary,
    int nNumOfDownLevels, tBoundary nDownBoundary )

{
    super::prepare( nIndex, nNumOfUpLevels, nUpBoundary,
        nNumOfDownLevels, nDownBoundary );

    tSlotLayer& L = layer();
        
        // A new slot has already been created:

    int nSlot = numOfSlots() - 1;
    tSlot& S = slot( nSlot );

    MTG_ASSERT( S.m_nIndex == nIndex );

        // Compute right side of matrix equation for
        // implicit problem.

    int nLevel = -S.m_nNumOfDownLevels;
    tRBWeights* pW = &S.m_DownRBWeights;

        // Note that gLastD always exists, even at the boundary.

    double gLastD = L.lastValue( nLevel - 1, nIndex );
    double gLastM = L.lastValue( nLevel, nIndex );

    while( nLevel <= S.m_nNumOfUpLevels ) {
        double gLastU = L.lastValue( nLevel + 1, nIndex );

        last( nLevel, nSlot ) =
            gLastD * pW->m_gLastD +
            gLastM * pW->m_gLastM +
            gLastU * pW->m_gLastU;

        if( ++nLevel <= S.m_nNumOfUpLevels ) {
            gLastD = gLastM;
            gLastM = gLastU;        

            if( nLevel == S.m_nNumOfUpLevels )
                pW = &S.m_UpRBWeights;
            else
                pW = &RBWeights( nLevel );
        }
    }

        // If there is a barrier, we can bring the
        // known value to the right side (the side
        // of the past slice) right away.

    if( S.m_nDownBoundary == xGivenValue ) {
        last( -S.m_nNumOfDownLevels, nSlot ) -= S.m_DownRBWeights.m_gCurD *
            L.curValue( -S.m_nNumOfDownLevels - 1, nIndex );
        S.m_DownRBWeights.m_gCurD = 0;
    }

    if( S.m_nUpBoundary == xGivenValue ) {
        last( S.m_nNumOfUpLevels, nSlot ) -= S.m_UpRBWeights.m_gCurU *
            L.curValue( S.m_nNumOfUpLevels + 1, nIndex );
        S.m_UpRBWeights.m_gCurU = 0;
    }
}


//
//   r e f i n e
//

tRetCode tOFImplicit::refine( tIncrement& Incr )

{
        // One over-relaxation step consists of computing
        // 
        //  Wi = ( Zi - Ai V[i-1] - Ci V[i+1] ) / Bi
        //
        // and then replacing Vi with Vi + omega ( Wi - Vi ).
        // The error is the sum over [ omega ( Wi - Vi ) ]^2.
    
        // First, establish the range over which the relaxation
        // is performed.

    tSlotLayer& L = layer();

        // Extrapolation data:

    bool bUpExtra = false;
    bool bDownExtra = false;
    int nUpExLevel, nDownExLevel;

    int nNumOfUpLevels, nNumOfDownLevels;

    for( int i = 0; i < numOfSlots(); ++i ) {
        tSlot& S = slot( i );

        int nUp = S.m_nNumOfUpLevels;
        int nDown = S.m_nNumOfDownLevels;

        if( i == 0 || nUp > nNumOfUpLevels )
            nNumOfUpLevels = nUp;
        if( i == 0 || nDown > nNumOfDownLevels )
            nNumOfDownLevels = nDown;

        if( S.m_nUpBoundary == xZeroGamma ) {
            if( bUpExtra ) {
                MTG_ASSERT( nUpExLevel == nUp + 1 );
            }
            else {
                nUpExLevel = nUp + 1;
                bUpExtra = true;
            }
        }

        if( S.m_nDownBoundary == xZeroGamma ) {
            if( bDownExtra ) {
                MTG_ASSERT( nDownExLevel == -nDown - 1 );
            }
            else {
                nDownExLevel = -nDown - 1;
                bDownExtra = true;
            }
        }
    }

        // Note that the extend of the levels of each instrument
        // and the boundary condition is uniform under non-linearity.
        // This function should, however, also work under linearity
        // conditions.

#if defined(_DEBUG)
    for( MTG_FOR_INIT( int ) i = 0; i < numOfSlots(); ++i ) {
        if( slot( i ).m_nNumOfUpLevels < nNumOfUpLevels )
            MTG_ASSERT( slot( i ).m_nUpBoundary == xGivenValue );
        if( slot( i ).m_nNumOfDownLevels < nNumOfDownLevels )
            MTG_ASSERT( slot( i ).m_nDownBoundary == xGivenValue );
    }
#endif

    int nIterCount = 0;
    double gLastError;
    tRetCode nRet = OK;
    tHeap<int> Pos;

    Pos.numOfElems( 1 );

    while( true ) {
            // Now do the refinement. Loop over levels, then instruments,
            // to follow the Gauss-Seidel method of reusing refined
            // values (including early exercise) immediately.

        saveCurValues();
        Incr.beginIncrement( nNumOfDownLevels, nNumOfUpLevels );

        for( int nLevel = -nNumOfDownLevels;
                nLevel <= nNumOfUpLevels; ++nLevel ) {
            for( int i = 0; i < numOfSlots(); ++i ) {
                const tRBWeights* pW = getRBWeights( nLevel, i );

                if( pW == 0 )
                    continue;

                tSlot& S = slot( i );

                double& Value = L.curValue( nLevel, S.m_nIndex ); 

                double gDiff = m_gOmega *
                    ( ( last( nLevel, i ) -
                            pW->m_gCurD * L.curValue( nLevel - 1, S.m_nIndex ) -
                            pW->m_gCurU * L.curValue( nLevel + 1, S.m_nIndex ) ) /
                                pW->m_gCurM - Value );

                Value += gDiff;
                L.curTotal( nLevel ) += gDiff * S.m_gMultiplier;
            }

                // Do the monitoring:

            Pos[0] = nLevel;
            Incr.doIncrement( Pos );
        }

        int nAdjUp = nNumOfUpLevels;
        int nAdjDown = nNumOfDownLevels;

            // Now take care of the boundary.                    

        if( bUpExtra || bDownExtra ) {
            for( int i = 0; i < numOfSlots(); ++i )
                extrapolate( i, false );

            if( bUpExtra ) {
                Pos[0] = nUpExLevel;
                Incr.doIncrement( Pos );
                ++nAdjUp;
            }

            if( bDownExtra ) {
                Pos[0] = nDownExLevel;
                Incr.doIncrement( Pos );
                ++nAdjDown;
            }
        }

            // endIncrement might do some more refinement:

        Incr.endIncrement( nAdjDown, nAdjUp );
        ++nIterCount;

        double gError = calcError();
        //MTG_TRACE( "Iteration %d: error %lg, omega %lg\n",
        //    nIterCount, gError, m_gOmega );

        if( gError <= m_gErrorThreshold )
            break;

        if( nIterCount > 1 && 2 * gError > gLastError )
            break;

        if( nIterCount >= m_nMaxIterCount ) {
            nRet = NO_CONVERGENCE;
            break;
        }

        gLastError = gError;
    }

        // Adjust the over relaxation parameter.

    if( m_nEstIterCount > 0 && nIterCount > m_nEstIterCount )
        m_gOmegaStep = -m_gOmegaStep;

    m_gOmega += m_gOmegaStep;
    if( m_gOmega < 1 )
        m_gOmega = 1;
    else
    if( m_gOmega >= 2 )
        m_gOmega = 2 - m_gOmegaStep;

    m_nEstIterCount = nIterCount;

    return nRet;
}

MTG_END_NAMESPACE