// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#include "MtgIncl.h"
#include "MtgOFSolver.h"
#include "MtgOFEngine.h"
#include "MtgSlotLayer.h"

MTG_BEGIN_NAMESPACE


//
//   i n i t
//

void tOFSolver::init()

{
    m_pEngine = 0;
    m_pLayer = 0;
}


//
//   t O F S o l v e r
//

tOFSolver::tOFSolver()

{
    init();
}


//
//   c o p y F r o m
//

void tOFSolver::copyFrom( const tOFSolver& Solver )

{
    if( this == &Solver )
        return;

    m_nRootLevel = Solver.m_nRootLevel;
    m_pLayer = Solver.m_pLayer;
    m_pEngine = Solver.m_pEngine;
    m_gAbsDuration = Solver.m_gAbsDuration;
    m_gSqrtAbsDuration = Solver.m_gSqrtAbsDuration;
    m_gRelDuration = Solver.m_gRelDuration;
    m_RBWeights = Solver.m_RBWeights;
    m_Slot = Solver.m_Slot;
}


//
//   c o p y C u r T o P r e p
//

void tOFSolver::copyCurToPrep( int nSlot, bool bInitial )

{
    tSlotLayer& L = layer();
    tSlot& S = slot( nSlot );

    int nDown = -S.m_nNumOfDownLevels;
    int nUp = S.m_nNumOfUpLevels;

        // Don't forget the extrapolated nodes:

    if( S.m_nDownBoundary == xZeroGamma )
        --nDown;
    if( S.m_nUpBoundary == xZeroGamma )
        ++nUp;

        // Now copy current values to preparatory storage:

    if( bInitial ) {
        for( int k = nDown; k <= nUp; ++k ) {
            L.setPrepValueAndTotal( k, S.m_nIndex,
                L.curValue( k, S.m_nIndex ), S.m_gMultiplier );
        }
    }
    else {
        for( int k = nDown; k <= nUp; ++k ) {
            L.replPrepValueAndTotal( k, S.m_nIndex,
                L.curValue( k, S.m_nIndex ), S.m_gMultiplier );
        }
    }
}


//
//   g e t R B W e i g h t s
//

const tOFSolver::tRBWeights* tOFSolver::getRBWeights( int nLevel, int nSlot )

{
    tSlot& S = slot( nSlot );

        // Choose the right weight. Note that we assume that
        // m_gCurU resp. m_gCurD are set to 0 at the boundary
        // levels.

    if( nLevel > -S.m_nNumOfDownLevels &&
        nLevel < S.m_nNumOfUpLevels ) {
        return &RBWeights( nLevel );
    }

    if( nLevel == -S.m_nNumOfDownLevels ) {
        MTG_ASSERT( S.m_DownRBWeights.m_gCurD == 0 );
        return &S.m_DownRBWeights;
    }

    if( nLevel == S.m_nNumOfUpLevels ) {
        MTG_ASSERT( S.m_UpRBWeights.m_gCurU == 0 );
        return &S.m_UpRBWeights;
    }

    return 0;
}


//
//   t O F S o l v e r
//

tOFSolver::tOFSolver( int nRootLevel, int nNumOfUpLevels,
    int nNumOfDownLevels )

{
    MTG_ASSERT( nRootLevel >= nNumOfDownLevels );

    init();

    m_nRootLevel = nRootLevel;
    m_RBWeights.numOfElems( m_nRootLevel + nNumOfUpLevels + 1 );
}


//
//   t O F S o l v e r
//

tOFSolver::tOFSolver( const tOFSolver& Solver )

{
    init();
    copyFrom( Solver );
}


//
//   ~ t O F S o l v e r
//

tOFSolver::~tOFSolver()

{
}


//
//   p r e p a r e
//

void tOFSolver::prepare( tOFEngine& Engine, tSlotLayer& Layer,
     int nNumOfUpLevels, int nNumOfDownLevels,
     int nLastNumOfUpLevels, int nLastNumOfDownLevels )

{
        // Prepare the rollback for the entire slice. (The following
        // prepare() function prepares the rollback for individual
        // instruments, subsequently.)

    m_pEngine = &Engine;
    m_pLayer = &Layer;

        // Notice that m_gAbsDuration may be based on days or years,
        // depending on the preferences of the engine. The engine,
        // of course, has to provide the correct parameters for
        // the duration.

    m_gAbsDuration = m_pEngine->scaledAbsDuration();
    m_gSqrtAbsDuration = sqrt( m_gAbsDuration );
    m_gRelDuration = m_pEngine->relDuration();

    int nLevel = -nNumOfDownLevels;

    while( nLevel <= nNumOfUpLevels ) {
        int nFromLevel = nLevel;
        const tProcessParamsStub& Params =
                m_pEngine->getProcessParams( nLevel, nNumOfUpLevels );

            // nLevel has been advanced over the entire uniform sequence

        calcWeights( nFromLevel, nLevel, Params );
        ++nLevel;
    }

    m_Slot.reset();
}


//
//   p r e p a r e
//

void tOFSolver::prepare( int nIndex,
    int nNumOfUpLevels, tBoundary nUpBoundary,
    int nNumOfDownLevels, tBoundary nDownBoundary )

{
    MTG_ASSERT( m_pLayer != 0 );

    ++m_Slot;
    tSlot& S = m_Slot.last();

    S.m_nIndex = nIndex;
    S.m_gMultiplier = engine().multiplier( nIndex );
    S.m_nNumOfUpLevels = nNumOfUpLevels;
    S.m_nNumOfDownLevels = nNumOfDownLevels;
    S.m_nUpBoundary = nUpBoundary;
    S.m_nDownBoundary = nDownBoundary;

        // For zero gamma boundary conditions, the last level 
        // is not computed via rollback, but rather via
        // extrapolation. Remove boundary level from implicit/explicit
        // problem and adjust weights (this depends on the
        // geometry and must be done in a subclass). ..ZGWeights
        // are used for extrapolation.

    if( S.m_nDownBoundary == xZeroGamma ) {
        int k = --S.m_nNumOfDownLevels;

        S.m_DownRBWeights = RBWeights( -k );
        adjustZeroGammaDownBoundary( -k,
            S.m_DownRBWeights, S.m_DownZGWeights );
    }
    else {
        S.m_DownRBWeights = RBWeights( -S.m_nNumOfDownLevels );
    }

    if( S.m_nUpBoundary == xZeroGamma ) {
        int k = --S.m_nNumOfUpLevels;

        S.m_UpRBWeights = RBWeights( k );
        adjustZeroGammaUpBoundary( k, S.m_UpRBWeights, S.m_UpZGWeights );
    }
    else {
        S.m_UpRBWeights = RBWeights( S.m_nNumOfUpLevels );
    }

        // If there are at least 5 levels in the lattice,
        // there cannot be two zero gamma boundary conditions
        // for only one remaining level at the same time.

    MTG_ASSERT( S.m_nNumOfUpLevels >= -S.m_nNumOfDownLevels );
    MTG_ASSERT( S.m_nNumOfUpLevels > -S.m_nNumOfDownLevels ||
                S.m_nUpBoundary != xZeroGamma ||
                S.m_nDownBoundary != xZeroGamma );
}


//
//   s o l v e
//

void tOFSolver::solve()

{
    for( int i = 0; i < numOfSlots(); ++i )
        solve( i );
}


//
//   r e f i n e
//

tRetCode tOFSolver::refine( tIncrement& Incr )

{
    return OK;  // don't do anything; override in the implicit solver
}

MTG_END_NAMESPACE