// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#if ! defined(_MTG_PATH_SPACE_)
#define _MTG_PATH_SPACE_

#include "MtgDateRange.h"
#include "MtgFileName.h"
#include "MtgHeap.h"
#include "MtgObject.h"

MTG_BEGIN_NAMESPACE

class tModel;
class tParser;
class tPortfolio;
class tRandom;
class tSamplePath;


//
//   t P a t h S p a c e
//

class tPathSpace : public tObject, public tDateRange {

    typedef tObject super;

    struct tParseInfo : super::tParseInfoStub {
        bool m_bBase;
        bool m_bScale;
        bool m_bDayCount;
        bool m_bSeed;
        bool m_bSamples;
    };

    tDate m_Maturity;           // entered or from portfolio
    int m_nNumOfFactors;        // factors per sample point

    long m_nSeed1;              // for random numbers
    long m_nSeed2;              // for random numbers
    int m_nSize;                // how many paths?

    bool m_bAntithetic;         // use antithetic variables?

    bool m_bHasWeights;         // are there individual weigths
                                // per path?
    tHeap<double> m_Weight;     // if yes, they are here!
    tFileName m_WeightFile;     // ...loaded from here
    double m_gGlobalWeight;     // otherwise, this is just 1/m_nSize

    tFileName m_OutFile;        // for saving the entire space, later

    tModel* m_pModel;           // reference
    tPortfolio* m_pPortfolio;   // reference

    void init();
    void copyFrom( const tPathSpace& Space );

    tRetCode loadWeights();
    void normalizeWeights();

    tRetCode parsePrefix( tParser& Parser, tParseInfoStub& Info );
    tRetCode parseParam( tParser& Parser, tParseInfoStub& Info );
    tRetCode parsePostfix( tParser& Parser, tParseInfoStub& Info );

public:

    tPathSpace();
    tPathSpace( const tPathSpace& Space );

    ~tPathSpace();

    tPathSpace& operator=( const tPathSpace& Space );

    tObject* clone() const;

    tRetCode finalize();

    tDate maturity() const {
        MTG_ASSERT( isFinalized() );
        return m_Maturity;
    }

    int numOfFactors() const {
        MTG_ASSERT( isFinalized() );
        return m_nNumOfFactors;
    }

    int size() const {
        MTG_ASSERT( isFinalized() );
        return m_nSize;
    }

        // The model is not the primary information of the
        // path space, but it might hold a reference to it. 

    bool hasModel() const;
    tModel& model() const;

    bool hasPortfolio() const;
    tPortfolio& portfolio() const;

        // A model matches if the number of factors are
        // the same, or if m_pModel points to the same model.

    bool modelMatches( const tModel& Model ) const;

    void generatePath( tSamplePath& Path, tRandom& Random,
        int nSeqNo, double& gWeight ) const;

    void generatePath( tSamplePath& Path, tRandom& Random,
        int nSeqNo ) const;

        // Output the entire object, plus extra output defined
        // in attributes (if the weight parameter is present,
        // it will be substituted for the existing weights):

    tRetCode save( ostream& Out, const tHeap<double>* pWeight ) const;
    tRetCode save( ostream& Out, const tHeap<double>& Weight ) const;
    tRetCode save( ostream& Out ) const;

        // In this case, the filename overrides m_OutFile if it is
        // not empty.

    tRetCode save( const tFileName& Out, const tHeap<double>* pWeight ) const;
    tRetCode save( const tFileName& Out, const tHeap<double>& Weight ) const;
    tRetCode save( const tFileName& Out ) const;

        // Finally, here we list only if m_OutFile is valid.

    tRetCode save( const tHeap<double>* pWeight ) const;
    tRetCode save( const tHeap<double>& Weight ) const;
    tRetCode save() const;

    static tRetCode parse( tParser& Parser, tSystem& System, tObject*& pObj );
};

MTG_END_NAMESPACE

#endif

