// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#include "MtgIncl.h"
#include "MtgPortfolio.h"
#include "MtgCashflow.h"
#include "MtgClaim.h"
#include "MtgDataScanner.h"
#include "MtgEvGenerator.h"
#include "MtgFactor.h"
#include "MtgModel.h"
#include "MtgParser.h"

#include <float.h>

MTG_BEGIN_NAMESPACE


//
//   t A u x
//

class tPortfolio::tAux {
public:

    tPortfolio::tWrapper *m_pWrapper;
    tHeap<tAux*> m_aChild;
    int m_nNumOfParents;

    int m_nHeapPos;
    int m_nMinSlot;
    int m_nWeight;
    
    int cmp( const tAux* A ) const {
        if( m_nNumOfParents < A->m_nNumOfParents )
            return -1;
        if( m_nNumOfParents > A->m_nNumOfParents )
            return 1;

        tClaim& C1 = *m_pWrapper->m_pClaim;
        tClaim& C2 = *A->m_pWrapper->m_pClaim;

        if( C1.maturity() > C2.maturity() )
            return -1;
        if( C2.maturity() < C2.maturity() )
            return 1;

        if( m_nWeight > A->m_nWeight )
            return -1;
        if( m_nWeight < A->m_nWeight )
            return 1;

        if( C1.birth() < C2.birth() )
            return -1;
        if( C1.birth() > C2.birth() )
            return 1;

        return 0;
    }
};


//
//   i n i t
//

void tPortfolio::init()

{
}


//
//   c l e a n u p
//

void tPortfolio::cleanup()

{
    for( int k = 0; k < m_aWrapper.numOfElems(); ++k )
        delete m_aWrapper[k].m_pClaim;
}


//
//   c o p y F r o m
//

void tPortfolio::copyFrom( const tPortfolio& Pf )

{
    if( &Pf == this )
        return;

    cleanup();

    m_aWrapper.copyFrom( Pf.m_aWrapper );
    m_aDep.copyFrom( Pf.m_aDep );

    for( int k = 0; k < m_aWrapper.numOfElems(); ++k ) {
        m_aWrapper[k].m_pClaim = 
            static_cast<tClaim*>( m_aWrapper[k].m_pClaim->clone() );
    }

    m_ClaimLkup = Pf.m_ClaimLkup;
    m_FactorLkup = Pf.m_FactorLkup;

    m_Factor = Pf.m_Factor;

    m_MultiplierFile = Pf.m_MultiplierFile;

    if( Pf.isFinalized() ) {
        buildId2Wrapper();
        buildCategories();
            // This is all what's needed! The rest is in m_aWrapper.
    }

    super::copyFrom( Pf );
}


//
//   h a s h
//

int tPortfolio::hash( int nId ) const

{
    int h = nId % m_aId2Wrapper.numOfElems();
    if( h < 0 )
        h = -h;
    return h;
}


//
//   f i n d
//

tPortfolio::tWrapper* tPortfolio::find( int nId ) const

{
    int h = h = hash( nId );
    int n = m_aId2Wrapper.numOfElems();

    if( m_aId2Wrapper[h] != 0 ) {
        if( m_aId2Wrapper[h]->m_pClaim->id() == nId )
            return m_aId2Wrapper[h];

        int h1 = 1;
        while( true ) {
            MTG_ASSERT( h1 < n );

            h = h + h1;
            if( h >= n )
                h -= n;
            if( m_aId2Wrapper[h] == 0 )
                break;
            if( m_aId2Wrapper[h]->m_pClaim->id() == nId )
                return m_aId2Wrapper[h];
            h1 += 2;
        }
    }

    return 0;
}


//
//   b u i l d I d 2 W r a p p e r
//

void tPortfolio::buildId2Wrapper()

{
    int n = NextPrime( m_aWrapper.numOfElems() * 2 + 1 );

    m_aId2Wrapper.numOfElems( n );
    for( int k = 0; k < n; ++k )
        m_aId2Wrapper[k] = 0;

    for( MTG_FOR_INIT( int ) k = 0; k < m_aWrapper.numOfElems(); ++k ) {
        int h = hash( m_aWrapper[k].m_pClaim->id() );
        if( m_aId2Wrapper[h] != 0 ) {
            int h1 = 1;

            do{
                // note that we cannot have a table overflow, since
                // there are at least twice as many hash entries as
                // basic instruments, and quadratic hashing covers
                // at least half of the hash table:
                MTG_ASSERT( h1 < n );

                h = h + h1;
                if( h >= n )
                    h -= n;
                h1 += 2;
            } while( m_aId2Wrapper[h] != 0 );
        }

        m_aId2Wrapper[h] = &m_aWrapper[k];
    }
}


//
//   s o r t
//

void tPortfolio::sort( tHeap<tWrapper*>& aHeap, int tClaim::* pField )

{
    int m = aHeap.numOfElems();
    int gLognb2 = (int) ( log( m ) * 1.442695022 + 1.0e-5 );

        // Shellsort for a change; ses Wirth's book.

    for( int nn = 1; nn <= gLognb2; ++nn ) {
        m /= 2;
        for( int j = m; j < aHeap.numOfElems(); ++j ) {
            int i = j - m;
            tWrapper* t = aHeap[j];
            while( i >= 0 &&
                t->m_pClaim->*pField > aHeap[i]->m_pClaim->*pField ) {
                aHeap[i + m] = aHeap[i];
                i -= m;
            }
            aHeap[i + m] = t;
        }
    }
}


//
//   b u i l d C a t e g o r i e s
//

void tPortfolio::buildCategories()

{
    m_aAuxByBirth.reset();
    m_aAuxByMaturity.reset();
    m_Priced.reset();
    m_NotPriced.reset();
    m_Sorted.reset();

    m_nMaturity = 0;
    m_MaturityDate = system().base();

    if( m_aWrapper.numOfElems() == 0 )
        return;

    for( int k = 0; k < m_aWrapper.numOfElems(); ++k ) {
        if( m_aWrapper[k].m_pClaim->isAuxiliary() ) {

            ++m_aAuxByBirth;
            ++m_aAuxByMaturity;
            m_aAuxByBirth.last() = &m_aWrapper[k];
            m_aAuxByMaturity.last() = &m_aWrapper[k];
        }
        else {
            m_aWrapper[k].m_nIndex = m_Sorted.numOfElems();

            if( m_aWrapper[k].m_pClaim->isPriced() ) {
                ++m_Priced;
                ++m_Sorted;
                m_Priced.last() = &m_aWrapper[k];
                m_Sorted.last() = &m_aWrapper[k];
            }
            else {
                ++m_NotPriced;
                ++m_Sorted;
                m_NotPriced.last() = &m_aWrapper[k];
                m_Sorted.last() = &m_aWrapper[k];
            }
        }
        m_aWrapper[k].m_pClaim->setIndex( m_aWrapper[k].m_nIndex );
    }

    sort( m_aAuxByBirth, &tClaim::m_nBirth );
    sort( m_aAuxByMaturity, &tClaim::m_nMaturity );
    sort( m_Sorted, &tClaim::m_nMaturity );

    if( m_aAuxByMaturity.numOfElems() > 0 ) {
        m_nMaturity = m_aAuxByMaturity[0]->m_pClaim->maturity();
        m_MaturityDate = m_aAuxByMaturity[0]->m_pClaim->maturityDate();

        for( k = 1; k < m_aAuxByMaturity.numOfElems(); ++k ) {
            if( m_aAuxByMaturity[k]->m_pClaim->maturity() < m_nMaturity )
                break;

            tDate M = m_aAuxByMaturity[k]->m_pClaim->maturityDate();
            if( M > m_MaturityDate )
                m_MaturityDate = M;
        }
    }

    if( m_Sorted.numOfElems() > 0 ) {
        if( m_nMaturity < m_Sorted[0]->m_pClaim->maturity() ) {
            m_nMaturity = m_Sorted[0]->m_pClaim->maturity();
            m_MaturityDate = m_Sorted[0]->m_pClaim->maturityDate();

            for( k = 1; k < m_Sorted.numOfElems(); ++k ) {
                if( m_Sorted[k]->m_pClaim->maturity() < m_nMaturity )
                    break;

                tDate M = m_Sorted[k]->m_pClaim->maturityDate();
                if( M > m_MaturityDate )
                    m_MaturityDate = M;
            }
        }
    }
}


//
//   d e m o t e I n A u x
//

void tPortfolio::demoteInAux( int nPos, int nSize,
    tHeap<tAux*>& aAux ) const

{
    MTG_ASSERT( nSize <= aAux.numOfElems() );

    int i = nPos;
    int j = 2 * i + 1;
    tAux* x = aAux[i];

    while( j < nSize ) {
        if( j < nSize - 1 && aAux[j]->cmp( aAux[j + 1] ) > 0 )
            ++j;
        if( x->cmp( aAux[j] ) <= 0 )
            break;
        aAux[i] = aAux[j];
        aAux[i]->m_nHeapPos = i;
        i = j;
        j = 2 * i + 1;
    }

    aAux[i] = x;
    x->m_nHeapPos = i;
}


//
//   p r o m o t e I n A u x
//

void tPortfolio::promoteInAux( int nPos, tHeap<tAux*>& aAux ) const

{
    tAux* x = aAux[nPos];

        // Never touch position 0.

    while( nPos > 2 ) {
        int j = ( nPos - 1 ) / 2;
        if( x->cmp( aAux[j] ) >= 0 )
            break;
        aAux[nPos] = aAux[j];
        aAux[nPos]->m_nHeapPos = nPos;
        nPos = j;
    }

    aAux[nPos] = x;
    x->m_nHeapPos = nPos;
}


//
//   b u i l d A u x H e a p
//

tRetCode tPortfolio::buildAuxHeap( tHeap<tAux*>& aAux ) const

{
    tRetCode nRet;
    tWrapper *pW1, *pW2;

    aAux.numOfElems( m_aAuxByMaturity.numOfElems() );

    for( int k = 0; k < m_aAuxByMaturity.numOfElems(); ++k ) {
        m_aAuxByMaturity[k]->m_nSlot = k;
        aAux[k] = new tAux;
        aAux[k]->m_pWrapper = m_aAuxByMaturity[k];
        aAux[k]->m_nNumOfParents = 0;
        aAux[k]->m_nHeapPos = k;
        aAux[k]->m_nMinSlot = 0;
        aAux[k]->m_nWeight = 0;
    }

    for( MTG_FOR_INIT( int ) k = 0; k < m_aDep.numOfElems(); ++k ) {
        if( ( pW1 = find( m_aDep[k].m_nDepOnId ) ) == 0 ||
            ( pW2 = find( m_aDep[k].m_nHiLevId ) ) == 0 ) {
            nRet = DANGLING_DEPENDENCY;
            goto error;
        }

        if( ! pW1->m_pClaim->isAuxiliary() ) {
            nRet = ILLEGAL_DEPENDENCY;
            goto error;
        }

        if( pW2->m_pClaim->isAuxiliary() ) {
            tAux& A1 = *aAux[pW1->m_nSlot];
            tAux& A2 = *aAux[pW2->m_nSlot];

            ++A1.m_aChild;
            A1.m_aChild.last() = &A2;
            A1.m_nWeight += A2.m_nWeight + 1;

            ++A2.m_nNumOfParents;
        }
    }

    for( MTG_FOR_INIT( int ) k = m_aAuxByMaturity.numOfElems() / 2 - 1; 
         k >= 0; --k ) {
        demoteInAux( k, m_aAuxByMaturity.numOfElems(), aAux );
    }
 
    return OK;

error:

    for( MTG_FOR_INIT( int ) k = 0; k < aAux.numOfElems(); ++k )
        delete aAux[k];
    aAux.reset();

    return nRet;
}


//
//   a s s i g n A u x S l o t s
//

tRetCode tPortfolio::assignAuxSlots()

{
    tRetCode nRet;
    tHeap<tAux*> aAux;

    m_nNumOfAuxSlots = 0;

    if( m_aAuxByMaturity.numOfElems() == 0 )
        return OK;

    if( ( nRet = buildAuxHeap( aAux ) ) != OK )
        return nRet;

    int* aSlot = new int[aAux.numOfElems()];

    for( int k = aAux.numOfElems() - 1; k >= 0; --k ) {
        tAux& A = *aAux[0];

        MTG_ASSERT( A.m_nHeapPos == 0 );

        if( A.m_nNumOfParents > 0 ) {
            delete[] aSlot;
            nRet = CYCLIC_DEPENDENCY;
            goto stop;
        }

        tWrapper* pWrapper = A.m_pWrapper;
        int nSlot = A.m_nMinSlot;

        while( nSlot < m_nNumOfAuxSlots &&
                aSlot[nSlot] <= pWrapper->m_pClaim->m_nMaturity ) {
            ++nSlot;
        }

        MTG_ASSERT( nSlot < aAux.numOfElems() );

        pWrapper->m_nSlot = nSlot;
        aSlot[nSlot] = pWrapper->m_pClaim->m_nBirth;
        if( nSlot == m_nNumOfAuxSlots )
            ++m_nNumOfAuxSlots;

        for( int j = 0; j < A.m_aChild.numOfElems(); ++j ) {
            MTG_ASSERT( A.m_aChild[j]->m_nNumOfParents > 0 );
            if( A.m_aChild[j]->m_nMinSlot <= nSlot )
                A.m_aChild[j]->m_nMinSlot = nSlot + 1;
            --A.m_aChild[j]->m_nNumOfParents;
            promoteInAux( A.m_aChild[j]->m_nHeapPos, aAux );
        }

        if( k > 0 ) {
            tAux *t = aAux[0];
            aAux[0] = aAux[k];
            aAux[k] = t;
            aAux[0]->m_nHeapPos = 0;
            demoteInAux( 0, k, aAux );
        }
    }

    delete[] aSlot;
    nRet = OK;

stop:

    for( MTG_FOR_INIT( int ) k = 0; k < aAux.numOfElems(); ++k )
        delete aAux[k];
    aAux.reset();

    return nRet;
}


//
//   a s s i g n N o n A u x S l o t s
//

tRetCode tPortfolio::assignNonAuxSlots()

{
    for( int k = 0; k < m_Sorted.numOfElems(); ++k )
        m_Sorted[k]->m_nSlot = k;
    m_nNumOfNonAuxSlots = m_Sorted.numOfElems();
    return OK;
}


//
//   p a r s e C l a i m
//

tRetCode tPortfolio::parseClaim( tParser& Parser, tClaim*& pClaim )

{
    MTG_ASSERT( Parser.curToken() == xTokClaim ||
                Parser.curToken() == xTokId ||
                Parser.beginOfObj() );

    tRetCode nRet;
    char* sName;
    int nId;

    if( Parser.curToken() == xTokClaim ) {
        if( ( nRet = Parser.readToken() ) != OK )
            return nRet;
    }

    if( Parser.curToken() == xTokId ) {
        if( m_ClaimLkup.retrieve( Parser.curText(), nId ) == OK )
            return Parser.setError( "name exists already in portfolio" );

        sName = StrCopy( Parser.curText() );
        if( ( nRet = Parser.readToken() ) != OK ) {
            delete sName;
            return nRet;
        }
    }
    else {
        sName = 0;
    }

    if( Parser.beginOfObj() ) {
        tObject *pObj;

        if( ( nRet = tClaim::parse( Parser, system(), pObj ) ) != OK ) {
            if( sName != 0 )
                delete sName;
            return nRet;
        }
        pClaim = static_cast<tClaim*>( pObj );
        if( sName != 0 )
            pClaim->setName( sName );
    } 
    else {
        if( sName == 0 )
            return Parser.setError( "claim name expected" );

        tObject* pObj = Parser.findObject( sName );

        if( pObj == 0 ) {
            delete sName;
            return Parser.setError( NOT_FOUND );
        }
        if( dynamic_cast<tClaim*>( pObj ) == 0 ) {
            delete sName;
            return Parser.setError( "claim expected" );
        }
        pClaim = static_cast<tClaim*>( pObj->clone() );
    }

    if( sName != 0 ) {
        nId = pClaim->id();
        if( ( nRet = m_ClaimLkup.insert( sName, nId ) ) != OK ) {
            delete sName;
            delete pClaim;
            return Parser.setError( nRet );
        }
        delete sName;
    }

    return OK;
}


//
//   p a r s e P a r a m
//

tRetCode tPortfolio::parseParam( tParser& Parser, tParseInfoStub& Info )

{
    tRetCode nRet;
    tClaim* pClaim;

    if( Parser.curToken() == xTokClaim ||
        Parser.curToken() == xTokId ||
        Parser.beginOfObj() ) {

        if( ( nRet = parseClaim( Parser, pClaim ) ) != OK )
            return nRet;

            // Can override multiplier:

        if( Parser.curToken() == xTokMultiplier ||
            Parser.curToken() == xTokPosition ||
            Parser.curToken() == xTokLong ||
            Parser.curToken() == xTokShort ||
            Parser.beginOfNumber() ) {

            bool bMult = false;
            if( ( nRet = pClaim->parseMultiplier( Parser, bMult ) ) != OK ) {                            
                delete pClaim;
                return nRet;
            }
        }

        regClaim( pClaim );
    }
    else {
        switch( Parser.curToken() ) {
            case xTokMultiplier :
                if( m_MultiplierFile.isValid() ) {
                    return Parser.setError(
                        "multiplier file defined more than once" );
                }
                if( ( nRet = Parser.readToken() ) != OK ||
                    ( nRet = m_MultiplierFile.parseRead( Parser ) ) != OK ) {
                    return nRet;
                }
                break;

            default :
                return super::parseParam( Parser, Info );
        }
    }

    return OK;
}


//
//   p a r s e P o s t f i x
//

tRetCode tPortfolio::parsePostfix( tParser& Parser, tParseInfoStub& Info )

{
    tRetCode nRet;
    
    m_FactorLkup.reset();
    m_Factor.reset();

    for( int k = 0; k < m_aWrapper.numOfElems(); ++k ) {
        if( ( nRet = m_aWrapper[k].m_pClaim->resolve(
                        m_ClaimLkup, m_FactorLkup,
                        m_Factor, *this ) ) != OK ) {
            return Parser.setError( nRet );
        }
    }

    buildId2Wrapper();

    if( m_MultiplierFile.isValid() != 0 ) {
        tDataScanner Scanner;

        if( ( nRet = Scanner.setSource( m_MultiplierFile ) ) != OK ) {
            if( nRet != GO )
                return Parser.setError( nRet );
        }
        else {
            char* sName;
            int nIndex, nId; 
            double gMultiplier;

            while( ! Scanner.atEOF() ) {
                if( ( nRet = Scanner.scanString( sName ) ) != OK )
                    return Parser.setError( nRet );
                if( ( nRet = Scanner.scanInteger( nIndex, 0 ) ) != OK ||
                    ( nRet = Scanner.scanDouble( gMultiplier ) ) != OK ) {
                    delete sName;
                    return Parser.setError( nRet );
                }

                    // change multiplier, if name is known
                if( m_ClaimLkup.retrieve( sName, nId ) == OK )
                    m_aId2Wrapper[nId]->m_pClaim->modifyMultiplier( gMultiplier );
                delete sName;
            }
        }
    }

    return super::parsePostfix( Parser, Info );
}


//
//   t P o r t f o l i o
//

tPortfolio::tPortfolio()

{
    init();
}


//
//   t P o r t f o l i o
//

tPortfolio::tPortfolio( const tPortfolio& Pf )

{
    init();
    copyFrom( Pf );
}


//
//   ~ t P o r t f o l i o
//

tPortfolio::~tPortfolio()

{
    cleanup();
}


//
//   o p e r a t o r =
//

tPortfolio& tPortfolio::operator=( const tPortfolio& Pf )

{
    if( &Pf != this )
        copyFrom( Pf );
    return *this;
}


//
//   c l o n e
//

tObject* tPortfolio::clone() const

{
    return new tPortfolio( *this );
}


//
//   r e g C l a i m
//

void tPortfolio::regClaim( tClaim& Claim )

{
    regClaim( &Claim );
}


//
//   r e g C l a i m
//

void tPortfolio::regClaim( tClaim* pClaim )

{
    MTG_ASSERT( pClaim != 0 );

    ++m_aWrapper;
    m_aWrapper.last().m_pClaim = pClaim;
    m_aWrapper.last().m_nIndex = -1;
    m_aWrapper.last().m_nSlot = -1;

    touch();
}


//
//   r e g D e p e n d e n c y
//

void tPortfolio::regDependency( int nHiLevId, int nDepOnId )

{
    if( nHiLevId == nDepOnId )
        return;

    ++m_aDep;
    m_aDep.last().m_nHiLevId = nHiLevId;
    m_aDep.last().m_nDepOnId = nDepOnId;

    touch();
}


//
//   f i n a l i z e
//

tRetCode tPortfolio::finalize()

{
    tRetCode nRet;

    if( isFinalized() )
        return OK;

    buildCategories();

    if( ( nRet = assignAuxSlots() ) != OK ||
        ( nRet = assignNonAuxSlots() ) != OK ) {
        return nRet;
    }

#if defined(_DEBUG)

    /*MTG_TRACE( "Portfolio {\n" );

    if( m_NotPriced.numOfElems() > 0 ) {
        for( int k = 0; k < m_NotPriced.numOfElems(); ++k ) {
            MTG_TRACE( "    #%02d: id %d, pos %lg, life %d-%d\n", k + 1,
                m_NotPriced[k]->m_pClaim->id(),
                m_NotPriced[k]->m_pClaim->multiplier(),
                m_NotPriced[k]->m_pClaim->birth(),
                m_NotPriced[k]->m_pClaim->maturity() );
        }
    }

    if( m_Priced.numOfElems() > 0 ) {
        for( int k = 0; k < m_Priced.numOfElems(); ++k ) {
            MTG_TRACE( "    #%02d: id %d, pos %lg, life %d-%d, price\n", k + 1,
                m_Priced[k]->m_pClaim->id(),
                m_Priced[k]->m_pClaim->multiplier(),
                m_Priced[k]->m_pClaim->birth(),
                m_Priced[k]->m_pClaim->maturity() );
        }
    }

    if( m_aAuxByMaturity.numOfElems() > 0 ) {
        for( int k = 0; k < m_aAuxByMaturity.numOfElems(); ++k ) {
            MTG_TRACE( "    #%02d: id %d, pos %lg, life %d-%d, aux slot %d\n",
                k + 1,
                m_aAuxByMaturity[k]->m_pClaim->id(),
                m_aAuxByMaturity[k]->m_pClaim->multiplier(),
                m_aAuxByMaturity[k]->m_pClaim->birth(),
                m_aAuxByMaturity[k]->m_pClaim->maturity(),
                m_aAuxByMaturity[k]->m_nSlot );
        }
    }

    MTG_TRACE( "    Factors:" );
    for( int k = 0; k < m_Factor.numOfElems(); ++k ) 
        MTG_TRACE( " %s", m_Factor[k]->name() );
    MTG_TRACE( "\n" );

    MTG_TRACE( "}\n" );*/

#endif

    for( int k = 0; k < m_Sorted.numOfElems(); ++k ) {
        if( k == 0 ) {
            m_Settlement = m_Sorted[k]->m_pClaim->settlement();
        }
        else {
            if( m_Sorted[k]->m_pClaim->settlement() != m_Settlement )
                return SETTLEMENT_CONFLICT;
        }
    }

    return super::finalize();
}


//
//   g e t E v e n t s
//

void tPortfolio::getEvents( tSeqEvGenerator& EvGen ) const

{
    for( int k = 0; k < m_aWrapper.numOfElems(); ++k )
        m_aWrapper[k].m_pClaim->getEvents( EvGen );
    EvGen.cap( m_nMaturity );
}


//
//   c l a i m
//

tClaim& tPortfolio::claim( int nPos ) const

{
    MTG_ASSERT( isFinalized() );
    return *m_Sorted[nPos]->m_pClaim;
}


//
//   n o t P r i c e d C l a i m
//

tClaim& tPortfolio::notPricedClaim( int nPos ) const

{
    MTG_ASSERT( isFinalized() );
    return *m_NotPriced[nPos]->m_pClaim;
}


//
//   p r i c e d C l a i m
//

tClaim& tPortfolio::pricedClaim( int nPos ) const

{
    MTG_ASSERT( isFinalized() );
    return *m_Priced[nPos]->m_pClaim;
}


//
//   g e t B a r r i e r s
//

void tPortfolio::getBarriers( const tFactor* pFactor, bool bMain,
    tHeap<double>& Barrier ) const

{
    MTG_ASSERT( isFinalized() );

    Barrier.reset();
    for( int i = 0; i < m_aWrapper.numOfElems(); ++i )
        m_aWrapper[i].m_pClaim->getBarriers( pFactor, bMain, Barrier );
}


//
//   g e t M u l t i p l i e r s
//

void tPortfolio::getMultipliers( tHeap<double>& Multiplier ) const

{
    MTG_ASSERT( isFinalized() );

    Multiplier.numOfElems( m_Sorted.numOfElems() ) ;

    for( int k = 0; k < m_Sorted.numOfElems(); ++k )  {
        Multiplier[m_Sorted[k]->m_nIndex] =
            m_Sorted[k]->m_pClaim->multiplier();            
    }
}


//
//   g e t P r i c e s
//

void tPortfolio::getPrices( tHeap<double>& Price ) const

{
    MTG_ASSERT( isFinalized() );

    Price.numOfElems( m_Sorted.numOfElems() ) ;

    for( int k = 0; k < m_Priced.numOfElems(); ++k )  {
        Price[m_Priced[k]->m_nIndex] =
            ( m_Priced[k]->m_pClaim->unitBidPrice() +
              m_Priced[k]->m_pClaim->unitAskPrice() ) / 2;
    }
}


//
//   c o p y P r i c e d A b s 2 I n d e x
//

void tPortfolio::copyPricedAbs2Index( const tHeap<double>& Source,
    tHeap<double>& Target ) const

{
    Target.numOfElems( m_Sorted.numOfElems() );
    for( int k = 0; k < m_Priced.numOfElems(); ++k )
        Target[m_Priced[k]->m_nIndex] = Source[k];
}


//
//   c o p y P r i c e d I n d e x 2 A b s
//

void tPortfolio::copyPricedIndex2Abs( const tHeap<double>& Source,
    tHeap<double>& Target ) const

{
    Target.numOfElems( m_Priced.numOfElems() );
    for( int k = 0; k < m_Priced.numOfElems(); ++k )
        Target[k] = Source[m_Priced[k]->m_nIndex];
}


//
//   m a t c h F a c t o r s
//

tRetCode tPortfolio::matchFactors( const tModel& Model,
    tHeap<int>& Xlat ) const

{
    return Model.matchFactors( m_Factor, Xlat );
}

//
//   m a t c h F a c t o r s
//

tRetCode tPortfolio::matchFactors( const tModel* pModel,
    tHeap<int>& Xlat ) const

{
    MTG_ASSERT( pModel != 0 );
    return matchFactors( *pModel, Xlat );
}


//
//   a d d P a y m e n t S t r e a m
//

bool tPortfolio::addPaymentStream( tInterestSpline& Spline ) const

{
    MTG_ASSERT( isFinalized() );

    bool bOk = false;

    for( int i = 0; i < m_Priced.numOfElems(); ++i ) { 
        if( m_Priced[i]->m_pClaim->addPaymentStream( Spline ) )
            bOk = true;
    }

    return bOk;
}


//
//   l o a d M u l t i p l i e r s
//

tRetCode tPortfolio::loadMultipliers( tDataScanner& Scanner,
    const tHeap<double>& Multiplier ) const

{
    MTG_ASSERT( isFinalized() );

    tRetCode nRet;
    char *sName, *sComment;
    int nId; 
    tDate Maturity;
    double gMultiplier;

    while( ! Scanner.atEOF() ) {
        if( ( nRet = Scanner.scanDouble( gMultiplier ) ) != OK ||
            ( nRet = Scanner.scanDate( Maturity ) ) != OK ||
            ( nRet = Scanner.scanString( sName ) ) != OK ) {
            return nRet;
        }
        if( ( nRet = Scanner.scanString( sComment ) ) != OK ) {
            delete sName;
            return nRet;
        }

            // accept multiplier, if name is known and multiplier
            // is priced

        if( m_ClaimLkup.retrieve( sName, nId ) == OK ) {
            tClaim& Claim = *m_aId2Wrapper[nId]->m_pClaim;

            if( Claim.isPriced() )
                Multiplier[m_aId2Wrapper[nId]->m_nIndex] = gMultiplier;
        }

        delete sName;
        delete sComment;
    }

    return OK;
}

    
//
//   l o a d M u l t i p l i e r s
//
    
tRetCode tPortfolio::loadMultipliers( const tFileName& FName,
    const tHeap<double>& Multiplier ) const

{
    tRetCode nRet;
    tDataScanner Scanner;

    if( ( nRet = Scanner.setSource( FName ) ) != OK )
        return nRet;
    return loadMultipliers( Scanner, Multiplier );
}


//
//   s a v e M u l t i p l i e r s
//

tRetCode tPortfolio::saveMultipliers( ostream& Out,
    const tHeap<double>& Multiplier ) const

{
    MTG_ASSERT( isFinalized() );

    Out << setprecision( DBL_DIG );

    for( int k = m_Sorted.numOfElems() - 1; k >= 0; --k ) { 
        tClaim& Claim = *m_Sorted[k]->m_pClaim;

        if( ! Claim.isPriced() || Claim.isAnonymous() ) {
                // ignore instruments without name
        }
        else {
            double gMultiplier = Multiplier[Claim.index()];

            if( fabs( gMultiplier ) < 0.1 )
                Out << setiosflags( ios::scientific );
            else
                Out << setiosflags( ios::fixed );

            Out << setw( DBL_DIG + 10 )
                << gMultiplier << " "
                << "\"" << Claim.maturityDate() << "\" "
                << Claim.name() << " "
                << "\"" << Claim.comment() << "\"" << endl;
        }
    }

    return Out.good() ? OK : WRITE_ERROR;
}


//
//   s a v e M u l t i p l i e r s
//

tRetCode tPortfolio::saveMultipliers( const tFileName& FileName,
    const tHeap<double>& Multiplier ) const

{
    MTG_ASSERT( FileName.isValid() );

    tRetCode nRet;
    ofstream Out;

    if( ( nRet = FileName.openWrite( Out, "multipliers" ) ) != OK )
        return nRet;

    Out << "// value maturity name comment\n";
    if( ( nRet = saveMultipliers( Out, Multiplier ) ) != OK )
        return nRet;
    
    return OK;
}


//
//   p a r s e
//

tRetCode tPortfolio::parse( tParser& Parser, tSystem& System, tObject*& pObj )

{
    tRetCode nRet;
    tPortfolio* pPf;

    if( ( nRet = Parser.scanBeginOfObj() ) != OK )
        return nRet;

    pPf = new tPortfolio;
    pPf->setSystem( System );

    if( ( nRet = pPf->super::parse( Parser ) ) != OK ) {
        delete pPf;
        return nRet;
    }

    pObj = pPf;
    return OK;
}

MTG_END_NAMESPACE
