// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#if ! defined(_MTG_PORTFOLIO_)
#define _MTG_PORTFOLIO_

#include "MtgFileName.h"
#include "MtgHeap.h"
#include "MtgLookup.h"
#include "MtgObject.h"

MTG_BEGIN_NAMESPACE

// A portfolio is a collection of basic instrument plus dependency
// relationships. Dependencies can be defined between auxilaries only.
// Dependencies determine that aux A has to be evaluated before
// aux B, because the value of aux B depends on the value of aux A.

class tClaim;
class tDataScanner;
class tFactor;
class tInterestSpline;
class tModel;
class tParser;
class tSeqEvGenerator;
    

//
//   t P o r t f o l i o
//
    
class tPortfolio : public tObject {

    typedef tObject super;

    class tAux;

    struct tWrapper {
        tClaim *m_pClaim;
        int m_nIndex;
        int m_nSlot;
    };

        // Dependencies are stored as pair of claim id's.
        // m_nHiLevId references the high level instrument that has
        // to be evaluated after the lower level instrument m_nDepOnId,
        // on which it dependes, has been evaluated.

    struct tDep {
        int m_nHiLevId;
        int m_nDepOnId;
    };

    tHeap<tWrapper> m_aWrapper;
    tHeap<tDep> m_aDep;

        // Access by id:

    tHeap<tWrapper*> m_aId2Wrapper;

        // Access of auxiliaries:

    tHeap<tWrapper*> m_aAuxByBirth;
    tHeap<tWrapper*> m_aAuxByMaturity;

        // Access sorted by maturity, descending:

    tHeap<tWrapper*> m_Priced;      // only priced instruments
    tHeap<tWrapper*> m_NotPriced;   // only non-priced instruments
    tHeap<tWrapper*> m_Sorted;      // all instruments

    int m_nNumOfAuxSlots;
    int m_nNumOfNonAuxSlots;

    tDate m_MaturityDate;
    int m_nMaturity;

        // The settlement date is the date on which the portfolio
        // starts. For all priced instruments, the settlement date
        // MUST be the same.

    tDate m_Settlement;

        // During parsing, these lookup tables are used.

    tLookup<int> m_ClaimLkup;
    tLookup<int> m_FactorLkup;

        // After parsing, the pointers to all factors
        // used in the instruments are stored here.
        // The factors' reference counters are not affected,
        // since this is done by the instruments!!

    tHeap<tFactor*> m_Factor;

        // After parsing, the multipliers can be adjusted
        // by reading in a multiplier file.

    tFileName m_MultiplierFile;

    void init();
    void cleanup();

    void copyFrom( const tPortfolio& Pf );

        // Functions for hashing by instrument id.

    int hash( int nId ) const;
    tWrapper* find( int nId ) const;
    void buildId2Wrapper();

        // Functions used during finalization.

    void sort( tHeap<tWrapper*>& aHeap, int tClaim::* pField );
    void buildCategories();

    void demoteInAux( int nPos, int nSize, tHeap<tAux*>& aAux ) const;
    void promoteInAux( int nPos, tHeap<tAux*>& aAux ) const;
    tRetCode buildAuxHeap( tHeap<tAux*>& aAux ) const;

    tRetCode assignAuxSlots();
    tRetCode assignNonAuxSlots();

    tRetCode parseClaim( tParser& Parser, tClaim*& pClaim );
    tRetCode parseParam( tParser& Parser, tParseInfoStub& Info );
    tRetCode parsePostfix( tParser& Parser, tParseInfoStub& Info );

public:

    tPortfolio();
    tPortfolio( const tPortfolio& Pf );

    ~tPortfolio();

    tPortfolio& operator=( const tPortfolio& Pf );

    tObject* clone() const;

        // Functions for the creation of the portfolio:

    void regClaim( tClaim& Claim );
    void regClaim( tClaim* pClaim );

    void regDependency( int nHiLevId, int nDepOnId );

    tRetCode finalize();

        // The following functions require that finalize() has
        // been called and query attributes of the portfolio:

    int maturity() const {
        MTG_ASSERT( isFinalized() );
        return m_nMaturity;
    }

    tDate maturityDate() const {
        MTG_ASSERT( isFinalized() );
        return m_MaturityDate;
    }

    tDate settlement() const {
        MTG_ASSERT( isFinalized() );
        return m_Settlement;
    }

    int numOfClaims() const {
        MTG_ASSERT( isFinalized() );
        return m_Sorted.numOfElems();
    }

    int numOfNotPricedClaims() const {
        MTG_ASSERT( isFinalized() );
        return m_NotPriced.numOfElems();
    }

    int numOfPricedClaims() const {
        MTG_ASSERT( isFinalized() );
        return m_Priced.numOfElems();
    }

        // Access claims:

    tClaim& claim( int nPos ) const;
    tClaim& notPricedClaim( int nPos ) const;
    tClaim& pricedClaim( int nPos ) const;

    void getEvents( tSeqEvGenerator& EvGen ) const;

    void getBarriers( const tFactor* pFactor, bool bMain,
        tHeap<double>& Barrier ) const;

        // The arrays which are returned are indexed with the
        // instrument index (in the wrapper).

    void getMultipliers( tHeap<double>& Multiplier ) const;
    void getPrices( tHeap<double>& Price ) const;

        // Copies elements from Source to Traget, reindexing
        // based on instrument indexes in m_Priced.

    void copyPricedAbs2Index( const tHeap<double>& Source,
        tHeap<double>& Target ) const;

        // Copies elements from Source to Traget, reindexing
        // based on absolut instrument position in m_Priced.

    void copyPricedIndex2Abs( const tHeap<double>& Source,
        tHeap<double>& Target ) const;

        // The portfolio has a list of factors, and the model has
        // a list of factors. These have to be matched.

    tRetCode matchFactors( const tModel& Model, tHeap<int>& Xlat ) const;
    tRetCode matchFactors( const tModel* pModel, tHeap<int>& Xlat ) const;

        // This function examines all instruments in the portfolio
        // and puts those that are priced and simple into the
        // interest spline.

    bool addPaymentStream( tInterestSpline& Spline ) const;

        // Functions for optimization purposes.

        // Load multipliers for priced instruments from the
        // file and store them in the array (not in the
        // portfolio object itself!). Multipliers are indexed
        // via instrument index.

    tRetCode loadMultipliers( tDataScanner& Scanner,
        const tHeap<double>& Multiplier ) const;

    tRetCode loadMultipliers( const tFileName& FileName,
        const tHeap<double>& Multiplier ) const;

        // Print the instrument names of priced instruments
        // and substitute these multipliers, where the multipliers  
        // are indexed via instrument indexes:

    tRetCode saveMultipliers( ostream& Out,
        const tHeap<double>& Multiplier ) const;

    tRetCode saveMultipliers( const tFileName& FileName,
        const tHeap<double>& Multiplier ) const;

    static tRetCode parse( tParser& Parser, tSystem& System, tObject*& pObj );

    friend class tAux;
    friend class tFDEngine;
    friend class tLattice;
    friend class tSlotLayer;
};

MTG_END_NAMESPACE

#endif
