// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#include "MtgIncl.h"
#include "MtgProfile.h"

#if ! defined(_MTG_PROFILE_READONLY)

#include "MtgTimeAxis.h"
#include "MtgSpaceAxis.h"
#include "MtgSlotLayer.h"

#endif
								
MTG_BEGIN_NAMESPACE

#if ! defined(_MTG_PROFILE_READONLY)

//
//   w r i t e
//

tRetCode tProfile::tItem::write( FILE* fFile, const tArrayBounds& Bounds )

{
    if( fprintf( fFile, "%d", Bounds.dimension() ) < 0 )
        return WRITE_ERROR;

    for( int k = 0; k < Bounds.dimension(); ++k ) {
        if( fprintf( fFile, " %d %d", 
                Bounds.from( k ), Bounds.to( k ) ) < 0 ) {
            return WRITE_ERROR;
        }
    }

    return OK;
}

#endif

//
//   r e a d
//

tRetCode tProfile::tItem::read( FILE* fFile, tArrayBounds& Bounds )

{
    int nDim, nFrom, nTo;

    if( fscanf( fFile, "%d", &nDim ) != 1 )
        return READ_ERROR;

    Bounds.reset();
    for( int k = 0; k < nDim; ++k ) {
        if( fscanf( fFile, "%d%d", &nFrom, &nTo ) != 2 )
            return READ_ERROR;
        Bounds.append( nFrom, nTo );
    }

    return OK;
}


//
//   r e a d
//

tRetCode tProfile::tItem::read( FILE* fFile, char cExpected )

{
    MTG_ASSERT( ! isspace( cExpected ) );

    int c = getc( fFile );
    while( c != EOF && isspace( c ) )
        c = getc( fFile );
    if( c != cExpected )
        return FORMAT_ERROR;
    return OK;
}

#if ! defined(_MTG_PROFILE_READONLY)

//
//   t A l t e r n a t i v e I t e m
//

tProfile::tAlternativeItem::tAlternativeItem( int nComplexity, int nLayer )
    : tItem( 'A', 0 )

{
	m_nComplexity = nComplexity;
	m_nLayer = nLayer;
}


//
//   w r i t e
//

tRetCode tProfile::tAlternativeItem::write( FILE* fFile ) const

{
    if( fprintf( fFile, "%d %d", m_nComplexity, m_nLayer ) < 0 )
        return WRITE_ERROR;
    return OK;
}

#endif

//
//   r e a d
//

tRetCode tProfile::tAlternativeItem::read( FILE* fFile )

{
    if( fscanf( fFile, "%d%d", &m_nComplexity, &m_nLayer ) != 2 )
        return READ_ERROR;
    return OK;
}

#if ! defined(_MTG_PROFILE_READONLY)

//
//   w r i t e
//

tRetCode tProfile::tEngineItem::write( FILE* fFile ) const

{
    if( fprintf( fFile, "%d", m_nId ) < 0 )
        return WRITE_ERROR;
    return OK;
}

#endif

//
//   r e a d
//

tRetCode tProfile::tEngineItem::read( FILE* fFile )

{
    if( fscanf( fFile, "%d", &m_nId ) != 1 )
        return READ_ERROR;
    return OK;
}

#if ! defined(_MTG_PROFILE_READONLY)

//
//   t E x P o l i c y I t e m
//

tProfile::tExPolicyItem::tExPolicyItem( int nIndex, tExPolicy nExPolicy,
    double gPayoff ) : tItem( 'E', 0 )

{
    m_nIndex = nIndex;
    m_nExPolicy = nExPolicy;
    m_gPayoff = gPayoff;
}


//
//   w r i t e
//

tRetCode tProfile::tExPolicyItem::write( FILE* fFile ) const

{
    if( fprintf( fFile, "%d %d %g", m_nIndex, m_nExPolicy, m_gPayoff ) < 0 )
        return WRITE_ERROR;
    return OK;
}

#endif

//
//   r e a d
//

tRetCode tProfile::tExPolicyItem::read( FILE* fFile )

{
    if( fscanf( fFile, "%d%d%lg", &m_nIndex, 
            (int*) &m_nExPolicy, &m_gPayoff ) != 3 ) {
        return READ_ERROR;
    }
    return OK;
}

#if ! defined(_MTG_PROFILE_READONLY)

//
//   t L a t t i c e I t e m
//

tProfile::tLatticeItem::tLatticeItem( int nId, const tArrayBounds& Bounds,
    const tTimeAxis& Time, const tHeap<tSpaceAxis*>& Space )
    : tItem( 'L', nId )
    
{
    m_nNumOfSlices = Time.numOfSlices();
    m_Bounds = Bounds;
    m_pTime = &Time;
    m_pSpace = &Space;
}


//
//   w r i t e
//

tRetCode tProfile::tLatticeItem::write( FILE* fFile ) const

{
    MTG_ASSERT( m_pTime != 0 && m_pSpace != 0 );

    tRetCode nRet;

    if( fprintf( fFile, "%d %d ", m_nId, m_nNumOfSlices ) < 0 )
        return WRITE_ERROR;

    if( ( nRet = tItem::write( fFile, m_Bounds ) ) != OK )
        return nRet;

        // Now write time axis:

    if( fputs( "\nt", fFile ) == EOF )
        return WRITE_ERROR;

    for( int j = 0; j < m_nNumOfSlices; ++j ) {
        if( fprintf( fFile, " %d %g", m_pTime->day( j ),
                m_pTime->fractionOfDay( j ) ) < 0 ) {
            return WRITE_ERROR;
        }
    }

        // Now write space axes; rely on m_Bounds for extent.

    for( int k = 0; k < m_Bounds.dimension(); ++k ) {
        if( fputs( "\nx", fFile ) == EOF )
            return WRITE_ERROR;
        for( int j = m_Bounds.from( k ); j <= m_Bounds.to( k ); ++j ) {
            if( fprintf( fFile, " %g", (*m_pSpace)[k]->factor( j ) ) < 0 )
                return WRITE_ERROR;
        }
    }

    return OK;
}

#endif

//
//   r e a d
//

tRetCode tProfile::tLatticeItem::read( FILE* fFile )

{
    tRetCode nRet;

    if( fscanf( fFile, "%d%d", &m_nId, &m_nNumOfSlices ) != 2 )
        return READ_ERROR;

    if( ( nRet = tItem::read( fFile, m_Bounds ) ) != OK ||
        ( nRet = tItem::read( fFile, 't' ) ) != OK ) {
        return nRet;
    }

    m_AxesConcat.numOfElems( m_nNumOfSlices );
    for( int j = 0; j < m_nNumOfSlices; ++j ) {
        int nDay;
        double gFractionOfDay;

        if( fscanf( fFile, "%d%lg", &nDay, &gFractionOfDay ) != 2 ){
            m_AxesConcat.reset();
            return READ_ERROR;
        }
        m_AxesConcat[j] = nDay + gFractionOfDay;
    }

    for( int k = 0; k < m_Bounds.dimension(); ++k ) {
        if( ( nRet = tItem::read( fFile, 'x' ) ) != OK ) {
            m_AxesConcat.reset();
            return nRet;
        }

        double gFactor;
        for( int j = m_Bounds.from( k ); j <= m_Bounds.to( k ); ++j ) {
            if( fscanf( fFile, "%lg", &gFactor ) != 1 ) {
                m_AxesConcat.reset();
                return READ_ERROR;
            }
            m_AxesConcat.append( gFactor );
        }
    }

    return OK;
}

#if ! defined(_MTG_PROFILE_READONLY)

//
//   t L e v e l I t e m
//

tProfile::tLevelItem::tLevelItem( int nLevel )
    : tItem( 'P', 0 )

{
    m_Level.numOfElems( 1 );
    m_Level[0] = nLevel;
}


//
//   t L e v e l I t e m
//

tProfile::tLevelItem::tLevelItem( const tHeap<int>& Level )
    : tItem( 'P', 0 )

{
    m_Level = Level;
}


//
//   w r i t e
//

tRetCode tProfile::tLevelItem::write( FILE* fFile ) const

{
    if( fprintf( fFile, "%d", m_Level.numOfElems() ) < 0 )
        return WRITE_ERROR;
    for( int k = 0; k < m_Level.numOfElems(); ++k ) {
        if( fprintf( fFile, " %d", m_Level[k] ) < 0 )
            return WRITE_ERROR;
    }
    return OK;
}

#endif

//
//   r e a d
//

tRetCode tProfile::tLevelItem::read( FILE* fFile )

{
    int nNumOfElems;

    if( fscanf( fFile, "%d", &nNumOfElems ) != 1 )
        return READ_ERROR;
    m_Level.numOfElems( nNumOfElems );
    for( int k = 0; k < m_Level.numOfElems(); ++k ) {
        if( fscanf( fFile, "%d", &m_Level[k] ) != 1 )
            return READ_ERROR;
    }
    return OK;
}

#if ! defined(_MTG_PROFILE_READONLY)

//
//   t S l o t L a y e r I t e m
//

tProfile::tSlotLayerItem::tSlotLayerItem( int nId,
    const tArrayBounds& Bounds, const tSignature& Sig ) : tItem( 'S', nId )
    
{
    m_Bounds = Bounds;
    m_Sig = Sig;
}


//
//   w r i t e
//

tRetCode tProfile::tSlotLayerItem::write( FILE* fFile ) const

{
    tRetCode nRet;

    if( fprintf( fFile, "%d ", m_nId ) < 0 )
        return WRITE_ERROR;

    if( ( nRet = tItem::write( fFile, m_Bounds ) ) != OK )
        return nRet;

    if( fprintf( fFile, " %d %d ", m_Sig.tag(), m_Sig.numOfFlags() ) < 0 )
        return WRITE_ERROR;

    for( int k = 0; k < m_Sig.numOfFlags(); ++k ) {
        if( fprintf( fFile, "%c", m_Sig[k] ? '1' : '0' ) < 0 )
            return WRITE_ERROR;
    }

    return OK;
}

#endif

//
//   r e a d
//

tRetCode tProfile::tSlotLayerItem::read( FILE* fFile )

{
    tRetCode nRet;
    int nTag, nFlags;

    if( fscanf( fFile, "%d", &m_nId ) != 1 )
        return READ_ERROR;

    if( ( nRet = tItem::read( fFile, m_Bounds ) ) != OK )
        return OK;

    if( fscanf( fFile, "%d%d", &nTag, &nFlags ) != 2 )
        return READ_ERROR;

    m_Sig.reset( nFlags );
    m_Sig.setTag( nTag );
    for( int k = 0; k < nFlags; ++k ) {
        int cOn = getc( fFile );
        while( cOn != EOF && isspace( cOn ) )
            cOn = getc( fFile );
        if( cOn != '1' && cOn != '0' )
            return READ_ERROR;
        if( cOn == '1' )
            m_Sig.on( k );
    }

    return OK;
}

#if ! defined(_MTG_PROFILE_READONLY)

//
//   t T a s k I t e m
//

tProfile::tTaskItem::tTaskItem( int nLayer, int nSlice,
    int nDay, double gFractionOfDay ) : tItem( 'T', 0 )

{
    m_nLayer = nLayer;
    m_nSlice = nSlice;
    m_nDay = nDay;
    m_gFractionOfDay = gFractionOfDay;
}


//
//   w r i t e
//

tRetCode tProfile::tTaskItem::write( FILE* fFile ) const

{
    if( fprintf( fFile, "%d %d %d %g",
            m_nLayer, m_nSlice, m_nDay, m_gFractionOfDay ) < 0 ) {
        return WRITE_ERROR;
    }

    return OK;
}

#endif

//
//   r e a d
//

tRetCode tProfile::tTaskItem::read( FILE* fFile )

{
    if( fscanf( fFile, "%d%d%d%lg", 
            &m_nLayer, &m_nSlice, &m_nDay, &m_gFractionOfDay ) != 4 ) {
        return READ_ERROR;
    }

    return OK;
}

#if ! defined(_MTG_PROFILE_READONLY)

//
//   t V a l u e I t e m
//

tProfile::tValueItem::tValueItem( tType nType, double gValue )
	: tItem( 'V', 0 )

{
	m_nType = nType;
	m_gValue = gValue;
}


//
//   w r i t e
//

tRetCode tProfile::tValueItem::write( FILE* fFile ) const

{
    if( fprintf( fFile, "%d %g", m_nType, m_gValue ) < 0 )
        return WRITE_ERROR;
    return OK;
}

#endif

//
//   r e a d
//

tRetCode tProfile::tValueItem::read( FILE* fFile )

{
    if( fscanf( fFile, "%d%lg", (int*) &m_nType, &m_gValue ) != 2 )
        return READ_ERROR;
    return OK;
}


//
//   i n i t
//

void tProfile::init()

{
#if ! defined(_MTG_PROFILE_READONLY)
    m_nNextGeneralId = 1;
    m_nNextLayerId = 1;
    m_bActive = false;
    m_bExhaustive = false;
#endif

    m_sFileName = 0;
    m_fFile = 0;
}


//
//   s k i p I t e m
//

tRetCode tProfile::skipItem()

{
    int c;

    while( ( c = getc( m_fFile ) ) != EOF && c != '\n' );
    if( c == EOF )
        return END_OF_FILE;
    return OK;
}


//
//   g e t T y p e
//

tRetCode tProfile::getType( char& cType )

{
    int c = getc( m_fFile );
    while( c != EOF && isspace( c ) )
        c = getc( m_fFile );

    if( c == EOF )
        return END_OF_FILE;

    cType = (char) c;
    return OK;
}


//
//   g e t I t e m
//

tRetCode tProfile::getItem( tItem*& pItem, char cType )

{
    tRetCode nRet;
    
    pItem = 0;
    switch( cType ) {
        case 'A' : pItem = new tAlternativeItem; break;
        case 'G' : pItem = new tEngineItem;      break;
        case 'E' : pItem = new tExPolicyItem;    break;
        case 'L' : pItem = new tLatticeItem;     break;
        case 'P' : pItem = new tLevelItem;       break;
        case 'R' : pItem = new tRepeatItem;      break;
        case 'S' : pItem = new tSlotLayerItem;   break;
        case 'T' : pItem = new tTaskItem;        break;
        case 'V' : pItem = new tValueItem;       break;

        default :
            return FORMAT_ERROR;
    }

    pItem->m_cType = cType;
    if( ( nRet = pItem->read( m_fFile ) ) != OK ) {
        delete pItem;
        pItem = 0;
        if( nRet == END_OF_FILE )
            return FORMAT_ERROR;
        return nRet;
    }

    return OK;
}


//
//   g e t
//

tRetCode tProfile::get( tItem*& pItem )

{
    MTG_ASSERT( m_fFile != 0 );

    tRetCode nRet;
    char cType;
    
    if( ( nRet = getType( cType ) ) != OK )
        return nRet;
    return getItem( pItem, cType );
}


//
//   g e t
//

tRetCode tProfile::get( tItem*& pItem, char cType )

{
	tRetCode nRet;

	while( ( nRet = get( pItem ) ) == OK ) {
		if( pItem->m_cType == cType )
			return OK;
        else
            delete pItem;
	}

	return nRet;
}


//
//   g e t
//

tRetCode tProfile::get( tItem*& pItem, char cType1, char cType2 )

{
	tRetCode nRet;

	while( ( nRet = get( pItem ) ) == OK ) {
		if( pItem->m_cType == cType1 || pItem->m_cType == cType2 )
			return OK;
        else
            delete pItem;
	}

	return nRet;
}


//
//   g e t
//

tRetCode tProfile::get( tItem*& pItem, const char* sTypeSet )

{
    MTG_ASSERT( m_fFile != 0 );

    tRetCode nRet;
    char cType;

	while( ( nRet = getType( cType ) ) == OK ) {
        if( strchr( sTypeSet, cType ) != 0 )
			return getItem( pItem, cType );
        if( ( nRet = skipItem() ) != OK )
            return nRet;
	}

	return nRet;
}


//
//   t P r o f i l e
//

tProfile::tProfile()

{
    init();
}


//
//   t P r o f i l e
//

tProfile::tProfile( const char* sFileName )

{
    init();
    setFileName( sFileName );
}


//
//   ~ t P r o f i l e
//

tProfile::~tProfile()

{
    close();
    if( m_sFileName != 0 )
        delete m_sFileName;
}


//
//   s e t F i l e N a m e
//

void tProfile::setFileName( const char* sFileName )

{
    close();
    if( m_sFileName != 0 )
        delete m_sFileName;
    if( sFileName != 0 )
        m_sFileName = StrCopy( sFileName );
}


//
//   u s e F i l e N a m e
//

void tProfile::useFileName( const char* sFileName )

{
    int k = strlen( sFileName );
    char* s = new char[k + 6];

    strcpy( s, sFileName );
    while( k > 0 && s[k - 1] != '.' )
        --k;
    if( k > 0 )
        --k;
    else
        k = strlen( sFileName );
    strcpy( &s[k], ".prof" );
    setFileName( s );
    delete s;
}


//
//   o p e n R e a d
//

tRetCode tProfile::openRead()

{
    close();

    if( m_sFileName == 0 )
        return OPEN_ERROR;

    if( ( m_fFile = fopen( m_sFileName, "rt" ) ) == 0 )
        return OPEN_ERROR;

    return OK;
}


//
//   o p e n R e a d
//

tRetCode tProfile::openRead( const char* sFileName )

{
    setFileName( sFileName );
    return openRead();
}


//
//   r e w i n d
//

void tProfile::rewind()

{
    if( m_fFile != 0 )
        fseek( m_fFile, 0, SEEK_SET );
}


//
//   c l o s e
//

void tProfile::close()

{
    if( m_fFile != 0 ) {
        fclose( m_fFile );
        m_fFile = 0;
    }
}

#if ! defined(_MTG_PROFILE_READONLY)

//
//   u s e A r g s
//

tRetCode tProfile::useArgs( int& argc, const char* argv[] )

{    
    for( int k = 1; k < argc; ++k ) {
        if( strcmp( argv[k], "-exhaustive" ) == 0 ) {
            m_bExhaustive = true;
            argv[k] = 0;
        }
        else
        if( strcmp( argv[k], "-profile" ) == 0 ) {
            m_bActive = true;
            argv[k] = 0;
        }
    }

    if( m_bExhaustive && ! m_bActive )
        m_bActive = true;

        // Remove used arguments.

    int j = 1;
    for( MTG_FOR_INIT( int ) k = 1; k < argc; ++k ) {
        if( argv[k] != 0 )
            argv[j++] = argv[k];
    }
    argc = j;

    return OK;
}


//
//   a c t i v a t e
//

void tProfile::activate( bool bExhaustive )

{
    m_bActive = true;
    m_bExhaustive = bExhaustive;
}


//
//   o p e n W r i t e
//

tRetCode tProfile::openWrite()

{
    close();

    if( m_sFileName == 0 )
        return OPEN_ERROR;

    if( ( m_fFile = fopen( m_sFileName, "wt" ) ) == 0 )
        return OPEN_ERROR;

    m_nNextGeneralId = 1;
    m_nNextLayerId = 1;
    return OK;
}


//
//   o p e n W r i t e
//

tRetCode tProfile::openWrite( const char* sFileName )

{
    setFileName( sFileName );
    return openWrite();
}


//
//   p u t
//

void tProfile::put( const tItem& Item )

{
    MTG_ASSERT( m_fFile != 0 );

    tRetCode nRet;

    if( putc( Item.m_cType, m_fFile ) == EOF ||
        putc( ' ', m_fFile ) == EOF ) {
        throw tException( WRITE_ERROR );
    }
    if( ( nRet = Item.write( m_fFile ) ) != OK )
        throw tException( nRet );
    if( putc( '\n', m_fFile ) == EOF )
        throw tException( WRITE_ERROR );
}


//
//   p u t A l t e r n a t i v e
//

void tProfile::putAlternative( int nComplexity )

{
    tAlternativeItem A( nComplexity, -2 );
    put( A );
}


//
//   p u t A l t e r n a t i v e
//

void tProfile::putAlternative( int nComplexity, const tSlotLayer* pLayer )

{
    tAlternativeItem A( nComplexity, pLayer ? pLayer->m_nId : -1 );
    put( A );
}


//
//   p u t E n g i n e
//

int tProfile::putEngine()

{
    int nId = nextGeneralId();
    tEngineItem E( nId );
    put( E );
    return nId;
}


//
//   p u t E x P o l i c y
//

void tProfile::putExPolicy( int nIndex, tExPolicy nExPolicy, double gPayoff )

{
    tExPolicyItem E( nIndex, nExPolicy, gPayoff );
    put( E );
}


//
//   p u t L a t t i c e
//

int tProfile::putLattice( const tArrayBounds& Bounds, const tTimeAxis& Time,
    const tHeap<tSpaceAxis*>& Space )

{
    int nId = nextGeneralId();
    tLatticeItem L( nId, Bounds, Time, Space );
    put( L );
    return nId;
}


//
//   p u t L e v e l
//

void tProfile::putLevel( int nLevel )

{
    tLevelItem PP( nLevel );
    put( PP );
}


//
//   p u t L e v e l
//

void tProfile::putLevel( const tHeap<int>& Level )

{
    tLevelItem PP( Level );
    put( PP );
}


//
//   p u t R e p e a t
//

void tProfile::putRepeat()

{
    tRepeatItem R;
    put( R );
}


//
//   p u t S l o t L a y e r
//

int tProfile::putSlotLayer( const tArrayBounds& Bounds, const tSignature& Sig )

{
    int nId = nextLayerId();
    tSlotLayerItem S( nId, Bounds, Sig );
    put( S );
    return nId;
}


//
//   p u t T a s k
//

void tProfile::putTask( const tSlotLayer& Layer, int nSlice, int nDay,
    double gFractionOfDay )

{
    tTaskItem T( Layer.m_nId, nSlice, nDay, gFractionOfDay );
    put( T );
}


//
//   p u t V a l u e
//

void tProfile::putValue( tProfile::tValueItem::tType nType, double gValue )

{
	tValueItem V( nType, gValue );
	put( V );
}

#endif

MTG_END_NAMESPACE
