// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#include "MtgIncl.h"
#include "MtgService.h"

MTG_BEGIN_NAMESPACE

#if defined(_WIN32)
    #include <process.h>
    #include <direct.h>
#endif


//
//   a p p e n d T o P a t h
//

bool tService::appendToPath( char* sPath, size_t nLength,
    int argc, const char* argv[] )

{
    size_t l = strlen( sPath );

    for( int i = 1; i < argc; ++i ) {
        size_t h = strlen( argv[i] ) + 1;
        if( h + l >= nLength )
            return false;
        sprintf( &sPath[l], " %s", argv[i] );
        l += h;
    }

    return true;
}


//
//   m a k e P a t h
//

bool tService::makePath( char* sPath, size_t nLength,
    int argc, const char* argv[], const char* sArg0 )

{
    size_t l = sArg0 ? strlen( sArg0 ) : 0;
    if( l > 0 )
        ++l;

    if( nLength <= l )
        return false;

#if defined(_WIN32)
    if( GetModuleFileName( NULL, sPath, nLength - l ) == 0 )
        return false;
#else
    if( strlen( argv[0] ) >= nLength - l )
        return false;
    strcpy( sPath, argv[0] );
#endif

    if( l > 0 ) {
        strcat( sPath, " " );
        strcat( sPath, sArg0 );
    }

    return appendToPath( sPath, nLength, argc, argv );
}

#if defined(_WIN32)

//
//   v a r i a b l e s
//

tService* tService::m_pThis = 0;


//
//   o p e n S e r v i c e
//

tRetCode tService::openService( SC_HANDLE& hManager, SC_HANDLE& hService )

{
    if( ( hManager = OpenSCManager( NULL, NULL, 
            SC_MANAGER_ALL_ACCESS ) ) == NULL ) {
        return SERVICE_ERROR;
    }

    if( ( hService = OpenService( hManager,
            m_sName, SERVICE_ALL_ACCESS ) ) == NULL ) {
        CloseServiceHandle( hManager );
        return SERVICE_ERROR;
    }

    return OK;
}


//
//   c l o s e S e r v i c e
//

void tService::closeService( SC_HANDLE hManager, SC_HANDLE hService )

{
    CloseServiceHandle( hManager );
    CloseServiceHandle( hService );
}


//
//   s e r v i c e S t a r t H a n d l e r  
//

void _stdcall tService::serviceStartHandler( DWORD argc, LPTSTR *argv )

{
    MTG_ASSERT( m_pThis != 0 );
    m_pThis->serviceStart( argc, argv );
}


//
//   s e r v i c e C t r l H a n d l e r
//

void _stdcall tService::serviceCtrlHandler( DWORD nOpcode )

{
    MTG_ASSERT( m_pThis != 0 );
    m_pThis->serviceCtrl( nOpcode );
}


//
//   s e r v i c e S t o p T h r e a d
//

void tService::serviceStopThread( void* pThis )

{
    MTG_ASSERT( pThis != 0 );
    static_cast<tService*>( pThis )->serviceStop();
}


//
//   s e r v i c e R u n T h r e a d
//

void tService::serviceRunThread( void* pThis )

{
    MTG_ASSERT( pThis != 0 );
    static_cast<tService*>( pThis )->serviceRun();
}


//
//   s e r v i c e C h i l d T h r e a d
//

void tService::serviceChildThread( void* pThis )

{
    MTG_ASSERT( pThis != 0 );
    static_cast<tService*>( pThis )->serviceChild();
}


//
//   s e r v i c e S t a r t
//

void tService::serviceStart( DWORD argc, LPTSTR *argv )

{
    MTG_ASSERT( m_main != 0 );

        // The arguments in argv should be appended to 
        // the arguments in m_argv. They are explicitely
        // passed in from StartService.

    SERVICE_STATUS& Status = m_Status;

    Status.dwServiceType = SERVICE_WIN32 | SERVICE_INTERACTIVE_PROCESS; 
    Status.dwCurrentState = SERVICE_START_PENDING; 
    Status.dwControlsAccepted = SERVICE_ACCEPT_STOP;
    Status.dwWin32ExitCode = NO_ERROR; 
    Status.dwServiceSpecificExitCode = 0; 
    Status.dwCheckPoint = 0; 
    Status.dwWaitHint = 0;  

    if( ( m_hStatus = RegisterServiceCtrlHandler( argv[0],
            serviceCtrlHandler ) ) == NULL ) {
        m_nErrCode = 1;
        return;
    }

    if( ! SetServiceStatus( m_hStatus, &Status ) ) {
        m_nErrCode = 1;
        return;
    }

    ChToProgramDir( 0 );

    SECURITY_ATTRIBUTES StopReqSec, StopAckSec;

    StopReqSec.nLength = sizeof(StopReqSec);
    StopReqSec.lpSecurityDescriptor = 0;
    StopReqSec.bInheritHandle = true;

    StopAckSec.nLength = sizeof(StopAckSec);
    StopAckSec.lpSecurityDescriptor = 0;
    StopAckSec.bInheritHandle = false;

    m_bHasChild = false;

    if( ( m_hStopReq =
            CreateEvent( &StopReqSec, true, false, NULL ) ) == NULL ||
        ( m_hStopAck =
            CreateEvent( &StopAckSec, true, false, NULL ) ) == NULL ||
        ! serviceInit() ) {
        Status.dwCurrentState = SERVICE_STOPPED; 
        Status.dwWin32ExitCode = ERROR_SERVICE_SPECIFIC_ERROR; 
        Status.dwServiceSpecificExitCode = 1;
        if( m_nErrCode == 0 )
            m_nErrCode = 1;
    }
    else {        
        Status.dwCurrentState = SERVICE_RUNNING; 
    }

    if( ! SetServiceStatus( m_hStatus, &Status ) ) {
        m_nErrCode = 1;
        return;
    }

    if( Status.dwCurrentState == SERVICE_RUNNING ) {
        _beginthread( serviceRunThread, 0, this );
        WaitForSingleObject( m_hStopAck, INFINITE );

        if( m_bHasChild ) {
                // Child has supposedly terminated by now.

            DWORD nExitCode;

            if( GetExitCodeProcess( m_Child.hProcess, &nExitCode ) ) {
                if( nExitCode == STILL_ACTIVE ) {
                        // This shouldn't happen...
                    TerminateProcess( m_Child.hProcess, 1 );
                    m_nErrCode = 1;
                }
                else {
                    m_nErrCode = (int) nExitCode;
                }
            }
            else {
                m_nErrCode = 1;
            }
        }
    }

    Status.dwCurrentState = SERVICE_STOPPED; 
    if( ! SetServiceStatus( m_hStatus, &Status ) ) {
        if( m_nErrCode == 0 )
            m_nErrCode = 1;
        return;
    }
}


//
//   s e r v i c e C t r l
//

void tService::serviceCtrl( DWORD nOpcode )

{
    switch( nOpcode ) {
        case SERVICE_CONTROL_STOP: 
            m_Status.dwCurrentState = SERVICE_STOP_PENDING;
            break;
 
        default: 
            break;
    } 
 
    if( SetServiceStatus( m_hStatus, &m_Status ) ) {
        if( m_Status.dwCurrentState == SERVICE_STOP_PENDING ) {
            _beginthread( serviceStopThread, 0, this );
        }
    }

    return; 
}


//
//   s e r v i c e I n i t
//

bool tService::serviceInit()

{
    char sPath[2048], sChild[256];
    STARTUPINFO SInfo;

    sprintf( sChild, "-child %ld", (DWORD) m_hStopReq );

    if( ! makePath( sPath, sizeof(sPath), m_argc, m_argv, sChild ) )
        return false;

    memset( &SInfo, 0, sizeof(SInfo) );
    SInfo.cb = sizeof(SInfo);

    if( ! CreateProcess( 0, sPath, 0, 0, true, CREATE_NEW_PROCESS_GROUP,
            0, 0, &SInfo, &m_Child ) ) {
        return false;
    }
    m_bHasChild = true;

    return true;
}


//
//   s e r v i c e R u n
//

void tService::serviceRun()

{
    if( m_bHasChild ) {
        if( WaitForSingleObject( m_Child.hProcess, INFINITE ) !=
                WAIT_OBJECT_0 ) {
            TerminateProcess( m_Child.hProcess, 1 );
        }
            // Signal main routine that child has finished.
        SetEvent( m_hStopAck );
    }
}


//
//   s e r v i c e S t o p
//

void tService::serviceStop()

{
    if( m_bHasChild ) {
                // Request child to terminate.
        SetEvent( m_hStopReq );
                // m_hStopAck is set by serviceRun()
                // after the child has finished.
        if( WaitForSingleObject( m_hStopAck, 60 * 1000 ) != WAIT_OBJECT_0 ) {
                // Timeout, either the child process
                // doesn't respond or is dead. Kill it.
            TerminateProcess( m_Child.hProcess, 1 );
                // We want to finish in all cases.
            Sleep( 60 * 1000 );
            SetEvent( m_hStopAck );
        }
    }
}


//
//   s e r v i c e C h i l d
//

void tService::serviceChild()

{
    if( WaitForSingleObject( m_hStopReq, INFINITE ) == WAIT_OBJECT_0 ) {
        ExitProcess( 0 );
    }
}

#endif


//
//   t S e r v i c e
//

tService::tService()

{
    m_sName = StrCopy( "MtgService" );
    m_nActivity = xUnknown;
    m_bIsInteractive = true;
    m_argc = 0;
    m_argv = 0;
    m_nErrCode = 0;
    m_main = 0;
}


//
//   ~ t S e r v i c e
//

tService::~tService()

{
    if( m_sName != 0 )
        delete m_sName;
}


//
//   s e t N a m e
//

void tService::setName( const char* sName )

{
    if( m_sName != 0 )
        delete m_sName;
    if( sName == 0 )
        m_sName = 0;
    else
        m_sName = StrCopy( sName );
}


//
//   i n s t a l l
//

tRetCode tService::install( int argc, const char* argv[] )

{
        // Assume we're interactive.

#if defined(_WIN32)

    SC_HANDLE hService;
    SC_HANDLE hManager;
    char sPath[2048];
    tRetCode nRet;

    if( ! makePath( sPath, sizeof(sPath), argc, argv, "-service" ) )
        return SERVICE_ERROR;

    if( ( hManager = OpenSCManager( NULL, NULL, 
            SC_MANAGER_ALL_ACCESS ) ) == NULL ) {
        return SERVICE_ERROR;
    }

    hService = CreateService(
        hManager,
        m_sName, 
        m_sName, 
        SERVICE_ALL_ACCESS,        
        SERVICE_WIN32_OWN_PROCESS, 
        SERVICE_AUTO_START,        
        SERVICE_ERROR_NORMAL,      
        sPath,                      
        NULL,                       
        NULL,                       
        NULL,                       
        NULL,                       
        NULL );                     

    if( hService != NULL ) {
        CloseServiceHandle( hService );
        nRet = OK;
    }
    else {
        nRet = SERVICE_ERROR;
    }

    CloseServiceHandle( hManager );
    return nRet;

#else
    return NOT_IMPLEMENTED;
#endif
}


//
//   i n s t a l l
//

tRetCode tService::install( const char* sName, int argc, const char* argv[] )

{
    setName( sName );
    return install( argc, argv );
}


//
//   r e m o v e
//

tRetCode tService::remove()

{
        // Assume we're interactive.

#if defined(_WIN32)

    SC_HANDLE hManager, hService;
    tRetCode nRet;

    stop();
    if( ( nRet = openService( hManager, hService ) ) != OK )
        return nRet;

    DeleteService( hService );
    closeService( hManager, hService );
    return OK;

#else
    return NOT_IMPLEMENTED;
#endif
}


//
//   s t a r t
//

tRetCode tService::start( int argc, const char* argv[] )

{
#if defined(_WIN32)

    SC_HANDLE hManager, hService;
    tRetCode nRet;

    if( ( nRet = openService( hManager, hService ) ) != OK )
        return nRet;

    if( ! StartService( hService, argc, argv ) ) {
        closeService( hManager, hService );
        return SERVICE_ERROR;
    }

    closeService( hManager, hService );
    return OK;

#else
    return NOT_IMPLEMENTED;
#endif
}


//
//   s t o p
//

tRetCode tService::stop()

{
#if defined(_WIN32)

    SC_HANDLE hManager, hService;
    SERVICE_STATUS Status;
    tRetCode nRet;

    if( ( nRet = openService( hManager, hService ) ) != OK )
        return nRet;

    if( ! ControlService( hService, SERVICE_CONTROL_STOP, &Status ) ) {
        closeService( hManager, hService );
        return SERVICE_ERROR;
    }

    int k = 0;
    int n = 10;

    while( k < n ) {
        DWORD nSleep = 1000;

        if( QueryServiceStatus( hService, &Status ) ) {
            if( Status.dwCurrentState == SERVICE_STOPPED )
                break;
            
            if( Status.dwWaitHint > nSleep )
                nSleep = Status.dwWaitHint;
        }

        Sleep( nSleep );
        ++k;
    }

    closeService( hManager, hService );
    return k == n ? SERVICE_ERROR : OK;

#else
    return NOT_IMPLEMENTED;
#endif
}


//
//   q u e r y
//

tRetCode tService::query()

{
#if defined(_WIN32)

    SC_HANDLE hManager, hService;
    SERVICE_STATUS Status;
    tRetCode nRet;

    if( ( nRet = openService( hManager, hService ) ) != OK ) {
        printf( "*** No service '%s'\n", m_sName );
        return nRet;
    }

    if( ! ControlService( hService, SERVICE_CONTROL_INTERROGATE, &Status ) ) {
        printf( "*** Service '%s' does not respond, probably stopped\n",
            m_sName );
        closeService( hManager, hService );
        return SERVICE_ERROR;
    }

    printf( "Service name.: %s\n", m_sName );
    printf( "Current state: " );

    switch( Status.dwCurrentState ) {
        case SERVICE_STOPPED :          printf( "stopped\n" );          break;
        case SERVICE_START_PENDING :    printf( "start pending\n" );    break;
        case SERVICE_STOP_PENDING :     printf( "stop pending\n" );     break;
        case SERVICE_RUNNING :          printf( "running\n" );          break;
        case SERVICE_PAUSE_PENDING :    printf( "pause pending\n" );    break;
        case SERVICE_CONTINUE_PENDING : printf( "continue pending\n" ); break;
        case SERVICE_PAUSED :           printf( "paused\n" );           break;
        default :                       printf( "unknown\n" );          break;
    }

    printf( "\n" );

    closeService( hManager, hService );
    return OK;

#else
    return NOT_IMPLEMENTED;
#endif
}


//
//   r u n
//

tRetCode tService::run( int (*main)( int argc, const char* argv[] ),
    int argc, const char* argv[], int& nErrCode )

{
        // Assume we're not interactive.
        
    m_main = main;
    m_argc = argc;
    m_argv = argv;
    m_nErrCode = 0;

#if defined(_WIN32)

    SERVICE_TABLE_ENTRY DispatchTable[] = {
        { m_sName, serviceStartHandler },
        { NULL, NULL }
    };

    m_pThis = this;
    if( ! StartServiceCtrlDispatcher( DispatchTable ) )
        return SERVICE_ERROR;

    nErrCode = m_nErrCode;
    return OK;

#else

    nErrCode = m_nErrCode = m_main( argc, argv );
    return OK;

#endif
}


//
//   u s e A r g s
//

tRetCode tService::useArgs( int& argc, const char* argv[] )

{
    static struct {
        const char* m_sArg;
        tActivity m_nActivity;
    } Arg[] = {
        { "remove", xRemove },
        { "install", xInstall },
        { "start", xStart },
        { "stop", xStop },
        { "query", xQuery },
        { "service", xRun },
        { "interactive", xSkip }
    };


    for( int k = 1; k < argc; ++k ) {
        if( argv[k][0] == '-' || argv[k][0] == '/' ) {
            if( strcmp( &argv[k][1], "child" ) == 0 ) {

#if defined(_WIN32)
                bool bOk = ( m_nActivity == xUnknown && k + 1 < argc );

                DWORD h;

                if( bOk )
                    bOk = ( sscanf( argv[++k], "%ld", &h ) > 0 );

                if( ! bOk ) {
                        // Assume we're interactive.
                    printf( "*** Illegal option \'%s\'\n\n", argv[k] );
                    return INVALID_KEYWORD;
                }

                m_hStopReq = (HANDLE) h;

                argv[k] = argv[k - 1] = 0;
                m_nActivity = xChild;
#else
                if( m_nActivity != xUnknown ) {
                        // Assume we're interactive.
                    printf( "*** Illegal option \'%s\'\n\n", argv[k] );
                    return INVALID_KEYWORD;
                }
                m_nActivity = xChild;
                argv[k] = 0;
#endif

            }
            else {
                for( size_t i = 0; i < sizeof(Arg) / sizeof(Arg[0]); ++i ) {
                    if( strcmp( &argv[k][1], Arg[i].m_sArg ) == 0 ) {
                        if( m_nActivity != xUnknown ) {
                                // Assume we're interactive.
                            printf( "*** Illegal option \'%s\'\n\n",
                                argv[k] );
                            return INVALID_KEYWORD;
                        }
                        m_nActivity = Arg[i].m_nActivity;
                        argv[k] = 0;
                        break;
                    }
                }
            }
        }
    }

        // Remove used arguments.

    int j = 1;
    for( MTG_FOR_INIT( int ) k = 1; k < argc; ++k ) {
        if( argv[k] != 0 )
            argv[j++] = argv[k];
    }
    argc = j;

    return OK;
}


//
//   p e r f o r m
//

tRetCode tService::perform( int (*main)( int argc, const char* argv[] ),
    int argc, const char* argv[], int& nErrCode )    

{
    if( m_nActivity == xUnknown ) {

#if defined(_WIN32)

        m_nActivity = xRun;             // first guess
        for( int i = 1; i < argc; ++i ) {
            if( argv[i][0] != '-' && argv[i][0] != '/' ) {
                m_nActivity = xSkip;    // second guess
                break;
            }
            else {
                if( strcmp( &argv[i][1], "help" ) == 0 ||
                    strcmp( &argv[i][1], "h" ) == 0 ||
                    strcmp( &argv[i][1], "?" ) == 0 ) {
                    m_nActivity = xSkip;    // third guess
                    break;
                }
            }
        }

        if( m_nActivity == xRun )
            m_nActivity = xSkip;

#else
        m_nActivity = xSkip;
#endif
    }

    tRetCode nRet;

    m_bIsInteractive = true;

    switch( m_nActivity ) {
        case xChild :
#if defined(_WIN32)
                // This thread communicates with the parent.
            _beginthread( serviceChildThread, 0, this );
#endif
            m_bIsInteractive = false;
            // fall through

        case xSkip :
            nRet = GO;
            break;

        case xRun :
            if( ( nRet = run( main, argc, argv, nErrCode ) ) == GO )
                m_bIsInteractive = false;
            break;

        case xInstall :
            nRet = install( argc, argv );
            break;

        case xRemove :
            nRet = remove();
            break;

        case xStart :
            nRet = start( argc, argv );
            break;

        case xStop :
            nRet = stop();
            break;

        case xQuery :
            nRet = query();
            break;

        default :
            throw tException( INTERNAL_ERROR );
    }

    return nRet;
}


//
//   p e r f o r m
//

tRetCode tService::perform( int (*main)( int argc, const char* argv[] ),
    int argc, const char* argv[], int& nErrCode, bool& bIsInteractive )

{
    tRetCode nRet = perform( main, argc, argv, nErrCode );
    bIsInteractive = isInteractive();
    return nRet;
}

MTG_END_NAMESPACE
