// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#include "MtgIncl.h"
#include "MtgShell.h"
#include "MtgBootstrap.h"
#include "MtgDataScanner.h"
#include "MtgEvaluate.h"
#include "MtgRepository.h"

#if defined(_MTG_WITH_TCL)
    #include "MtgTclBondMath.h"
    #include "MtgTclCgi.h"
    #include "MtgTclHtmlReader.h"
#endif

MTG_BEGIN_NAMESPACE


//
//   c l e a n u p
//

void tShell::cleanup()

{
    for( int i = 0; i < m_At.numOfElems(); ++i ) {
        MTG_ASSERT( m_At[i].m_sScript != 0 );
        delete m_At[i].m_sScript;
    }
    m_At.reset();

    if( m_OnError.m_sScript != 0 ) {
        delete m_OnError.m_sScript;
        m_OnError.m_sScript = 0;
    }
}


//
//   a d d A t
//

void tShell::addAt( tType nType, long nTime, char* sScript )

{
    MTG_ASSERT( nTime >= 0 && sScript != 0 );

    int k = m_At.numOfElems();

    ++m_At;

        // keep sorted
        
    while( k > 0 && m_At[k - 1].m_nTime > nTime ) {
        m_At[k] = m_At[k - 1];
        --k;
    }

    m_At[k].m_nType = nType;
    m_At[k].m_nTime = nTime;
    m_At[k].m_sScript = sScript;    
}


//
//   s e t O n E r r o r
//

void tShell::setOnError( tType nType, char* sScript )

{
    MTG_ASSERT( sScript != 0 );

    if( m_OnError.m_sScript != 0 )
        delete m_OnError.m_sScript;

    m_OnError.m_sScript = sScript;
    m_OnError.m_nType = nType;
}


//
//   r u n T c l
//

tRetCode tShell::runTcl( const char* sScript )

{
#if defined(_MTG_WITH_TCL)

    tRetCode nRet;

    if( m_nNesting >= 8 )
        return NESTED_SHELL;    // limit recursion

    ++m_nNesting;
    nRet = m_Kernel.runFile( sScript );
    --m_nNesting;
    return nRet;

#else
    return NOT_AVAILABLE;
#endif
}


//
//   r u n M t g
//

tRetCode tShell::runMtg( const char* sScript, bool bMakeSafe )

{
    tFileSource Source( sScript );   
    tParser Parser;
    tRepository Repository;
    tEvaluate* pEvaluate;
    tRetCode nRet;

    if( m_nNesting >= 8 )
        return NESTED_SHELL;    // limit recursion

    if( bMakeSafe )
        Source.makeSafe();
    if( ( nRet = Parser.setSource( Source ) ) != OK )
        return nRet;

    if( m_pParser != 0 && m_pParser->hasRepository() )
        Parser.setRepository( m_pParser->repository() );
    else
        Parser.setRepository( Repository );

    Parser.setShell( this );

    ++m_nNesting;

    while( ( nRet = Parser.parseObjects( pEvaluate ) ) == OK &&
                pEvaluate != 0 ) {
        nRet = pEvaluate->run();
        delete pEvaluate;
        if( nRet != OK )
            break;
    }

    --m_nNesting;

    if( nRet != OK && m_pParser != 0 ) 
        m_pParser->setError( Parser.getErrorMsg() );

    return nRet;
}


//
//   r u n O S
//

tRetCode tShell::runOS( const char* sCmdLine )

{
    if( ::system( sCmdLine ) != 0 )
        return SCRIPT_ERROR;
    return OK;
}


//
//   p a r s e A t
//

tRetCode tShell::parseAt( tParser& Parser, tType nType, char* sScript )

{
    MTG_ASSERT( Parser.curToken() == xTokAt );
    
    tRetCode nRet;
    long nTime;

    if( m_nNesting > 0 )
        return NESTED_SHELL;

    if( ( nRet = Parser.readToken() ) != OK )
        return nRet;

    do {
        if( Parser.beginOfNumber() ) {
            if( ( nRet = Parser.scanInteger( nTime, 0 ) ) != OK )
                return nRet;
            nTime *= 3600;      // interpret as hours
        }
        else 
            if( Parser.curToken() == xTokString ) {
                char *sTime = 0;

                if( ( nRet = Parser.scanString( sTime ) ) != 0 )
                    return nRet;

                long h;
                long m = 0;
                long s = 0;

                if( sscanf( sTime, "%ld:%ld:%ld", &h, &m, &s ) < 1 ) {
                    delete sTime;
                    return INVALID_TIME;
                }

                nTime = h * 3600 + m * 60 + s;
                delete sTime;
            }
            else {
                return Parser.setError( INVALID_TIME );
            }

        addAt( nType, nTime, StrCopy( sScript ) );
    } while ( Parser.beginOfNumber() || Parser.curToken() == xTokString );

    return OK;
}


//
//   p a r s e P r e f i x
//

tRetCode tShell::parsePrefix( tParser& Parser, tParseInfoStub& Info )

{
    if( m_nNesting == 0 )
        cleanup();  // otherwise, just simply commands are allowed
    return super::parsePrefix( Parser, Info );
}


//
//   p a r s e P a r a m
//

tRetCode tShell::parseParam( tParser& Parser, tParseInfoStub& Info )

{
    tRetCode nRet;

    char *s1 = 0;
    char *s2 = 0;

    switch( Parser.curToken() ) {
        case xTokCD :
            if( ( nRet = Parser.readToken() ) != OK ||
                ( nRet = Parser.scanString( s1 ) ) != OK ) { 
                return nRet;
            }
            SetDir( s1 );
            delete s1;
            break;

        case xTokSet :

#if defined(_MTG_WITH_TCL)

            if( ( nRet = Parser.readToken() ) != OK ||
                ( nRet = Parser.scanString( s1 ) ) != OK ) { 
                return nRet;
            }
            if( ( nRet = Parser.scanString( s2 ) ) != OK ) { 
                delete s1;
                return nRet;
            }
            m_Kernel.setVar( s1, s2 );
            delete s1;
            delete s2;
            break;

#else
            return Parser.setError( NOT_AVAILABLE );
#endif

        case xTokTcl :

#if defined(_MTG_WITH_TCL)

            if( ( nRet = Parser.readToken() ) != OK ||
                ( nRet = Parser.scanString( s1 ) ) != OK ) { 
                return nRet;
            }
            if( Parser.curToken() == xTokOnError ) {
                setOnError( xTcl, s1 );
                if( ( nRet = Parser.readToken() ) != OK )
                    return nRet;
            }
            else
            if( Parser.curToken() == xTokAt ) {
                if( ( nRet = parseAt( Parser, xTcl, s1 ) ) != OK ) {
                    delete s1;
                    return nRet;
                }
                delete s1;
            }
            else {
                nRet = runTcl( s1 );
                delete s1;

                if( nRet != OK )
                    return Parser.setError( nRet );
            }
            break;

#else
            return Parser.setError( NOT_AVAILABLE );
#endif

        case xTokMtg :
            if( ( nRet = Parser.readToken() ) != OK ||
                ( nRet = Parser.scanString( s1 ) ) != OK ) { 
                return nRet;
            }
            if( Parser.curToken() == xTokOnError ) {
                setOnError( xMtg, s1 );
                if( ( nRet = Parser.readToken() ) != OK )
                    return nRet;
            }
            else
            if( Parser.curToken() == xTokAt ) {
                if( ( nRet = parseAt( Parser, xMtg, s1 ) ) != OK ) {
                    delete s1;
                    return nRet;
                }
                delete s1;
            }
            else {
                nRet = runMtg( s1, true );
                delete s1;

                if( nRet != OK )
                    return Parser.setError( nRet );
            }
            break;

        case xTokRun :
            if( ( nRet = Parser.readToken() ) != OK ||
                ( nRet = Parser.scanString( s1 ) ) != OK ) { 
                return nRet;
            }
            if( Parser.curToken() == xTokOnError ) {
                setOnError( xOS, s1 );
                if( ( nRet = Parser.readToken() ) != OK )
                    return nRet;
            }
            else
            if( Parser.curToken() == xTokAt ) {
                if( ( nRet = parseAt( Parser, xOS, s1 ) ) != OK ) {
                    delete s1;
                    return nRet;
                }
                delete s1;
            }
            else {
                nRet = runOS( s1 );
                delete s1;

                if( nRet != OK )
                    return Parser.setError( nRet );
            }
            break;

        default :   
            return super::parseParam( Parser, Info );
    }

    return OK;
}

#if defined(_MTG_WITH_TCL)


//
//   c r e a t e O b j C m d
//

int tShell::createObjCmd( ClientData clientData, Tcl_Interp *pInterp,
    int objc, Tcl_Obj *CONST objv[], bool (*test)( const tObject* pObj ),
    Tcl_ObjCmdProc *CmdProc, Tcl_CmdDeleteProc *DeleteProc )

{
    tRetCode nRet;
    tObject* pObj;
    tRepository Rep, *pRep;

    tSource* pSource = 0;
    char* sObj = 0;
    char* sName = 0;

        // reuse repository

    if( m_pParser != 0 && m_pParser->hasRepository() )
        pRep = &m_pParser->repository();
    else
        pRep = &Rep;

    for( int i = 1; i < objc; ++i ) {
        int nLength;
        char* sString = StrToLower( Tcl_GetStringFromObj( objv[i], &nLength ) );

        if( strcmp( sString, "-file" ) == 0 ) {
            delete sString;
            if( pSource != 0 || i + 1 == objc )
                goto error;
            pSource =
                new tFileSource( Tcl_GetStringFromObj( objv[++i], &nLength ) );
        }
        else
        if( strcmp( sString, "-spec" ) == 0 ) {
            delete sString;
            if( pSource != 0 || i + 1 == objc )
                goto error;
            pSource =
                new tStringSource( Tcl_GetStringFromObj( objv[++i], &nLength ) );
        }
        else
        if( strcmp( sString, "-name" ) == 0 ) {
            delete sString;
            if( sName != 0 || i + 1 == objc )
                goto error;
            sName = StrCopy( Tcl_GetStringFromObj( objv[++i], &nLength ) );
        }
        else
        if( strcmp( sString, "-obj" ) == 0 ) {
            delete sString;
            if( sObj != 0 || i + 1 == objc )
                goto error;
            sObj = StrCopy( Tcl_GetStringFromObj( objv[++i], &nLength ) );
        }
        else {
            delete sString;
            goto error;
        }
    }

    if( sObj == 0 || sName == 0 )
        goto error;

    if( pSource != 0 ) {
            // nested source

        tParser Parser;
        tEvaluate* pEvaluate;

        pSource->makeSafe();
        Parser.setRepository( pRep );
        Parser.setShell( this );

        if( Parser.setSource( pSource ) != OK )
            goto error;

        ++m_nNesting;

        while( ( nRet = Parser.parseObjects( pEvaluate ) ) == OK &&
                    pEvaluate != 0 ) {
            nRet = pEvaluate->run();
            delete pEvaluate;
            if( nRet != OK )
                break;
        }

        --m_nNesting;

        if( nRet != OK ) {
            if( m_pParser != 0 ) 
                m_pParser->setError( Parser.getErrorMsg() );
            goto error;
        }
    }

    if( ( pObj = pRep->find( sObj ) ) != 0 && test( pObj ) ) {
        tObject* p = pObj->clone();

        if( Tcl_CreateObjCommand( pInterp, sName, CmdProc,
                (ClientData) p, DeleteProc ) == NULL ) {
            delete p;
        }
    }
    else {
        goto error;
    }

    if( pSource != 0 )
        delete pSource;
    delete sName;
    delete sObj;

    return TCL_OK;

error:

    if( pSource != 0 )
        delete pSource;
    if( sName != 0 )
        delete sName;
    if( sObj != 0 )
        delete sObj;

    return TCL_ERROR;
}


//
//   c r e a t e D r i f t C m d
//

int tShell::createDriftCmd( ClientData clientData,
    Tcl_Interp *pInterp, int objc, Tcl_Obj *CONST objv[] )

{
    tShell* pShell = static_cast<tShell*>( clientData );

    return pShell->createObjCmd( clientData, pInterp, objc, objv,
        tDrift::tclTest, tDrift::tclCommand, tDrift::tclDelete );
}


//
//   c r e a t e B o o t s t r a p C m d
//

int tShell::createBootstrapCmd( ClientData clientData,
    Tcl_Interp *pInterp, int objc, Tcl_Obj *CONST objv[] )

{
    tShell* pShell = static_cast<tShell*>( clientData );

    return pShell->createObjCmd( clientData, pInterp, objc, objv,
        tBootstrap::tclTest, tBootstrap::tclCommand, tBootstrap::tclDelete );
}


//
//   s h e l l F r o m T c l
//

int tShell::shellFromTcl( ClientData clientData,
    Tcl_Interp *pInterp, int objc, Tcl_Obj *CONST objv[] )

{
    tShell* pShell = static_cast<tShell*>( clientData );

    if( objc != 2 ) {
        Tcl_WrongNumArgs( pInterp, 1, objv, "filename" );
        return TCL_ERROR;
    }

    int nLength;
    char* sCmd = Tcl_GetStringFromObj( objv[1], &nLength );

    tRetCode nRet = pShell->runMtg( sCmd, false );

    if( nRet != OK ) {
        Tcl_AddErrorInfo( pInterp, "" );           
        return TCL_ERROR;
    }

    return TCL_OK;
}


//
//   d a t a F r o m T c l
//

int tShell::dataFromTcl( ClientData clientData,
    Tcl_Interp *pInterp, int objc, Tcl_Obj *CONST objv[] )

{
    if( objc != 2 ) {
        Tcl_WrongNumArgs( pInterp, 1, objv, "filename" );
        return TCL_ERROR;
    }

    tRetCode nRet;

    int nLength;
    char* sFileName = Tcl_GetStringFromObj( objv[1], &nLength );

    tFileSource Source( sFileName );
    tDataScanner Scanner;

    if( ( nRet = Scanner.setSource( Source ) ) != OK ) {
        Tcl_AddErrorInfo( pInterp, 
            const_cast<char*>( Scanner.getErrorMsg() ) );
        return TCL_ERROR;
    }

    tHeap<Tcl_Obj*> Item;

    if( ( nRet = Scanner.append( Item ) ) != OK ) {
        printf( "%d\n", nRet );
        for( int i = 0; i < Item.numOfElems(); ++i )
            delete Item[i];
        Tcl_AddErrorInfo( pInterp, 
            const_cast<char*>( Scanner.getErrorMsg() ) );
        return TCL_ERROR;
    }

    if( Item.numOfElems() == 0 ) { 
        Tcl_SetObjResult( pInterp, Tcl_NewListObj( 0, 0 ) );
    }
    else {
        Tcl_SetObjResult( pInterp, 
            Tcl_NewListObj( Item.numOfElems(), &Item[0] ) );
    }

    return TCL_OK;
}



#endif


//
//   t T c l S h e l l
//

tShell::tShell()

{
    m_pParser = 0;
    m_nNesting = 0;

    m_OnError.m_sScript = 0;

#if defined(_MTG_WITH_TCL)

        // Mtg scripts can be run from within Tcl.

    Tcl_CreateObjCommand( m_Kernel.interp(),
        "Mtg::createDriftCmd", createDriftCmd,
        static_cast<ClientData>( this ), 0 );

    Tcl_CreateObjCommand( m_Kernel.interp(),
        "Mtg::createBootstrapCmd", createBootstrapCmd,
        static_cast<ClientData>( this ), 0 );

    Tcl_CreateObjCommand( m_Kernel.interp(),
        "Mtg::shell", shellFromTcl, static_cast<ClientData>( this ), 0 );

    Tcl_CreateObjCommand( m_Kernel.interp(),
        "Mtg::data", dataFromTcl, static_cast<ClientData>( this ), 0 );

        // Add some useful Tcl extensions.

    tRetCode nRet;

    if( ( nRet = tTclBondMath::createExtension( m_Kernel ) ) != OK ||
        ( nRet = tTclCgi::createExtension( m_Kernel ) ) != OK ||
        ( nRet = tTclHtmlReader::createExtension( m_Kernel ) ) != OK ) {
        throw tException( nRet );
    }

#endif
}


//
//   ~ t T c l S h e l l
//

tShell::~tShell()

{
    cleanup();
}


//
//   s e t P a r s e r
//

void tShell::setParser( tParser* pParser )

{
    m_pParser = pParser;
}


//
//   f i n a l i z e
//

tRetCode tShell::finalize()

{
    return super::finalize();
}


//
//   r u n
//

tRetCode tShell::run()

{
    if( m_nNesting == 0 && m_At.numOfElems() > 0 ) {
        tRetCode nRet;

        long nSecs = SecsOfDay();
        int nAt = 0;

            // go to right starting place

        while( nAt < m_At.numOfElems() && m_At[nAt].m_nTime < nSecs )
            ++nAt;
        if( nAt == m_At.numOfElems() )
            nAt = 0;

            // loop forever or until error

        while( true ) {
            MTG_TRACE( "Waiting until %02ld:%02ld:%02ld\n",
                m_At[nAt].m_nTime / 3600, 
                m_At[nAt].m_nTime / 60 % 60,
                m_At[nAt].m_nTime % 60 );

            WaitUntil( m_At[nAt].m_nTime );

            switch( m_At[nAt].m_nType ) {
                case xTcl :

#if defined(_MTG_WITH_TCL )
                    if( ( nRet = runTcl( m_At[nAt].m_sScript ) ) != OK )
                        return nRet;
                    break;
#else
                    return NOT_AVAILABLE;
#endif

                case xMtg :
                    if( ( nRet = runMtg( m_At[nAt].m_sScript, true ) ) != OK )
                        return nRet;
                    break;

                case xOS :
                    if( ( nRet = runOS( m_At[nAt].m_sScript ) ) != OK )
                        return nRet;
                    break;

                default :
                    throw tException( INTERNAL_ERROR );
            }

            if( ++nAt >= m_At.numOfElems() )
                nAt = 0;
        }
    }

    return OK;
}


//
//   o n E r r o r
//

tRetCode tShell::onError()

{
    tRetCode nRet;

    if( m_OnError.m_sScript == 0 )
        return OK;

    switch( m_OnError.m_nType ) {
        case xTcl :

#if defined(_MTG_WITH_TCL )
            if( m_pParser != 0 )
                m_Kernel.setVar( "Mtg::errorMsg", m_pParser->getErrorMsg() );
            else
                m_Kernel.setVar( "Mtg::errorMsg", "" );
            if( ( nRet = runTcl( m_OnError.m_sScript ) ) != OK )
                return nRet;
            break;
#else
            return NOT_AVAILABLE;
#endif

        case xMtg :
            if( ( nRet = runMtg( m_OnError.m_sScript, true ) ) != OK )
                return nRet;
            break;

        case xOS :
            if( ( nRet = runOS( m_OnError.m_sScript ) ) != OK )
                return nRet;
            break;

        default :
            throw tException( INTERNAL_ERROR );
    }

    return OK;
}


//
//   p a r s e
//

tRetCode tShell::parse( tParser& Parser )

{
    tRetCode nRet;

    if( m_pParser == 0 )
        m_pParser = &Parser;

    if( ( nRet = Parser.scanBeginOfObj() ) != OK )
        return nRet;
    return super::parse( Parser );
}


//
//   p a r s e
//

tRetCode tShell::parse( tParser& Parser, tSystem& System, tObject*& pObj )

{
    tRetCode nRet;
    tShell* pShell;

    if( ( nRet = Parser.scanBeginOfObj() ) != OK )
        return nRet;

    pShell = new tShell;
    pShell->setSystem( System );

    if( ( nRet = pShell->parse( Parser ) ) != OK ) {
        delete pShell;
        return nRet;
    }

    pObj = pShell;
    return OK;
}

MTG_END_NAMESPACE
