// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#include "MtgIncl.h"
#include "MtgShortRateModel.h"
#include "MtgBootstrap.h"
#include "MtgDrift.h"
#include "MtgPathSpace.h"
#include "MtgShortRateEngine.h"

MTG_BEGIN_NAMESPACE


//
//   t E v o l u t i o n
//

tShortRateModel::tEvolution::tEvolution( const tShortRateModel& Model,
    const tPathSpace& PathSpace )
    : m_Model( Model ), m_PathSpace( PathSpace )

{
    if( m_Model.m_gInitial > 0 ) {
        m_gInitial = m_Model.m_gInitial;
        return;
    }

    if( m_Model.m_pInitialDrift != 0 ) {
            // here we don't need datebase-compatibility
        const tDrift& Drift = *m_Model.m_pInitialDrift;
        m_gInitial = Drift.forward( PathSpace.base() );
        return;
    }

    if( m_Model.m_pInitialBoot != 0 ) {
        tInterestSpline& Spline = m_Model.m_pInitialBoot->spline();
        tInterest I( tInterest::xExponential, tInterest::xYield,
            PathSpace.dayCount() );

        tDate End;

        Spline.getFirstLeg( End, m_gInitial, I );
        while( PathSpace.base() >= End &&
               Spline.getNextLeg( End, m_gInitial, I ) );

        return;
    }

    throw tException( INTERNAL_ERROR );    
}


//
//   i n i t
//

void tShortRateModel::init()

{
    m_pInitialBoot = 0;
    m_pInitialDrift = 0;
    m_gInitial = 0;
}


//
//   c o p y F r o m
//

void tShortRateModel::copyFrom( const tShortRateModel& Model )

{
    if( &Model == this )
        return;

    m_gInitial = Model.m_gInitial;
    setObjRefToZeroOrSo( m_pInitialBoot, Model.m_pInitialBoot );
    setObjRefToZeroOrSo( m_pInitialDrift, Model.m_pInitialDrift );

    super::copyFrom( Model );
}


//
//   p a r s e P a r a m
//

tRetCode tShortRateModel::parseParam( tParser& Parser,
    tParseInfoStub& Info )

{
    tRetCode nRet;

    switch( Parser.curToken() ) {
        case xTokInitial :
            if( ( nRet = Parser.readToken() ) != OK )
                return nRet;

            if( m_gInitial > 0 ||
                m_pInitialBoot != 0 || m_pInitialDrift != 0 ) {
                return Parser.setError( ATTR_REDEFINITION );
            }

            if( Parser.curToken() == xTokId ) {
                tObject* pObj;
                tBootstrap* pBoot;
                tDrift* pDrift;

                if( ( pObj = Parser.findObject() ) == 0 )
                    return Parser.setError( NOT_FOUND );

                if( ( pBoot = dynamic_cast<tBootstrap*>( pObj ) ) != 0 ) {
                    setObjRefToZeroOrSo( m_pInitialBoot, pBoot );
                }
                else
                if( ( pDrift = dynamic_cast<tDrift*>( pObj ) ) != 0 ) {
                    setObjRefToZeroOrSo( m_pInitialDrift, pDrift );
                }
                else {
                    return Parser.setError( "bootstrap/drift expected" );
                }
                if( ( nRet = Parser.readToken() ) != OK )
                    return nRet;
            }
            else
            if( Parser.beginOfNumber() ) {
                if( ( nRet = Parser.scanPosPercentage( m_gInitial ) ) != OK )
                    return nRet;
            }
            else {
                return Parser.setError( INVALID_KEYWORD );
            }
            break;

        default :
            return super::parseParam( Parser, Info );
    }

    return OK;
}


//
//   p a r s e P o s t f i x
//

tRetCode tShortRateModel::parsePostfix( tParser& Parser,
    tParseInfoStub& Info )

{
    if( m_gInitial <= 0 && m_pInitialBoot == 0 && m_pInitialDrift == 0 )
        return Parser.setError( MISSING_ROOT );
    if( numOfFactors() == 0 )
        return Parser.setError( MISSING_FACTOR );
    return super::parsePostfix( Parser, Info );
}


//
//   t S h o r t R a t e M o d e l
//

tShortRateModel::tShortRateModel()

{
    init();
}


//
//   t S h o r t R a t e M o d e l
//

tShortRateModel::tShortRateModel( const tShortRateModel& Model )

{
    init();
    copyFrom( Model );
}


//
//   ~ t S h o r t R a t e M o d e l
//

tShortRateModel::~tShortRateModel()

{
    setObjRefToZeroOrSo( m_pInitialBoot );
    setObjRefToZeroOrSo( m_pInitialDrift );
}


//
//   c r e a t e E n g i n e
//

tRetCode tShortRateModel::createEngine( tMCEngine*& pEngine )

{
    MTG_ASSERT( isFinalized() );

    pEngine = new tShortRateEngine;
    pEngine->setModel( this );
    return OK;
}


//
//   f i n a l i z e
//

tRetCode tShortRateModel::finalize()

{
    return super::finalize();
}

MTG_END_NAMESPACE
