// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#include "MtgIncl.h"
#include "MtgShortRateTermStruct.h"
#include "MtgInterestSpline.h"
#include "MtgStepDrift.h"

MTG_BEGIN_NAMESPACE


//
//   p r o g r e s s
//

tRetCode tShortRateTermStruct::tEvolutionStub::progress( 
    const tShortRateTermStruct& ShortRate, tHeap<double>& Forward )

{
    Forward.fill( 0 );  // dummy
    return OK;
}


//
//   i n i t
//

void tShortRateTermStruct::init()

{
    m_bIsFinalized = false;
}


//
//   a l l o c S p a c e
//

void tShortRateTermStruct::allocSpace( int nNumOfSamples )

{
    m_Forward.numOfElems( nNumOfSamples );
}


//
//   a l l o c S p a c e
//

void tShortRateTermStruct::allocSpace( const tDrift& Drift )

{
    MTG_ASSERT( compatibleDateBaseWith( Drift ) );

    int nNumOfSamples = Drift.maturity();

    if( scale() == xYear && Drift.numOfPeriods() != numOfPeriods() ) {
        double f = (double) numOfPeriods() / Drift.numOfPeriods();
        nNumOfSamples = (int) ceil( nNumOfSamples * f );
    }

    allocSpace( nNumOfSamples  );
}


//
//   l o a d D r i f t
//

tRetCode tShortRateTermStruct::loadDrift( const tDrift& Drift,
    int nNumOfSamples )

{
    tRetCode nRet;
    tStepDrift NewDrift;
    const tDrift* pDrift;

    if( sameDateBaseAs( Drift ) ) {
        pDrift = &Drift;
    }
    else {
            // need to convert first
        static_cast<tDateBase&>( NewDrift ) = *this;

        if( ( nRet = NewDrift.addDrift( Drift ) ) != OK ||
            ( nRet = NewDrift.finalize() ) != OK ) {
            return nRet;
        }
        pDrift = &NewDrift;
    }

    if( nNumOfSamples <= 0 )
        allocSpace( *pDrift );
    else
        allocSpace( nNumOfSamples );

        // load forward rate curve one-to-one
    for( int n = 0; n < m_Forward.numOfElems(); ++n )
        m_Forward[n] = pDrift->forwardSI( n );

        // compute the cumulative forward rates
    for( int i = 1; i < m_Forward.numOfElems(); ++i )
        m_Forward[i] += m_Forward[i - 1];

    return OK;
}


//
//   c o p y F r o m
//

void tShortRateTermStruct::copyFrom( const tDrift& Drift )

{
    tRetCode nRet;

        // inherit date base first
    super::copyFrom( Drift );
    
    if( ( nRet = loadDrift( Drift, -1 ) ) != OK ) {
        m_bIsFinalized = false;
        throw tException( nRet );
    }

    m_bIsFinalized = true;
}


//
//   c o p y F r o m
//

void tShortRateTermStruct::copyFrom( const tShortRateTermStruct& ShortRate )

{
    if( this == &ShortRate )
        return;

    m_Forward = ShortRate.m_Forward;
    m_bIsFinalized = ShortRate.m_bIsFinalized;

    super::copyFrom( ShortRate );
}


//
//   t S h o r t R a t e T e r m S t r u c t
//

tShortRateTermStruct::tShortRateTermStruct()

{
    init();
}


//
//   t S h o r t R a t e T e r m S t r u c t
//

tShortRateTermStruct::tShortRateTermStruct( const tDrift& Drift )

{
    init();
    copyFrom( Drift );
}


//
//   t S h o r t R a t e T e r m S t r u c t
//

tShortRateTermStruct::tShortRateTermStruct( tScale nScale )
    : tDateBase( nScale )

{
    init();
}


//
//   t S h o r t R a t e T e r m S t r u c t
//

tShortRateTermStruct::tShortRateTermStruct( tScale nScale,
    int nNumOfPeriods )
    : tDateBase( nScale, nNumOfPeriods )

{
    init();
}


//
//   t S h o r t R a t e T e r m S t r u c t
//

tShortRateTermStruct::tShortRateTermStruct( tScale nScale,
    const tDayCount& DayCount )
    : tDateBase( nScale, DayCount )

{
    init();
}


//
//   t S h o r t R a t e T e r m S t r u c t
//

tShortRateTermStruct::tShortRateTermStruct( tScale nScale,
    int nNumOfPeriods, const tDayCount& DayCount )
    : tDateBase( nScale, nNumOfPeriods, DayCount )

{
    init();
}


//
//   t S h o r t R a t e T e r m S t r u c t
//

tShortRateTermStruct::tShortRateTermStruct( tScale nScale,
    const tDayCount& DayCount, tDate Base )
    : tDateBase( nScale, DayCount, Base )

{
    init();
}


//
//   t S h o r t R a t e T e r m S t r u c t
//

tShortRateTermStruct::tShortRateTermStruct( tScale nScale,
    int nNumOfPeriods, const tDayCount& DayCount, tDate Base )
    : tDateBase( nScale, nNumOfPeriods, DayCount, Base )

{
    init();
}


//
//   t S h o r t R a t e T e r m S t r u c t
//

tShortRateTermStruct::tShortRateTermStruct(
    const tShortRateTermStruct& ShortRate )

{
    init();
    copyFrom( ShortRate );
}


//
//   ~ t S h o r t R a t e T e r m S t r u c t
//

tShortRateTermStruct::~tShortRateTermStruct()

{
}


//
//   o p e r a t o r =
//

tShortRateTermStruct& tShortRateTermStruct::operator=(
    const tShortRateTermStruct& ShortRate )

{
    if( this != &ShortRate )
        copyFrom( ShortRate );
    return *this;
}


//
//   o p e r a t o r =
//

tShortRateTermStruct& tShortRateTermStruct::operator=(
    const tDrift& Drift )

{
    copyFrom( Drift );
    return *this;
}


//
//   l o a d
//

tRetCode tShortRateTermStruct::load( const tDrift& Drift,
    int nNumOfSamples )

{
    tRetCode nRet;

    if( ( nRet = loadDrift( Drift, nNumOfSamples ) ) != OK ) {
        m_bIsFinalized = false;
        return nRet;
    }

    m_bIsFinalized = true;
    return OK;
}


//
//   l o a d I n i t i a l
//

tRetCode tShortRateTermStruct::load( tInterestSpline& Spline,
    int nNumOfSamples )

{
    tRetCode nRet;

        // create temporary tStepDrift object:
    tStepDrift Drift( scale(), numOfPeriods(), dayCount(), base() );
        // load drift object with interest spline:
    
    if( ( nRet = Drift.addSpline( Spline ) ) != OK ||
        ( nRet = Drift.finalize() ) != OK ) {
        return nRet;
    }

        // and use the result:
    return load( Drift, nNumOfSamples );
}


//
//   f i n a l i z e
//

tRetCode tShortRateTermStruct::finalize( tEvolutionStub& Evolution )

{
    tRetCode nRet;

        // Finalize always, even if already finalized!
        // Propagate the initial value by calling evolution().
        // As a side-effect, the size of m_Forward will be fixed.

    if( ( nRet = Evolution.progress( *this, m_Forward ) ) != OK ) {
        m_bIsFinalized = false;
        return nRet;
    }

#if defined(_DEBUG)
    /*
    MTG_TRACE( "Forward:" );
    for( int l = 0; l < m_Forward.numOfElems(); ++l )
        MTG_TRACE( " % .3lf", m_Forward[l] );
    MTG_TRACE( "\n" );
    */
#endif

        // Now compute the cumulative forward rates.

    for( int i = 1; i < m_Forward.numOfElems(); ++i )
        m_Forward[i] += m_Forward[i - 1];

#if defined(_DEBUG)
    /*
    MTG_TRACE( "Accumulated:" );
    for( int j = 0; j < m_Forward.numOfElems(); ++j )
        MTG_TRACE( " % .3lf", m_Forward[j] );
    MTG_TRACE( "\n" );
    */
#endif

    m_bIsFinalized = true;
    return OK;
}


//
//   i s C o v e r e d S I
//

bool tShortRateTermStruct::isCoveredSI( int nSub ) const

{
    return nSub >= 0 && nSub < m_Forward.numOfElems();
}


//
//   i s C o v e r e d S I
//

bool tShortRateTermStruct::isCoveredSI( double gSub ) const

{
    return gSub >= 0 && gSub < m_Forward.numOfElems();
}


//
//   i s C o v e r e d T U
//

bool tShortRateTermStruct::isCoveredTU( int nUnit ) const

{
    return isCoveredSI( nUnit * numOfPeriods() );
}


//
//   i s C o v e r e d T U
//

bool tShortRateTermStruct::isCoveredTU( double gUnit ) const

{
    return isCoveredSI( gUnit * numOfPeriods() );
}


//
//   i s C o v e r e d
//

bool tShortRateTermStruct::isCovered( tDate Date ) const

{
    if( scale() == xDay ) {
        int n;

        getSI( Date, n );
        return isCoveredSI( n );
    }

    return isCoveredSI( getSI( Date ) );
}


//
//   f o r w a r d T U
//

double tShortRateTermStruct::forwardTU( int nUnit ) const

{
    if( scale() == xDay )
        return forwardSI( nUnit );
    return forwardSI( nUnit * numOfPeriods() );
}


//
//   f o r w a r d T U
//

double tShortRateTermStruct::forwardTU( int nFromUnit, int nToUnit ) const

{
    if( scale() == xDay )
        return forwardSI( nFromUnit, nToUnit );
    return forwardSI( nFromUnit * numOfPeriods(), nToUnit * numOfPeriods() );
}


//
//   f o r w a r d T U
//

double tShortRateTermStruct::forwardTU( double gUnit ) const

{
    if( scale() == xDay )
        return forwardSI( gUnit );
    return forwardSI( gUnit * numOfPeriods() );
}


//
//   f o r w a r d T U
//

double tShortRateTermStruct::forwardTU( double gFromUnit,
    double gToUnit ) const

{
    if( scale() == xDay )
        return forwardSI( gFromUnit, gToUnit );
    return forwardSI( gFromUnit * numOfPeriods(), gToUnit * numOfPeriods() );
}


//
//   f o r w a r d S I
//

double tShortRateTermStruct::forwardSI( int nUnit ) const

{
    MTG_ASSERT( m_bIsFinalized );

    int n = m_Forward.numOfElems();

    if( n == 0 )
        return 0;

    if( nUnit >= n ) {
            // be lenient and return tail
        if( n == 1 )
            return m_Forward.last();
        return m_Forward.last() - m_Forward[n - 2];
    }

    if( nUnit <= 0 ) 
        return m_Forward[0];

    return m_Forward[nUnit] - m_Forward[nUnit - 1];
}


//
//   f o r w a r d S I
//

double tShortRateTermStruct::forwardSI( int nFromSub, int nToSub ) const

{
    MTG_ASSERT( m_bIsFinalized );
    MTG_ASSERT( nFromSub <= nToSub );

    if( nFromSub == nToSub )
        return forwardSI( nFromSub );
    if( nFromSub == 0 ) 
        return m_Forward[nToSub - 1] / nToSub;
    return ( m_Forward[nToSub - 1] - m_Forward[nFromSub - 1] ) / 
                ( nToSub - nFromSub );
}


//
//   f o r w a r d S I
//

double tShortRateTermStruct::forwardSI( double g ) const

{
    return forwardSI( (int) floor( g ) );
}


//
//   f o r w a r d S I
//

double tShortRateTermStruct::forwardSI( double gFromSub,
    double gToSub ) const

{
    MTG_ASSERT( m_bIsFinalized );
    MTG_ASSERT( gFromSub <= gToSub );

    if( gFromSub == gToSub )
        return forwardSI( gFromSub );

    int nCeilFrom = (int) ceil( gFromSub );
    int nFloorTo = (int) floor( gToSub );

    if( nCeilFrom > nFloorTo ) {
            // interval is within the same sub interval
        return forwardSI( gFromSub );
    }

    double gRate = 0;

    if( nCeilFrom < nFloorTo ) {
            // interval spans several sub intervals
        if( nCeilFrom == 0 )
            gRate += m_Forward[nFloorTo - 1];
        else
            gRate += m_Forward[nFloorTo - 1] - m_Forward[nCeilFrom - 1];
    }

        // now look at head and tail
    if( nCeilFrom > gFromSub )
        gRate += ( nCeilFrom - gFromSub ) * forwardSI( gFromSub );
    if( nFloorTo < gToSub )
        gRate += ( gToSub - nFloorTo ) * forwardSI( nFloorTo );

    return gRate / ( gToSub - gFromSub );
}


//
//   f o r w a r d
//

double tShortRateTermStruct::forward( tDate Date ) const

{
    if( scale() == xDay ) {
        int nSub;

        getSI( Date, nSub );
        return forwardSI( nSub );
    }

    return forwardSI( getSI( Date ) );
}


//
//   f o r w a r d
//

double tShortRateTermStruct::forward( tDate Start, tDate End ) const

{
    if( scale() == xDay ) {
        int nFromSub, nToSub;

        getSI( Start, nFromSub );
        getSI( End, nToSub );
        return forwardSI( nFromSub, nToSub );
    }

    return forwardSI( getSI( Start ), getSI( End ) );
}


//
//   p r e s e n t V a l u e T U
//

double tShortRateTermStruct::presentValueTU( int nFromUnit, int nToUnit ) const

{
    double gFwd = forwardTU( nFromUnit, nToUnit );
    return exp( -gFwd * ( nToUnit - nFromUnit ) * dtTU() );
}


//
//   f u t u r e V a l u e T U
//

double tShortRateTermStruct::futureValueTU( int nFromUnit, int nToUnit ) const

{
    return 1 / presentValueTU( nFromUnit, nToUnit );
}


//
//   p r e s e n t V a l u e S I
//

double tShortRateTermStruct::presentValueSI( int nFromSub, int nToSub ) const

{
    double gFwd = forwardSI( nFromSub, nToSub );
    return exp( -gFwd * ( nToSub - nFromSub ) * dtSI() );
}


//
//   f u t u r e V a l u e S I
//

double tShortRateTermStruct::futureValueSI( int nFromSub, int nToSub ) const

{
    return 1 / futureValueSI( nFromSub, nToSub );
}


//
//   p r e s e n t V a l u e T U
//

double tShortRateTermStruct::presentValueTU( double gFromUnit, double gToUnit ) const

{
    double gFwd = forwardTU( gFromUnit, gToUnit );
    return exp( -gFwd * ( gToUnit - gFromUnit ) * dtTU() );
}


//
//   f u t u r e V a l u e T U
//

double tShortRateTermStruct::futureValueTU( double gFromUnit, double gToUnit ) const

{
    return 1 / presentValueTU( gFromUnit, gToUnit );
}


//
//   p r e s e n t V a l u e S I
//

double tShortRateTermStruct::presentValueSI( double gFromSub, double gToSub ) const

{
    double gFwd = forwardSI( gFromSub, gToSub );
    return exp( -gFwd * ( gToSub - gFromSub ) * dtSI() );
}


//
//   f u t u r e V a l u e S I
//

double tShortRateTermStruct::futureValueSI( double gFromSub, double gToSub ) const

{
    return 1 / presentValueSI( gFromSub, gToSub );
}


//
//   p r e s e n t V a l u e
//

double tShortRateTermStruct::presentValue( tDate Start, tDate End ) const

{
    double gFwd = forward( Start, End );
    return exp( -gFwd * dayCount().fraction( Start, End ) );
}


//
//   f u t u r e V a l u e
//

double tShortRateTermStruct::futureValue( tDate Start, tDate End ) const

{
    return 1 / presentValue( Start, End );
}

MTG_END_NAMESPACE
