// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#if ! defined(_MTG_SHORT_RATE_TERM_STRUCT_)
#define _MTG_SHORT_RATE_TERM_STRUCT_

#include "MtgDateBase.h"
#include "MtgHeap.h"

MTG_BEGIN_NAMESPACE

class tDrift;
class tInterestSpline;


//
//   t H J M T e r m S t r u c t
//

class tShortRateTermStruct : public tDateBase {

    typedef tDateBase super;

public:

        // required by finalize(); progress() ist supposed to resize
        // the forward array (as opposed to tHJMTermStruct)

    class tEvolutionStub {
    public:
        virtual tRetCode progress( const tShortRateTermStruct& ShortRate,
            tHeap<double>& Forward );
    };

private:

        // store accumulated forward rates here:

    tHeap<double> m_Forward;
    bool m_bIsFinalized;

    void init();

    void allocSpace( int nNumOfSamples );
    void allocSpace( const tDrift& Drift );
    tRetCode loadDrift( const tDrift& Drift, int nNumOfSamples );

protected:

    void copyFrom( const tDrift& Drift );
    void copyFrom( const tShortRateTermStruct& ShortRate );

public:

    tShortRateTermStruct();
    tShortRateTermStruct( const tDrift& Drift );

        // Passed on to the tDateBase constructor:

    tShortRateTermStruct( tScale nScale );
    tShortRateTermStruct( tScale nScale, int nNumOfPeriods );
    tShortRateTermStruct( tScale nScale, const tDayCount& DayCount );
    tShortRateTermStruct( tScale nScale, int nNumOfPeriods,
        const tDayCount& DayCount );
    tShortRateTermStruct( tScale nScale, const tDayCount& DayCount,
        tDate Base );
    tShortRateTermStruct( tScale nScale, int nNumOfPeriods,
        const tDayCount& DayCount, tDate Base );

    tShortRateTermStruct( const tShortRateTermStruct& Drift );

    ~tShortRateTermStruct();

    tShortRateTermStruct& operator=( const tShortRateTermStruct& ShortRate );
    tShortRateTermStruct& operator=( const tDrift& Drift );

        // the load() functions finalize also

    tRetCode load( const tDrift& Drift, int nNumOfSamples = -1 );
    tRetCode load( tInterestSpline& Spline, int nNumOfSamples = -1 );

        // note that the evolution object is also supposed to
        // resize the curve

    tRetCode finalize( tEvolutionStub& Evolution );

    int numOfSamples() const {
        return m_Forward.numOfElems();
    }

        // The following functions test whether a subinterval,
        // time unit or date is covered by the term structure.

    bool isCoveredSI( int nSub ) const;
    bool isCoveredSI( double gSub ) const;

    bool isCoveredTU( int nUnit ) const;
    bool isCoveredTU( double gUnit ) const;

    bool isCovered( int nSub ) const {
        return isCoveredSI( nSub );
    }

    bool isCovered( double gSub ) const {
        return isCoveredSI( gSub );
    }

    bool isCovered( tDate Date ) const;

        // Retrieval based on time unit and scaling:

    double forwardTU( int nUnit ) const;
    double forwardTU( int nFromUnit, int nToUnit ) const;
    double forwardTU( double gUnit ) const;
    double forwardTU( double gFromUnit, double gToUnit ) const;

        // Retrieval independent of time unit and scaling:

    double forwardSI( int nSub ) const;
    double forwardSI( int nFromSub, int nToSub ) const;
    double forwardSI( double gSub ) const;
    double forwardSI( double gFromSub, double gToSub ) const;

        // By default, retrieval is based on subintervals:

    double forward( int nSub ) const {
        return forward( nSub );
    }

    double forward( int nFromSub, int nToSub ) const {
        return forward( nFromSub, nToSub );
    }

    double forward( double gFromSub, double gToSub ) const {
        return forward( gFromSub, gToSub );
    }

        // Date based retrieval:

    double forward( tDate Date ) const;
    double forward( tDate Start, tDate End ) const;

        // Apply the forward rates:

    double presentValueTU( int nFromUnit, int nToUnit ) const;
    double futureValueTU( int nFromUnit, int nToUnit ) const;

    double presentValueSI( int nFromSub, int nToSub ) const;
    double futureValueSI( int nFromSub, int nToSub ) const;

        // The default is to retrieve by subinterval:

    double presentValue( int nFromSub, int nToSub ) const {
        return presentValueSI( nFromSub, nToSub );
    }

    double futureValue( int nFromSub, int nToSub ) const {
        return futureValueSI( nFromSub, nToSub );
    }

        // The following functions work only if the
        // scale is years, not days.

    double presentValueTU( double gFromUnit, double gToUnit ) const;
    double futureValueTU( double gFromUnit, double gToUnit ) const;

    double presentValueSI( double gFromSub, double gToSub ) const;
    double futureValueSI( double gFromSub, double gToSub ) const;

        // The default is to retrieve by subinterval:

    double presentValue( double gFromSub, double gToSub ) const {
        return presentValueSI( gFromSub, gToSub );
    }                         

    double futureValue( double gFromSub, double gToSub ) const {
        return futureValueSI( gFromSub, gToSub );
    }                         

        // The following two functions are consistently based
        // on the day count convention and work under all scales.
        // It is assumed the investment is made at the beginning
        // of date Start, and paid back at the beginning of
        // date End.

    double presentValue( tDate Start, tDate End ) const;
    double futureValue( tDate Start, tDate End ) const;
};

MTG_END_NAMESPACE

#endif
