// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#if ! defined(_MTG_SIG_BAG_)
#define _MTG_SIG_BAG_

#include "MtgHeap.h"
#include "MtgHeap2.h"
#include "MtgSignature.h"

MTG_BEGIN_NAMESPACE


//
//   t S i g B a g
//

class tSigBag {

    const tSignature* m_pRefSignature;

    tHeap<int> m_ForceExIndex;
    tHeap<int> m_MayExIndex[2];

        // To temporarily reset m_MayExIndex[i], the following
        // array stores the original number of elements.

    int m_NumOfMayExIndex[2];

    bool m_bHasAlternatives;

        // Temporary for retrieval:

    int m_nAlternative0;
    int m_nAlternative1;

    int m_nAltNumOfExIndexes;
    int m_AltMayExIndex[2];

    tSignature m_AltSignature;

        // Some helper functions.

    void init();

public:

    tSigBag();
    tSigBag( const tSigBag& Bag );

    ~tSigBag();

    void reset() {
        m_ForceExIndex.reset();
        m_MayExIndex[0].reset();
        m_MayExIndex[1].reset();
        m_bHasAlternatives = false;
    }

    tSigBag& operator=( const tSigBag& Bag );
    bool operator==( const tSigBag& Bag ) const;

    void copyFrom( const tSigBag& Bag );

    void setRefSignature( const tSignature* pSig );

    void forceExercise( int nIndex ) {
        MTG_ASSERT( (*m_pRefSignature)[nIndex] );
        m_ForceExIndex.append( nIndex );
        m_bHasAlternatives = true;
    }

    void mayExercise( int nBag, int nIndex ) {
        MTG_ASSERT( (*m_pRefSignature)[nIndex] );
        m_MayExIndex[nBag].append( nIndex );
        m_bHasAlternatives = true;
    }

    void mayExercise( bool bUnderControl, int nIndex ) {
        if( bUnderControl )
            mayExercise( 0, nIndex );
        else
            mayExercise( 1, nIndex );
    }

    void forceExercise( bool, int nIndex ) {
        forceExercise( nIndex );
    }

    bool hasForceExAlternatives() const {
        return m_ForceExIndex.numOfElems() > 0;
    }

    bool hasMayExAlternatives() const {
        return m_MayExIndex[0].numOfElems() > 0 || 
               m_MayExIndex[1].numOfElems() > 0;
    }

    bool hasAlternatives() const {
        return m_bHasAlternatives;
    }

    int numOfAlternatives( int nBag ) const {
        return m_MayExIndex[nBag].numOfElems();
    }

    int numOfAlternatives() const {
        return m_MayExIndex[0].numOfElems() + 
               m_MayExIndex[1].numOfElems() + 1;
    }

    void hideMayExAlternatives( int nBag ) {
        m_NumOfMayExIndex[nBag] = m_MayExIndex[nBag].numOfElems();
        m_MayExIndex[nBag].numOfElems( 0 );
    }

    void showMayExAlternatives( int nBag ) {
        m_MayExIndex[nBag].numOfElems( m_NumOfMayExIndex[nBag] );
    }

    void hideMayExAlternatives( bool bUnderControl ) {
        if( bUnderControl )
            hideMayExAlternatives( 0 );
        else
            hideMayExAlternatives( 1 );
    }

    void showMayExAlternatives( bool bUnderControl ) {
        if( bUnderControl )
            showMayExAlternatives( 0 );
        else
            showMayExAlternatives( 1 );
    }

        // Prepare retrieval of specific alternative.
        // Returns true if alternative is original,
        // otherwise false. selectAlternative() is
        // slightly faster and should be used when
        // the signature is not needed (it can be
        // completed with a call to the second
        // version of considerAlternative(), which
        // assumes that a selection has already been made.

    bool considerAlternative( int nAlternative0, int nAlternative1 );
    void considerAlternative();

    bool selectAlternative( int nAlternative0, int nAlternative1 );

    int alternative0() const {
        return m_nAlternative0;
    }

    int alternative1() const {
        return m_nAlternative1;
    }

    const tSignature& altSignature() const {
        return m_AltSignature;
    }

        // Number of instruments exercised in the alternative.

    int altNumOfExIndexes() const {
        return m_nAltNumOfExIndexes;
    }

    int altExIndex( int nPos ) const {
        if( nPos < m_ForceExIndex.numOfElems() ) 
            return m_ForceExIndex[nPos];
        return m_AltMayExIndex[nPos - m_ForceExIndex.numOfElems()];
    }

        // Operations for refinement.

    void getForceExNeg( tSignature& Sig ) const;

    bool jumpForceExXor( const tSigBag& Adj, tSignature& Sig ) const;
    bool jumpForceExAnd( const tSigBag& Adj, tSignature& Sig ) const;

    void reduceMayEx( tSignature& Sig ) const;

    void extendMayEx( const tSignature& Sig,
        const tHeap<bool>& UnderControl );

    void extendForceExNeg( const tSignature* pSig,
        const tHeap<bool>& UnderControl );
};

MTG_END_NAMESPACE

#endif
