// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#if ! defined(_MTG_SLOT_Layer_)
#define _MTG_SLOT_Layer_

#include "MtgMultiArray.h"
#include "MtgPortfolio.h"
#include "MtgProfile.h"
#include "MtgSignature.h"

MTG_BEGIN_NAMESPACE

// A "slot layer" is a collection of slots for the rollback
// of a particular subset of the instruments in the portfolio.
// Slot layers are constructed after the grid; the number of levels
// is thus already known. A slot layer corresponds to a signature
// (see MtgSignature.h) - exactly the instruments that are switched
// on in the signature are included in the slot layer.

// A slot layer also contains a number of helper arrays.


//
//   t S l o t L a y e r
//

class tSlotLayer : public tWithProfile {

public:

        // In order to loop through the current slots, 
        // an iterator must be used.

    class tIter;
    friend class tIter;

private:

        // Accesses private data in the portfolio.

    typedef tPortfolio::tWrapper tWrapper;

    struct tSeq {
        tWrapper *m_pByMaturity;
        tWrapper *m_pByBirth;
    };

        // A slot can be reused many times.

    struct tSlot {
        bool m_bIsActive;
        tWrapper *m_pWrapper;
    };

        // The sequence of instruments that belong to this slot set
        // is stored here, ordered by maturity and birth date,
        // respectively.

    tHeap<tSeq> m_Seq;

        // The slots themselves are described here. A slot can be active
        // or inactive, and there's a reference to the instrument which
        // it is assigned to.

    tHeap<tSlot> m_Slot;

        // We store only as many slots as are needed (so for a signature
        // that contains only one instrument, only one slot is allocated,
        // and this slot has position 0, naturally). However, since in
        // the portfolio, a (global) slot assignment of slots to instruments
        // has already been done, a mapping from the global slot number
        // to the local one is required. This is done in m_aXlatSlot.
        // Similarly, a translation from the claim index in the portfolio
        // to the local slot number is provided in m_XlatIndex.

    tHeap<int> m_XlatSlot;
    tHeap<int> m_XlatIndex;

        // There are two instances of value arrays, one acting
        // as source (last) and one acting as target (current)
        // at each rollback step.

    tMultiArray<double> m_Buffer[2];

        // If m_Prep is initialized (it needn't be), it can be used
        // in a preparation step for the current rollback (i.e,
        // store the values *before* early exercise and
        // other adjustments).

    tMultiArray<double> m_Prep;
    
        // Finally, there's two temporary instances which can be used
        // for anything else. There access is rather limited.

    tMultiArray<double> m_Temp1;
    tMultiArray<double> m_Temp2;

        // The following array serves an entirely different
        // purpose. It can be used to record the chain of
        // slot layers through which certain values are 
        // passed.

    tMultiArray<const tSlotLayer*> m_Chain;

    bool m_bWithPrep;
    bool m_bWithTemp1;
    bool m_bWithTemp2;
    bool m_bWithChain;

        // The slot set is fabricated from this information:

    tArrayBounds m_Bounds;
    tHeap<tWrapper*>* m_pWrapper;        // referenced
    tSignature* m_pSignature;            // owned

    bool m_bIsFinalized;

        // Slot management. The slot set offers services to activate
        // or deactivate instruments for the current slice of the grid.

    int m_nFirstActiveSlot;
    int m_nLastActiveSlot;

    int m_nNextByMaturity;
    int m_nNextByBirth;

        // Where are the individual claim positions?

    int m_nClaimPos;

        // Where in a row is the total stored?

    int m_nTotalPos;

        // Which is the current, which the last?

    int m_nCurrent;
    int m_nLast;

        // Some helper functions.

    void init();
    
    void sortSeq();

    void setValueAndTotal( double* Row, int nIndex,
        double gValue, double gMultiplier ) {
        Row[m_nTotalPos] += gValue * gMultiplier;
        Row[m_XlatIndex[nIndex]] = gValue;
    }

    void replValueAndTotal( double* Row, int nIndex,
        double gValue, double gMultiplier ) {
        int k = m_XlatIndex[nIndex];
        Row[m_nTotalPos] += ( gValue - Row[k] ) * gMultiplier;
        Row[k] = gValue;
    }

public:

    tSlotLayer();
    tSlotLayer( const tSlotLayer& Layer );

    ~tSlotLayer();

    void reset();

    tSlotLayer& operator=( const tSlotLayer& Layer );

    void copyFrom( const tSlotLayer& Layer );

    tSlotLayer* clone() const;

    void setBounds( const tArrayBounds& Bounds );
    void setBounds( const tArrayBounds* pBounds );

        // Instead of Layer bounds, one can also define
        // a one-dimensional Layer directly:

    void setNumOfLevels( int nNumOfUpLevels, int nNumOfDownLevels );

        // Takes the non-auxiliaries:

    void setPortfolio( tPortfolio& Pf );
    void setPortfolio( tPortfolio* pPf );

        // Takes any subset:

    void setPortfolio( tHeap<tWrapper*>& Wrapper );
    void setPortfolio( tHeap<tWrapper*>* pWrapper );

        // If no signature is set, all claims are taken:

    void setSignature( const tSignature& Sig );
    void setSignature( const tSignature* pSig );

        // Note that the following four functions may also be
        // called after finalization.

    void setWithPrep( bool bWithPrep = true );
    void setWithTemp1( bool bWithTemp1 = true );
    void setWithTemp2( bool bWithTemp2 = true );
    void setWithChain( bool bWithChain = true );

    bool withPrep() const {
        return m_bWithPrep; 
    }

    bool withTemp1() const {
        return m_bWithTemp1;
    }

    bool withTemp2() const {
        return m_bWithTemp2;
    }

    bool withChain() const {
        return m_bWithChain;
    }

    tRetCode finalize();

        // This function tests whether any claims covered
        // by the layer are monitored, i.e. American.

    bool hasMonitoredClaims() const;

        // The following functions prepare the slot set for the
        // numerical computation, activate new instruments before
        // the slices that belong to a particular day are rolled
        // back, and deactivate obsolete instruments after
        // the slices of a particular day have been rolled back.
        
    void prepareRollback();
    void beforeRollback( int nDay );
    void afterRollback( int nDay );

        // Exchange current and last.

    void rotate();

        // Sets all values in all buffers to zero.

    void zero();

        // Sets all chain pointers to "self."

    void selfChain();

    const tSignature* signature() const {
        return m_pSignature;
    }

    const int tag() const {
        return m_pSignature == 0 ? 0 : m_pSignature->tag();
    }

    bool wrapperMatches( tWrapper* pWrapper ) const {
        return ( pWrapper == m_Slot[pWrapper->m_nSlot].m_pWrapper );
    }

    int dimension() const {
        return m_Buffer[m_nCurrent].dimension() - 1;
    }

    int rowSize() const {
        return m_Bounds.rowSize();
    }

    size_t rowBytes() const {
        return (size_t) rowSize() * sizeof(double);
    }

    tMultiArray<double>& current() {
        return m_Buffer[m_nCurrent];
    }

    tMultiArray<double>& last() {
        return m_Buffer[m_nLast];
    }

        // Since slots are compressed in the last dimension, slot
        // numbers have to be translated to array indeces.

    int xlatSlot( int nSlot ) const {
        return m_XlatSlot[nSlot];
    }

    int xlatIndex( int nIndex ) const {
        return m_XlatIndex[nIndex];
    }

    void curClearTotal();
    void prepClearTotal();

    void curCopyFrom( const tSlotLayer& Layer, int x1 );
    void curCopyFrom( const tSlotLayer& Layer, int x1, int x2 );
    void curCopyFrom( const tSlotLayer& Layer, int x1, int x2, int x3 );
    void curCopyFrom( const tSlotLayer& Layer, tHeap<int>& x );
    void curCopyFrom( const tSlotLayer& Layer, int x[] );

        // Move the entire curent layer to m_Prep.

    void copyCurToPrep();

        // Move the entire layer m_Prep to m_Temp1.

    void savePrep();
    void restorePrep();

        // Move an entire row of m_Prep to m_Temp1.

    void savePrep( int x1 );
    void savePrep( int x1, int x2 );
    void savePrep( int x1, int x2, int x3 );
    void savePrep( tHeap<int>& x );
    void savePrep( int x[] );

        // Move an entire row of m_Temp1 to m_Prep.

    void restorePrep( int x1 );
    void restorePrep( int x1, int x2 );
    void restorePrep( int x1, int x2, int x3 );
    void restorePrep( tHeap<int>& x );
    void restorePrep( int x[] );

        // Copy entire layer:

    void lastCopyFrom( const tSlotLayer& Layer );

        // Some common operations. The last coordinate is interpreted
        // as index!

    void setCurValueAndTotal( int x1, int x2,
        double gValue, double gMultiplier ) {
        setValueAndTotal( curRow( x1 ), x2, gValue, gMultiplier );
    }

    void replCurValueAndTotal( int x1, int x2,
        double gValue, double gMultiplier ) {
        replValueAndTotal( curRow( x1 ), x2, gValue, gMultiplier );
    }

    void setPrepValueAndTotal( int x1, int x2,
        double gValue, double gMultiplier ) {
        setValueAndTotal( prepRow( x1 ), x2, gValue, gMultiplier );
    }

    void replPrepValueAndTotal( int x1, int x2,
        double gValue, double gMultiplier ) {
        replValueAndTotal( prepRow( x1 ), x2, gValue, gMultiplier );
    }

        // Get individual values. The last coordinate is
        // interpreted as index (!!) and automatically translated.

    double& curValue( int x1 ) {
        return current()( m_XlatIndex[x1] );
    }

    double& curValue( int x1, int x2 ) {
        return current()( x1, m_XlatIndex[x2] );
    }

    double& curValue( int x1, int x2, int x3 ) {
        return current()( x1, x2, m_XlatIndex[x3] );
    }

    double& curValue( const tHeap<int>& x ) {
        return current().row( x )[m_XlatIndex[x[m_nClaimPos]]];
    }

    double& curValue( const int x[] ) {
        return current().row( x )[m_XlatIndex[x[m_nClaimPos]]];
    }

    double& curValue( const tHeap<int>& x1, int x2 ) {
        return current().row( x1 )[m_XlatIndex[x2]];
    }

    double& curValue( const int x1[], int x2 ) {
        return current().row( x1 )[m_XlatIndex[x2]];
    }

    double& lastValue( int x1 ) {
        return last()( m_XlatIndex[x1] );
    }

    double& lastValue( int x1, int x2 ) {
        return last()( x1, m_XlatIndex[x2] );
    }

    double& lastValue( int x1, int x2, int x3 ) {
        return last()( x1, x2, m_XlatIndex[x3] );
    }

    double& lastValue( const tHeap<int>& x ) {
        return last().row( x )[m_XlatIndex[x[m_nClaimPos]]];
    }

    double& lastValue( const int x[] ) {
        return last().row( x )[m_XlatIndex[x[m_nClaimPos]]];
    }

    double& lastValue( const tHeap<int>& x1, int x2 ) {
        return last().row( x1 )[m_XlatIndex[x2]];
    }

    double& lastValue( const int x1[], int x2 ) {
        return last().row( x1 )[m_XlatIndex[x2]];
    }

    double& prepValue( int x1 ) {
        return m_Prep( m_XlatIndex[x1] );
    }

    double& prepValue( int x1, int x2 ) {
        return m_Prep( x1, m_XlatIndex[x2] );
    }

    double& prepValue( int x1, int x2, int x3 ) {
        return m_Prep( x1, x2, m_XlatIndex[x3] );
    }

    double& prepValue( const tHeap<int>& x ) {
        return m_Prep.row( x )[m_XlatIndex[x[m_nClaimPos]]];
    }

    double& prepValue( const int x[] ) {
        return m_Prep.row( x )[m_XlatIndex[x[m_nClaimPos]]];
    }

    double& prepValue( const tHeap<int>& x1, int x2 ) {
        return m_Prep.row( x1 )[m_XlatIndex[x2]];
    }

    double& prepValue( const int x1[], int x2 ) {
        return m_Prep.row( x1 )[m_XlatIndex[x2]];
    }

    double& temp2Value( int x1 ) {
        return m_Temp2( m_XlatIndex[x1] );
    }

    double& temp2Value( int x1, int x2 ) {
        return m_Temp2( x1, m_XlatIndex[x2] );
    }

    double& temp2Value( int x1, int x2, int x3 ) {
        return m_Temp2( x1, x2, m_XlatIndex[x3] );
    }

    double& temp2Value( const tHeap<int>& x ) {
        return m_Temp2.row( x )[m_XlatIndex[x[m_nClaimPos]]];
    }

    double& temp2Value( const int x[] ) {
        return m_Temp2.row( x )[m_XlatIndex[x[m_nClaimPos]]];
    }

    double& temp2Value( const tHeap<int>& x1, int x2 ) {
        return m_Temp2.row( x1 )[m_XlatIndex[x2]];
    }

    double& temp2Value( const int x1[], int x2 ) {
        return m_Temp2.row( x1 )[m_XlatIndex[x2]];
    }

        // Each row has a total somewhere, which can be
        // accessed separately:

    double& curTotal( int x1 ) {
        return current()( x1, m_nTotalPos );
    }

    double& curTotal( int x1, int x2 ) {
        return current()( x1, x2, m_nTotalPos );
    }

    double& curTotal( int x1, int x2, int x3 ) {
        return current()( x1, x2, x3, m_nTotalPos );
    }

    double& curTotal( const tHeap<int>& x ) {
        return current().row( x )[m_nTotalPos];
    }

    double& curTotal( const int x[] ) {
        return current().row( x )[m_nTotalPos];
    }

    double& lastTotal( int x1 ) {
        return last()( x1, m_nTotalPos );
    }

    double& lastTotal( int x1, int x2 ) {
        return last()( x1, x2, m_nTotalPos );
    }

    double& lastTotal( int x1, int x2, int x3 ) {
        return last()( x1, x2, x3, m_nTotalPos );
    }

    double& lastTotal( tHeap<int>& x ) {
        return last().row( x )[m_nTotalPos];
    }

    double& lastTotal( int x[] ) {
        return last().row( x )[m_nTotalPos];
    }

    double& prepTotal( int x1 ) {
        return m_Prep( x1, m_nTotalPos );
    }

    double& prepTotal( int x1, int x2 ) {
        return m_Prep( x1, x2, m_nTotalPos );
    }

    double& prepTotal( int x1, int x2, int x3 ) {
        return m_Prep( x1, x2, x3, m_nTotalPos );
    }

    double& prepTotal( const tHeap<int>& x ) {
        return m_Prep.row( x )[m_nTotalPos];
    }

    double& prepTotal( const int x[] ) {
        return m_Prep.row( x )[m_nTotalPos];
    }

        // Get the row itself.
        // REMEMBER: when accessing components in the
        // tow, translation is necessary!

    double* curRow( int x1 ) {
        return current().row( x1 );
    }

    double* curRow( int x1, int x2 ) {
        return current().row( x1, x2 );
    }

    double* curRow( int x1, int x2, int x3 ) {
        return current().row( x1, x2, x3 );
    }

    double* curRow( const tHeap<int>& x ) {
        return current().row( x );
    }

    double* curRow( const int x[] ) {
        return current().row( x );
    }

    double* lastRow( int x1 ) {
        return last().row( x1 );
    }

    double* lastRow( int x1, int x2 ) {
        return last().row( x1, x2 );
    }

    double* lastRow( int x1, int x2, int x3 ) {
        return last().row( x1, x2, x3 );
    }

    double* lastRow( const tHeap<int>& x ) {
        return last().row( x );
    }

    double* lastRow( const int x[] ) {
        return last().row( x );
    }

    double* prepRow( int x1 ) {
        return m_Prep.row( x1 );
    }

    double* prepRow( int x1, int x2 ) {
        return m_Prep.row( x1, x2 );
    }

    double* prepRow( int x1, int x2, int x3 ) {
        return m_Prep.row( x1, x2, x3 );
    }

    double* prepRow( const tHeap<int>& x ) {
        return m_Prep.row( x );
    }

    double* prepRow( const int x[] ) {
        return m_Prep.row( x );
    }

    double* temp2Row( int x1 ) {
        return m_Temp2.row( x1 );
    }

    double* temp2Row( int x1, int x2 ) {
        return m_Temp2.row( x1, x2 );
    }

    double* temp2Row( int x1, int x2, int x3 ) {
        return m_Temp2.row( x1, x2, x3 );
    }

    double* temp2Row( const tHeap<int>& x ) {
        return m_Temp2.row( x );
    }

    double* temp2Row( const int x[] ) {
        return m_Temp2.row( x );
    }

        // The following functions access the slot layer
        // chain for each value.

    const tSlotLayer*& chain( int x1 ) {
        return m_Chain( x1 );
    }

    const tSlotLayer*& chain( int x1, int x2 ) {
        return m_Chain( x1, x2 );
    }

    const tSlotLayer*& chain( int x1, int x2, int x3 ) {
        return m_Chain( x1, x2, x3 );
    }

    const tSlotLayer*& chain( const tHeap<int>& x ) {
        return m_Chain( x );
    }

    const tSlotLayer*& chain( const int x[] ) {
        return m_Chain( x );
    }
};


//
//   t I t e r
//

class tSlotLayer::tIter {

    int m_nFirstActiveSlot;
    int m_nLastActiveSlot;
    tHeap<tSlot>& m_Slot;

    int m_nPos;

public:

        // Everthing is inline, for speed.

    tIter( tSlotLayer& Layer ) : m_Slot( Layer.m_Slot ) {
        m_nFirstActiveSlot = Layer.m_nFirstActiveSlot;
        m_nLastActiveSlot = Layer.m_nLastActiveSlot;
        reset();
    }

    void reset() {
        m_nPos = m_nFirstActiveSlot - 1;
        operator++();
    }

    operator bool() const {
        return m_nPos <= m_nLastActiveSlot;
    }

    operator int() const {
        return m_nPos <= m_nLastActiveSlot;
    }

    void operator++() {
        while( ++m_nPos <= m_nLastActiveSlot &&
                ! m_Slot[m_nPos].m_bIsActive ) {
            ++m_nPos;
        }
    }

        // The real position in the layer.
        // (Translation is not necessary.)

    int xlat() const {
        return m_nPos;
    }

        // The index in the wrapper array.

    int index() const {
        return m_Slot[m_nPos].m_pWrapper->m_nIndex;
    }

    tClaim& claim() const {
        return *m_Slot[m_nPos].m_pWrapper->m_pClaim;
    }
};

MTG_END_NAMESPACE

#endif
