// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#include "MtgIncl.h"
#include "MtgSource.h"

MTG_BEGIN_NAMESPACE


//
//   r e a d C h a r
//

tRetCode tSource::readChar()

{
    tRetCode nRet;

    if( m_bLookAhead ) {
        m_nCurChar = m_nNextChar;
        m_bLookAhead = false;
    }
    else {
        if( m_nCurChar != EOF ) {
            if( ( nRet = readChar( m_nCurChar ) ) != OK )
                return nRet;
        }
    }

    return OK;
}


//
//   n e x t C h a r
//

tRetCode tSource::nextChar( int& nNextChar )

{
    tRetCode nRet;

    if( ! m_bLookAhead ) {
        if( m_nCurChar != EOF ) {
            if( ( nRet = readChar( m_nNextChar ) ) != OK )
                return nRet;
        }
        else {
            m_nNextChar = EOF;
        }
        m_bLookAhead = true;
    }
    nNextChar = m_nNextChar;
    return OK;
}


//
//   s k i p S p a c e
//

tRetCode tSource::skipSpace()

{
    tRetCode nRet;

    while( isspace( m_nCurChar ) ) {
        if( m_nCurChar == '\n' )
            ++m_nCurLine;
        if( ( nRet = readChar() ) != OK )
            return nRet;
    }

    return OK;
}


//
//   s k i p C o m m e n t
//

tRetCode tSource::skipComment()

{
    tRetCode nRet;

    while( m_nCurChar != '\n' && m_nCurChar != EOF ) {
        if( ( nRet = readChar() ) != OK )
            return nRet;
    }

    if( m_nCurChar == '\n' ) {
        ++m_nCurLine;
        if( ( nRet = readChar() ) != OK )
            return nRet;
    }

    return OK;
}


//
//   r e a d I d
//

tRetCode tSource::readId()

{
    tRetCode nRet;

    int k = 0;
    do{
        if( k < nSourceMaxTokenLength )
            m_sCurText[k++] = (char) m_nCurChar;
        if( ( nRet = readChar() ) != OK )
            return nRet;
    } while( isalnum( m_nCurChar ) || m_nCurChar == '_' );

    m_sCurText[k] = '\0';
    m_nCurToken = Text2Token( m_sCurText );

    if( m_bIsSafe ) {
        if( m_nCurToken == xTokMtg || m_nCurToken == xTokTcl ||
            m_nCurToken == xTokShell || m_nCurToken == xTokRun ||
            m_nCurToken == xTokInclude || m_nCurToken == xTokSave ||
            m_nCurToken == xTokTry || m_nCurToken == xTokVerbose ) {
            m_nCurToken = xTokId;
            return INVALID_KEYWORD;
        }
    }

    return OK;
}


//
//   r e a d N u m b e r
//

tRetCode tSource::readNumber()

{
    char *s;
    int k, nNextChar;
    tRetCode nRet;
    bool bE, bDot, bAllowSign, bBreak;

    k = 0;
    bE = false;
    bDot = false;

    while( true ) {
        if( k >= nSourceMaxTokenLength )
            return INVALID_CONSTANT;

        m_sCurText[k++] = (char) m_nCurChar;
        bAllowSign = false;

        switch( m_nCurChar ) {
            case 'e' :
            case 'E' :
                MTG_ASSERT( ! bE );
                bAllowSign = bE = true;
                break;

            case '.' :
                MTG_ASSERT( ! bDot );
                bDot = true;
                break;
        }

        if( ( nRet = readChar() ) != OK )
            return nRet;

        if( ! isdigit( m_nCurChar ) ) {
            bBreak = false;

        switch( m_nCurChar ) {
                case 'e' :
                case 'E' :
                    if( ( nRet = nextChar( nNextChar ) ) != OK )
                        return nRet;
                    if( bE ||
                        nNextChar != '+' && nNextChar != '-' &&
                            ! isdigit( nNextChar ) ) {
                        bBreak = true;
                    }
                    break;

                case '.' :
                    if( ( nRet = nextChar( nNextChar ) ) != OK )
                        return nRet;
                    if( bE || bDot ||
                        nNextChar != 'e' && nNextChar != 'E' &&
                            ! isdigit( nNextChar ) ) {
                        bBreak = true;
                    }
                    break;

                case '+' :
                case '-' :
                    if( ! bAllowSign )
                        bBreak = true;
                    break;

                default :
                    bBreak = true;
                    break;
            }

            if( bBreak )
                break;
        }
    }

    if( m_sCurText[k - 1] != '.' && ! isdigit( m_sCurText[k - 1] ) )
        return INVALID_CONSTANT;

    m_sCurText[k] = '\0';

    m_gCurValue = strtod( m_sCurText, &s );
    if( m_gCurValue == HUGE_VAL || *s != '\0' )
        return INVALID_CONSTANT;

    m_nCurToken = xTokNumber;
    return OK;
}


//
//   r e a d S t r i n g
//

tRetCode tSource::readString()

{
    char cDelim = m_nCurChar;

    tRetCode nRet;

    int k = 0;

    if( ( nRet = readChar() ) != OK )
        return nRet;

    while( m_nCurChar != cDelim ) {
        if( m_nCurChar == EOF || m_nCurChar == '\n' )
            return INVALID_STRING;

        if( k < sizeof(m_sCurText) - 1 )
            m_sCurText[k++] = (char) m_nCurChar;

        if( ( nRet = readChar() ) != OK )
            return nRet;
    }

    m_sCurText[k] = 0;
    m_nCurToken = xTokString;

    return readChar();
}


//
//   r e a d S p e c i a l
//

tRetCode tSource::readSpecial()

{
    m_sCurText[0] = m_nCurChar;
    m_sCurText[1] = 0;

    switch( m_nCurChar ) {
        case '(' :
            m_nCurToken = xTokLeftParen;  return readChar();
        case ')' :
            m_nCurToken = xTokRightParen; return readChar();
        case '{' :
            m_nCurToken = xTokLeftBrace;  return readChar();
        case '}' :
            m_nCurToken = xTokRightBrace; return readChar();
        case '%' :
            m_nCurToken = xTokPercentage; return readChar();
        case '?' :
            m_nCurToken = xTokQuestion;   return readChar();
        case ',' :
            m_nCurToken = xTokComma;      return readChar();
        case ';' :
            m_nCurToken = xTokSeq;        return readChar();
    }

    tRetCode nRet;
    int nNextChar;

    if( ( nRet = nextChar( nNextChar ) ) != OK )
        return nRet;

    bool bAdvance = false;

    switch( m_nCurChar ) {
        case '+' :
            if( nNextChar == '=' ) {
                m_nCurToken = xTokPlusAssign;
                bAdvance = true;
            }
            else {
                m_nCurToken = xTokPlus;
            }
            break;

        case '-' :
            if( nNextChar == '=' ) {
                m_nCurToken = xTokMinusAssign;
                bAdvance = true;
            }
            else
            if( nNextChar == '>' ) {
                m_nCurToken = xTokReference;
                bAdvance = true;
            }
            else {
                m_nCurToken = xTokMinus;
            }
            break;

        case '*' :
            if( nNextChar == '=' ) {
                m_nCurToken = xTokTimesAssign;
                bAdvance = true;
            }
            else {
                m_nCurToken = xTokTimes;
            }
            break;

        case '/' :
            if( nNextChar == '=' ) {
                m_nCurToken = xTokOverAssign;
                bAdvance = true;
            }
            else {
                m_nCurToken = xTokOver;
            }
            break;

        case '=' :
            if( nNextChar == '=' ) {
                m_nCurToken = xTokEq;
                bAdvance = true;
            }
            else {
                m_nCurToken = xTokAssign;
            }
            break;

        case '!' :
            if( nNextChar == '=' ) {
                m_nCurToken = xTokNe;
                bAdvance = true;
            }
            else {
                m_nCurToken = xTokNot;
            }
            break;

        case '>' :
            if( nNextChar == '=' ) {
                m_nCurToken = xTokGe;
                bAdvance = true;
            }
            else {
                m_nCurToken = xTokGt;
            }
            break;

        case '<' :
            if( nNextChar == '=' ) {
                m_nCurToken = xTokLe;
                bAdvance = true;
            }
            else {
                m_nCurToken = xTokLt;
            }
            break;

        case '&' :
            if( nNextChar == '&' ) {
                m_nCurToken = xTokAnd;
                bAdvance = true;
            }
            else {
                return INVALID_KEYWORD;
            }          
            break;

        case '|' :
            if( nNextChar == '|' ) {
                m_nCurToken = xTokOr;
                bAdvance = true;
            }
            else {
                return INVALID_KEYWORD;
            }
            break;

        case ':' :
            if( nNextChar == ':' ) {
                m_nCurToken = xTokReference;
                bAdvance = true;
            }
            else
            if( nNextChar == '=' ) {
                m_nCurToken = xTokAssign;
                bAdvance = true;
            }
            else {
                m_nCurToken = xTokColon;
            }
            break;

        case '.' :
            if( nNextChar == '.' ) {
                m_nCurToken = xTokRange;
                bAdvance = true;
            }
            else {
                m_nCurToken = xTokReference;
            }
            break;

        default :
            return INVALID_KEYWORD;    
    }

    if( bAdvance ) {
        m_sCurText[1] = nNextChar;
        m_sCurText[2] = 0;

        if( ( nRet = readChar() ) != OK )
            return nRet;
    }

    return readChar();
}


//
//   s t a r t
//

tRetCode tSource::start()

{
    tRetCode nRet;

    m_nCurLine = 0;

    if( ( nRet = open() ) != OK )
        return nRet;

    m_nCurLine = 1;
    m_nCurChar = 0;
    m_bLookAhead = false;

    if( ( nRet = readChar() ) != OK )
        return nRet;

    return readToken();
}


//
//   e n d
//

void tSource::end()

{
    close();
}


//
//   r e a d T o k e n
//

tRetCode tSource::readToken()

{
    tRetCode nRet;

restart:

    if( ( nRet = skipSpace() ) != OK )
        return nRet;

    if( m_nCurChar == EOF ) {
        m_nCurToken = xTokEOF;
        return OK;
    } 

    if( m_nCurChar == '/' ) {
        int nNextChar;

        if( ( nRet = nextChar( nNextChar ) ) != OK )
            return nRet;
        if( nNextChar == '/' ) {
            if( ( nRet = skipComment() ) != OK )
                return nRet;
            goto restart;
        }
    }

    if( m_nCurChar == '\'' || m_nCurChar == '\"' )
        return readString();

    if( isalpha( m_nCurChar ) || m_nCurChar == '_' )
        return readId();

    if( isdigit( m_nCurChar ) )
        return readNumber();

    if( m_nCurChar == '.' ) {
        int nNextChar;

        if( ( nRet = nextChar( nNextChar ) ) != OK )
            return nRet;
        if( isdigit( m_nNextChar ) )
            return readNumber();
    }

    return readSpecial();
}


//
//   t r y D a t e
//

bool tSource::tryDate()

{
    if( m_nCurToken == xTokDate )
        return true;

    if( m_nCurToken != xTokString )
        return false;

    try {
        m_CurDate = m_sCurText;
    }
    catch( tException ) {
        return false;
    }

    m_nCurToken = xTokDate;
    return true;
}


//
//   t r y D a t e
//

bool tSource::tryDate( tDate& Date )

{
    if( ! tryDate() )
        return false;
    Date = m_CurDate;
    return true;
}


//
//   o p e n
//

tRetCode tSource::open()

{
    return OK;
}


//
//   c l o s e
//

void tSource::close()

{
}


//
//   t S o u r c e
//

tSource::tSource()

{
    m_bIsSafe = false;
}


//
//   ~ t S o u r c e
//

tSource::~tSource()

{
    close();
}


//
//   o p e n
//

tRetCode tStringSource::open()

{
    m_nCurPos = 0;
    return OK;
}


//
//   r e a d C h a r
//

tRetCode tStringSource::readChar( int& nNextChar )

{
    if( m_sInput[m_nCurPos] == 0 )
        nNextChar = EOF;
    else
        nNextChar = m_sInput[m_nCurPos++];
    return OK;
}


//
//   t S t r i n g S o u r c e
//

tStringSource::tStringSource()

{
    setInput( "" );
}


//
//   t S t r i n g S o u r c e
//

tStringSource::tStringSource( const char *sInput )

{
    setInput( sInput );
}


//
//   s e t I n p u t
//

void tStringSource::setInput( const char *sInput )

{
    m_sInput = sInput;
    m_nCurPos = 0;
}


//
//   n a m e
//

const char* tFileSource::name() const

{
    if( m_bByName )
        return m_sFileName;
    return "<file>";
}


//
//   o p e n
//

tRetCode tFileSource::open()

{
    if( m_bByName ) {
        if( ( m_pFile = fopen( m_sFileName, "r" ) ) == 0 )
            return OPEN_ERROR;
    }
    return OK;
}


//
//   r e a d C h a r
//

tRetCode tFileSource::readChar( int& nNextChar )

{
    if( m_pFile != 0 )
        nNextChar = fgetc( m_pFile );
    else
        nNextChar = EOF;
    return OK;
}


//
//   c l o s e
//

void tFileSource::close()

{
    if( m_bByName && m_pFile != 0 ) {
        fclose( m_pFile );
        m_pFile = 0;
    }
}


//
//   t F i l e S o u r c e
//

tFileSource::tFileSource()

{
    setInput( (FILE*) 0 );
}


//
//   t F i l e S o u r c e
//

tFileSource::tFileSource( const char *sFileName )

{
    setInput( sFileName );
}


//
//   t F i l e S o u r c e
//

tFileSource::tFileSource( FILE* pFile )

{
    setInput( pFile );
}


//
//   ~ t F i l e S o u r c e
//

tFileSource::~tFileSource()

{
    close();
}


//
//   s e t I n p u t
//

void tFileSource::setInput( const char *sFileName )

{
    int k;

    if( sFileName == 0 ) {
        m_sFileName[0] = 0;
    }
    else {
        k = strlen( sFileName );
        if( k < (int) sizeof(m_sFileName) ) {
            strcpy( m_sFileName, sFileName );
        }
        else {
            memcpy( m_sFileName, sFileName, sizeof(m_sFileName) - 1 );
            m_sFileName[sizeof(m_sFileName) - 1] = 0;
        }
    }

    m_pFile = 0;
    m_bByName = true;
}


//
//   s e t I n p u t
//

void tFileSource::setInput( FILE* pFile )

{
    m_sFileName[0] = 0;
    m_pFile = pFile;
    m_bByName = false;
}

MTG_END_NAMESPACE
