// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#include "MtgIncl.h"
#include "MtgSpaceAxis.h"
#include "MtgArrayBounds.h"
#include "MtgTimeAxis.h"

MTG_BEGIN_NAMESPACE


//
//   i n i t
//

void tSpaceAxis::init()

{
    m_nMethod = xExplicit;
    m_bIsBox = true;
    m_bIsTrimmed = true;
    m_gTrimDev = 3.5;
    m_pTime = 0;
    m_gStableDt = 0;
    m_aExtent.reset();
    touch();
}


//
//   c o p y F r o m
//

void tSpaceAxis::copyFrom( const tSpaceAxis& Axis )

{
    if( this == &Axis )
        return;

    m_nMethod = Axis.m_nMethod;
    m_bIsBox = Axis.m_bIsBox;
    m_bIsTrimmed = Axis.m_bIsTrimmed;
    m_gTrimDev = Axis.m_gTrimDev;
    m_pTime = Axis.m_pTime;
    m_gStableDt = Axis.m_gStableDt;
    m_aExtent = Axis.m_aExtent;
    m_bFinalized = Axis.m_bFinalized;
}


//
//   t S p a c e A x i s
//

tSpaceAxis::tSpaceAxis()

{
    init();
}


//
//   t S p a c e A x i s
//

tSpaceAxis::tSpaceAxis( const tSpaceAxis& Axis )

{
    init();
    copyFrom( Axis );
}


//
//   ~ t S p a c e A x i s
//

tSpaceAxis::~tSpaceAxis()

{
}


//
//   s e t B o x
//

void tSpaceAxis::setBox( bool bOn )

{
    m_bIsBox = bOn;
    touch();
}


//
//   s e t T r i m m i n g
//

void tSpaceAxis::setTrimming( bool bOn )

{
    m_bIsTrimmed = bOn;
    touch();
}


//
//   s e t T r i m D e v
//

void tSpaceAxis::setTrimDev( double gTrimDev )

{
    m_gTrimDev = gTrimDev;
    touch();
}


//
//   s e t T i m e A x i s
//

void tSpaceAxis::setTimeAxis( tTimeAxis& Time )

{
    setTimeAxis( &Time );
}


//
//   s e t T i m e A x i s
//

void tSpaceAxis::setTimeAxis( tTimeAxis* pTime )

{
    m_pTime = pTime;
    touch();
}


//
//   s e t M e t h o d
//

void tSpaceAxis::setMethod( tFDMethod nMethod )

{
    m_nMethod = nMethod;
    touch();
}


//
//   f i n a l i z e
//

void tSpaceAxis::finalize()

{
    MTG_ASSERT( m_pTime != 0 );
    MTG_ASSERT( m_gStableDt > 0 );

    int nNumOfSlices, nNumOfUpLevels, nNumOfDownLevels;

    if( m_bFinalized )
        return;

    m_bFinalized = true;

    m_pTime->finalize();

    nNumOfSlices = m_pTime->numOfSlices();
    m_aExtent.numOfElems( nNumOfSlices );

    if( m_bIsBox ) {
        if( m_bIsTrimmed ) {
            nNumOfUpLevels =
                (int) floor( m_gTrimDev * sqrt( nNumOfSlices - 1 ) );

            if( nNumOfUpLevels >= nNumOfSlices )
                nNumOfUpLevels = nNumOfSlices - 1;
            else
                if( nNumOfUpLevels == 0 )
                    nNumOfUpLevels = 1;
        }
        else {
            nNumOfUpLevels = nNumOfSlices - 1;
        }

        if( isImplicit() && nNumOfUpLevels < 2 ) {
                // want at least 5 levels for implicit methods
            nNumOfUpLevels = 2;
        }
        else {
            if( nNumOfUpLevels == 0 )
                nNumOfUpLevels = 1;
        }

        nNumOfDownLevels = nNumOfUpLevels;

        for( int k = 0; k < nNumOfSlices; ++k ) {
            m_aExtent[k].m_nNumOfUpLevels = nNumOfUpLevels;
            m_aExtent[k].m_nNumOfDownLevels = nNumOfDownLevels;
        }
    }
    else {
        int nAdd = isImplicit() ? 2 : 1;

        if( m_bIsTrimmed ) {
            for( int k = 0; k < nNumOfSlices; ++k ) {
                nNumOfUpLevels = (int) floor( m_gTrimDev * sqrt( k ) );
                if( nNumOfUpLevels > k )
                    nNumOfUpLevels = k;
                else
                    if( k > 0 && nNumOfUpLevels == 0 )
                        nNumOfUpLevels = 1;

                    // add 1 resp. 2, to get 3 resp. 5 at the root

                nNumOfUpLevels += nAdd;
                nNumOfDownLevels = nNumOfUpLevels;

                m_aExtent[k].m_nNumOfUpLevels = nNumOfUpLevels;
                m_aExtent[k].m_nNumOfDownLevels = nNumOfDownLevels;
            }
        }
        else {
                // always one or two more, to get delta and gamma
            for( int k = 0; k < nNumOfSlices; ++k ) {
                m_aExtent[k].m_nNumOfUpLevels = k + nAdd;
                m_aExtent[k].m_nNumOfDownLevels = k + nAdd;
            }
        }
    }

    finalize( m_aExtent.last().m_nNumOfUpLevels + 1,
              m_aExtent.last().m_nNumOfDownLevels + 1 );
}


//
//   a p p e n d
//

void tSpaceAxis::append( tArrayBounds& Bounds ) const

{
    MTG_ASSERT( m_bFinalized );
    Bounds.append( -m_aExtent.last().m_nNumOfDownLevels - 1,
        m_aExtent.last().m_nNumOfUpLevels + 1 );
}

MTG_END_NAMESPACE