// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#if ! defined(_MTG_SPACE_AXIS_)
#define _MTG_SPACE_AXIS_

#include "MtgHeap.h"

MTG_BEGIN_NAMESPACE

class tArrayBounds;
class tOFSolver;
class tTimeAxis;


//
//   t S p a c e A x i s
//

class tSpaceAxis {

    struct tExtent {
        int m_nNumOfUpLevels;
        int m_nNumOfDownLevels;
    };

    tFDMethod m_nMethod;

    bool m_bIsBox;

    bool m_bIsTrimmed;
    double m_gTrimDev;

        // Only used during finalization; may dangle otherwise:

    tTimeAxis* m_pTime;
    tHeap<tExtent> m_aExtent;

    bool m_bFinalized;

    void init();

protected:

    double m_gStableDt; // day-based!

    void touch()             { m_bFinalized = false; }
    bool isFinalized() const { return m_bFinalized; }

    void copyFrom( const tSpaceAxis& Axis );

    virtual void finalize( int nNumOfUpLevels, int nNumOfDownLevels ) = 0;
    
public:

    tSpaceAxis();
    tSpaceAxis( const tSpaceAxis& Axis );

    virtual ~tSpaceAxis();

    virtual tSpaceAxis* clone() const = 0;

    void setBox( bool bOn = true );
    void setTrimming( bool bOn = true );
    void setTrimDev( double gTrimDev = 3.5 );

    void setTimeAxis( tTimeAxis& Time );
    void setTimeAxis( tTimeAxis* pTime = 0 );

    void setMethod( tFDMethod nMethod );

    void finalize();

    void append( tArrayBounds& Bounds ) const;

    double stableDt() const {
        return m_gStableDt;
    }

    int numOfUpLevels() const {
        return m_aExtent.last().m_nNumOfUpLevels;
    }

    int numOfDownLevels() const {
        return m_aExtent.last().m_nNumOfDownLevels;
    }

    int numOfUpLevels( int nSlice ) const {
        return m_aExtent[nSlice].m_nNumOfUpLevels;
    }

    int numOfDownLevels( int nSlice ) const {
        return m_aExtent[nSlice].m_nNumOfDownLevels;
    }

    tFDMethod method() const {
        return m_nMethod;
    }

    bool isExplicit() const {
        return m_nMethod == xExplicit;
    }

    bool isImplicit() const { 
        return m_nMethod == xImplicit;
    }

    virtual double factor( int nLevel ) const = 0;

    virtual tOFSolver* createOFSolver() = 0;

    virtual void calcProb( int nLevel, double gVol, double gDrift,
        double gRelDuration, double& gPU, double& gPD ) const = 0;

    virtual double calcDelta( int nLevel, double gVU, double gVM,
        double gVD ) const = 0;

    virtual double calcGamma( int nLevel, double gVU, double gVM,
        double gVD ) const = 0;
};

MTG_END_NAMESPACE

#endif
