// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#include "MtgIncl.h"
#include "MtgSqTermStruct.h"

MTG_BEGIN_NAMESPACE


//
//   t e s t I m p
//

tRetCode tSqTermStruct::testImp( double gImp ) const

{
    return gImp > 0 ? OK : INVALID_VOLATILITY;
}


//
//   t e s t F w d
//

tRetCode tSqTermStruct::testFwd( double gFwd ) const

{
    return gFwd > 0 ? OK : INVALID_VOLATILITY;
}


//
//   c a l c F w d
//

tRetCode tSqTermStruct::calcFwd( int nMaturity1, double gImp1,
    int nMaturity2, double gImp2, double& gFwd ) const

{
    MTG_ASSERT( nMaturity1 >= 1 && nMaturity2 > nMaturity1 );

    gFwd = ( gImp2 * gImp2 * nMaturity2 -
             gImp1 * gImp1 * nMaturity1 ) / ( nMaturity2 - nMaturity1 );

    if( gFwd < 0 ) {
        gFwd = 0;
        return INVALID_VOLATILITY;
    }

    gFwd = sqrt( gFwd );
    return OK;
}


//
//   t r i m
//

void tSqTermStruct::trim( double& gFwd ) const

{   
    if( gFwd < 0 )
        gFwd = 0;
}


//
//   s c a l e U p
//

double tSqTermStruct::scaleUp( double gFwd ) const

{
    return gFwd * gFwd;
}


//
//   s c a l e D o w n
//

double tSqTermStruct::scaleDown( double gFwd ) const

{
    MTG_ASSERT( gFwd >= 0 );
    return sqrt( gFwd );
}

MTG_END_NAMESPACE

#if defined(_TEST)

#if defined(_WIN32)
    #include <conio.h>
#endif


//
//   m a i n
//

void main( int argc, char *argv[] )

{
    bool bGo;
    int cCmd, nRet, nMaturity, nFromUnit, nToUnit;
    double gPrior, gMin, gMax, gBand;
    tSqTermStruct TS, TS1;

    printf( "\nTest tSqTermStruct\n\n" );

    bGo = true;
    while( bGo ) {
        printf( "<R>eset <I>mp <F>wd <B>and <G>etFwd Get<S>tat E<x>it: " );
        cCmd = getche();
        printf( "\n" );

        nRet = OK;
        switch( cCmd ) {
            case 'r' :
            case 'R' :
                TS.reset();
                break;

            case 'i' :
            case 'I' :
                printf( "  Maturity Prior: " );
                scanf( "%d%lg", &nMaturity, &gPrior );
                nRet = TS.addImplied( nMaturity, gPrior );
                break;

            case 'f' :
            case 'F' :
                printf( "  Maturity Prior: " );
                scanf( "%d%lg", &nMaturity, &gPrior );
                nRet = TS.addForward( nMaturity, gPrior );
                break;

            case 'b' :
            case 'B' :
                printf( "  Band: " );
                scanf( "%lg", &gBand );
                TS.setBand( gBand );
                break;

            case 'g' :
            case 'G' :
                TS1 = TS;
                printf( "  FromUnit ToUnit: " );
                scanf( "%d%d", &nFromUnit, &nToUnit );
                printf( "  Prior=%lg\n", TS1.forward( nFromUnit, nToUnit ) );
                break;

            case 's' :
            case 'S' :
                TS.getFwdRange( gMin, gMax );
                printf( "  Min=%lg Max=%lg\n", gMin, gMax );
                printf( "  ConstantUntil=%d\n", TS.constantUntil() );
                break;

            case 'x' :
            case 'X' :
                bGo = false;
                break;
        }

        if( nRet != OK )
            printf( "  Error=%d\n", nRet );
    }
}

#endif
