// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#include "MtgIncl.h"
#include "MtgStdClaim.h"
#include "MtgFactor.h"
#include "MtgFDEngine.h"
#include "MtgParser.h"

MTG_BEGIN_NAMESPACE


//
//   i n i t
//

void tStdClaim::init()

{
    m_nType = xLinearCall;
    m_bIsAmerican = false;
    m_gStrike = 0;
}


//
//   p a y o f f
//

double tStdClaim::payoff( tEngine& Engine )

{
    MTG_ASSERT( isResolved() );

    tFDEngine* pFD = dynamic_cast<tFDEngine*>( &Engine );

    if( pFD == 0 )
        throw tException( NOT_AVAILABLE );

    double gFactor = pFD->dominant();

    switch( m_nType ) {
        case xLinearCall :
            if( gFactor >= m_gStrike )
                return gFactor - m_gStrike;
            break;

        case xLinearPut :
            if( gFactor <= m_gStrike )
                return m_gStrike - gFactor;
            break;

        case xDigitalCall :
            if( gFactor >= m_gStrike )
                return 1;
            break;

        case xDigitalPut :
            if( gFactor <= m_gStrike )
                return 1;
            break;
    }

    return 0;
}


//
//   m o n i t o r
//

tExPolicy tStdClaim::monitor( tEngine& Engine, double gUnitValue )

{
    MTG_ASSERT( isResolved() && isMonitored() );

    tFDEngine* pFD = dynamic_cast<tFDEngine*>( &Engine );

    if( pFD == 0 )
        throw tException( NOT_AVAILABLE );

    if( m_bIsAmerican )
        return xMayExercise;
    return xDontExercise;
}


//
//   p a r s e P a r a m
//

tRetCode tStdClaim::parseParam( tParser& Parser, tParseInfoStub& Info )

{
    tRetCode nRet;
    tParseInfo& I = static_cast<tParseInfo&>( Info );

    switch( Parser.curToken() ) {
        case xTokStrike :
            if( I.m_bParam1 )
                return Parser.setError( "strike already defined" );
            if( ( nRet = Parser.readToken() ) != OK ||
                ( nRet = Parser.scanNonNegDouble( m_gStrike ) ) != OK ) {
                return nRet;
            }
            I.m_bParam1 = true;
            break;

        case xTokPayoff :
            if( ( nRet = Parser.readToken() ) != OK )
                return nRet;
            if( Parser.curToken() != xTokLinear && 
                Parser.curToken() != xTokDigital ) {
                return Parser.setError( INVALID_KEYWORD );
            }

                // fall through

        case xTokLinear :
        case xTokDigital :
            if( I.m_bParam2 )
                return Parser.setError( "payoff already defined" );
            if( Parser.curToken() == xTokLinear ) {
                switch( m_nType ) {
                    case xDigitalCall : m_nType = xLinearCall; break;
                    case xDigitalPut  : m_nType = xLinearPut;  break;
                    default           : break;
                }
            }
            else {
                switch( m_nType ) {
                    case xLinearCall : m_nType = xDigitalCall; break;
                    case xLinearPut  : m_nType = xDigitalPut;  break;
                    default          : break;
                }
            }
            if( ( nRet = Parser.readToken() ) != OK )
                return nRet;
            I.m_bParam2 = true;
            break;

        default :
            return super::parseParam( Parser, Info );
    }

    return OK;
}


//
//   t S t d C l a i m
//

tStdClaim::tStdClaim()

{
    init();
}


//
//   t S t d C l a i m
//

tStdClaim::tStdClaim( tType nType, bool bIsAmerican )

{
    init();

    m_nType = nType;
    m_bIsAmerican = bIsAmerican;

    if( m_bIsAmerican )
        setMonitor();
}


//   t S t d C l a i m
//

tStdClaim::tStdClaim( const tStdClaim& Claim )

{
    init();
    copyFrom( Claim );
}


//
//   ~ t S t d C l a i m
//

tStdClaim::~tStdClaim()

{
}


//
//   o p e r a t o r =
//

tStdClaim& tStdClaim::operator=( const tStdClaim& Claim )

{
    if( &Claim != this )
        copyFrom( Claim );
    return *this;
}


//
//   c o p y F r o m
//

void tStdClaim::copyFrom( const tStdClaim& Claim )

{
    if( &Claim == this )
        return;

    m_nType = Claim.m_nType;
    m_bIsAmerican = Claim.m_bIsAmerican;
    m_gStrike = Claim.m_gStrike;

    super::copyFrom( Claim );
}


//
//   c l o n e
//

tObject* tStdClaim::clone() const

{
    return new tStdClaim( *this );
}

    
//
//   r e s o l v e
//

tRetCode tStdClaim::resolve( tLookup<int>& Claim,
    tLookup<int>& Factor, tHeap<tFactor*>& FactorPtr, tPortfolio& Pf )

{
    if( isResolved() )
        return OK;
    return super::resolve( Claim, Factor, FactorPtr, Pf );
}


//
//   f i n a l i z e
//

tRetCode tStdClaim::finalize()

{
    if( isFinalized() )
        return OK;
    return super::finalize();
}

MTG_END_NAMESPACE
