// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#include "MtgIncl.h"
#include "MtgStepDrift.h"
#include "MtgBootstrap.h"
#include "MtgCurve.h"
#include "MtgInterestSpline.h"
#include "MtgParser.h"

MTG_BEGIN_NAMESPACE


//
//   c l e a n u p
//

void tStepDrift::cleanup()

{
    for( int i = 0; i < m_AddOther.numOfElems(); ++i ) {
        MTG_ASSERT( m_AddOther[i] != 0 );

        m_AddOther[i]->downRef();
        if( m_AddOther[i]->canDelete() )
            delete m_AddOther[i];
    }

    m_AddOther.reset();
}


//
//   c o p y F r o m
//

void tStepDrift::copyFrom( const tStepDrift& Drift )

{
    if( this == &Drift )
        return;

    cleanup();

    m_TermStruct = Drift.m_TermStruct;

    for( int i = 0; i < Drift.m_AddOther.numOfElems(); ++i ) {
        MTG_ASSERT( Drift.m_AddOther[i] != 0 );

        m_AddOther[i] = Drift.m_AddOther[i];
        m_AddOther[i]->upRef();
    }

    super::copyFrom( Drift );
}


//
//   p a r s e P a r a m
//

tRetCode tStepDrift::parseParam( tParser& Parser, tParseInfoStub& Info )

{
    tRetCode nRet;
    tObject* pObj;

    tParseInfo& I = static_cast<tParseInfo&>( Info );

    switch( Parser.curToken() ) {
        case xTokBootstrap :
        case xTokCurve :
        case xTokDrift :
            if( ( nRet = Parser.readToken() ) != OK )
                return nRet;
            if( Parser.curToken() != xTokId )
                return Parser.setError( INVALID_KEYWORD );
            pObj = Parser.findObject( Parser.curText() );
            if( pObj == 0 )
                return Parser.setError( NOT_FOUND );        
            if( dynamic_cast<tBootstrap*>( pObj ) == 0 &&
                dynamic_cast<tCurve*>( pObj ) == 0 &&
                dynamic_cast<tDrift*>( pObj ) == 0 ) {
                return Parser.setError( OBJECT_MISMATCH );
            }
            if( ( nRet = Parser.readToken() ) != OK )
                return nRet;
            pObj->upRef();
            m_AddOther.append( pObj );
            return OK;

        default :
            break;
    }

    double gPlusBand = -1;
    double gMinusBand = -1;

    nRet = m_TermStruct.parseParam( Parser, gPlusBand, gMinusBand );

    if( nRet == GO )
        nRet = super::parseParam( Parser, Info );

    return nRet;
}


//
//   p a r s e P o s t f i x
//

tRetCode tStepDrift::parsePostfix( tParser& Parser, tParseInfoStub& Info )

{
    tBootstrap* pBoot;
    tCurve* pCurve;
    tDrift* pDrift;
    tRetCode nRet;

    for( int i = 0; i < m_AddOther.numOfElems(); ++i ) {
        if( ( pBoot = dynamic_cast<tBootstrap*>( m_AddOther[i] ) ) != 0 ) {
            if( ( nRet = addBootstrap( *pBoot ) ) != OK )
                return Parser.setError( nRet );
        }
        else
        if( ( pCurve = dynamic_cast<tCurve*>( m_AddOther[i] ) ) != 0 ) {
            if( ( nRet = addCurve( *pCurve ) ) != OK )
                return Parser.setError( nRet );
        }
        else
        if( ( pDrift = dynamic_cast<tDrift*>( m_AddOther[i] ) ) != 0 ) {
            if( ( nRet = addDrift( *pDrift ) ) != OK )
                return Parser.setError( nRet );
        }
        else {
            throw tException( INTERNAL_ERROR );
        }
    }

    if( ( nRet = m_TermStruct.parsePostfix( Parser, -1, -1 ) ) != OK )
        return nRet;
    return super::parsePostfix( Parser, Info );
}


//
//   t S t e p D r i f t
//

tStepDrift::tStepDrift()

{
}


//
//   t S t e p D r i f t
//

tStepDrift::tStepDrift( tScale nScale )
    : tDrift( nScale )

{
}


//
//   t S t e p D r i f t
//

tStepDrift::tStepDrift( tScale nScale, int nNumOfPeriods )
    : tDrift( nScale, nNumOfPeriods )

{
}


//
//   t S t e p D r i f t
//

tStepDrift::tStepDrift( tScale nScale, const tDayCount& DayCount )
    : tDrift( nScale, DayCount )

{
}


//
//   t S t e p D r i f t
//

tStepDrift::tStepDrift( tScale nScale, int nNumOfPeriods,
    const tDayCount& DayCount )
    : tDrift( nScale, nNumOfPeriods, DayCount )

{
}


//
//   t S t e p D r i f t
//

tStepDrift::tStepDrift( tScale nScale, const tDayCount& DayCount,
    tDate Base )
    : tDrift( nScale, DayCount, Base )

{
}


//
//   t S t e p D r i f t
//

tStepDrift::tStepDrift( tScale nScale, int nNumOfPeriods,
    const tDayCount& DayCount, tDate Base )
    : tDrift( nScale, nNumOfPeriods, DayCount, Base )

{
}


//
//   t S t e p D r i f t
//

tStepDrift::tStepDrift( const tStepDrift& Drift )

{
    copyFrom( Drift );
}


//
//   ~ t S t e p D r i f t
//

tStepDrift::~tStepDrift()

{
    cleanup();
}


//
//   o p e r a t o r =
//

tStepDrift& tStepDrift::operator=( const tStepDrift& Drift )

{
    if( this != &Drift )
        copyFrom( Drift );
    return *this;
}


//
//   o p e r a t o r =
//

tStepDrift& tStepDrift::operator=( const tLinTermStruct& TS )

{
    m_TermStruct = TS;
    touch();
    finalize();
    return *this;
}


//
//   c l o n e
//

tObject* tStepDrift::clone() const

{
    return new tStepDrift( *this );
}


//
//   a d d F o r w a r d
//

tRetCode tStepDrift::addForward( int nMaturity, double gFwd )

{
    tRetCode nRet = m_TermStruct.addForward( nMaturity, gFwd );

    if( nRet == OK )
        touch();
    return nRet;
}


//
//   a d d I m p l i e d
//

tRetCode tStepDrift::addImplied( int nMaturity, double gImp )

{
    tRetCode nRet = m_TermStruct.addImplied( nMaturity, gImp );

    if( nRet == OK )
        touch();
    return nRet;
}


//
//   a d d S p l i n e
//

tRetCode tStepDrift::addSpline( tInterestSpline& Spline )

{
        // use our own day count convention (might be   
        // different from the one used in IS!)
    const tDayCount& D = dayCount();

        // we need continuous compounding:
    tInterest I( tInterest::xExponential, tInterest::xYield, D );

        // we're looping through the legs of the spline, which
        // have end dates, and convert those end dates to 
        // subinterval indexes

    tRetCode nRet;
    tDate End;
    double gYield;

    int nSub = 0;               // current subinterval
    double gLastFraction = 0;   // used for carry calculation
    double gCarryYield = 0;

    Spline.getFirstLeg( End, gYield, I );
    do {
        if( End > base() ) {
            if( scale() == xDay ) {
                    // If the scale is in days, the job is easy:
                    // simply add an appropriate forward rate.

                int n = (int) D.numOfDays( base(), End );

                if( n < nSub )
                    continue;   // pathetic!

                if( n > nSub ) {
                    if( ( nRet = addForward( n, gYield ) ) != OK )
                        return nRet;
                    nSub = n;
                }
            }
            else {
                    // If the scale is years, boundaries might
                    // be missed.

                double f = D.fraction( base(), End ) * numOfPeriods();
                int n = (int) floor( f );

                if( n < nSub )
                    continue;   // pathetic!

                    // The previous carry must be worked into
                    // the yield. This will be the quantity y
                    // in both cases.

                if( n > nSub ) {
                    double y =
                        ( ( gLastFraction - nSub ) * gCarryYield +
                          ( nSub + 1 - gLastFraction ) * gYield );

                    if( y != gYield ) {
                        if( ( nRet = addForward( nSub + 1, y ) ) != OK )
                            return nRet;
                    }
                    if( n > nSub + 1 || y == gYield ) {
                        if( ( nRet = addForward( n, gYield ) ) != OK )
                            return nRet;
                    }
                    gCarryYield = gYield;
                    nSub = n;
                }
                else
                if( f > nSub ) {
                    double y =
                        ( ( gLastFraction - nSub ) * gCarryYield +
                          ( f - gLastFraction ) * gYield ) /
                        ( f - nSub );

                    gCarryYield = y;
                }

                gLastFraction = f;
            }
        }
    } while( Spline.getNextLeg( End, gYield, I ) );

    if( scale() == xYear && gLastFraction > nSub ) {
        if( ( nRet = addForward( nSub + 1, gCarryYield ) ) != OK )
            return nRet;
    }

    return nRet;
}


//
//   a d d B o o t s t r a p
//

tRetCode tStepDrift::addBootstrap( tBootstrap& Boot )

{
    return addSpline( Boot.spline() );
}


//
//   a d d C u r v e
//

tRetCode tStepDrift::addCurve( const tCurve& Curve )

{
    return NOT_IMPLEMENTED;
}


//
//   a d d D r i f t
//

tRetCode tStepDrift::addDrift( const tDrift& Drift )

{
    tRetCode nRet;
    int nMaturity = Drift.maturity();

    if( nMaturity == 0 )
        return OK;

    if( sameDateBaseAs( Drift ) ) {
            // copy one-to-one
        for( int i = 0; i < nMaturity; ++i ) {
            if( ( nRet = addForward( i + 1, Drift.forwardSI( i ) ) ) != OK )
                return nRet;
        }
    }
    else {
        double gAlpha, gBeta;
        int nSub = 0;

            // convert via subintervals, without going through 
            // calendar dates

        if( ( nRet = getTranslationSI( Drift, gAlpha, gBeta ) ) != OK )
            return nRet;

            // zero subinterval of Drift in our terms:

        double gZero = -gBeta / gAlpha;

        if( gZero > 0 ) {
                // advance if necessary
            nSub = (int) floor( gZero );

            double gFwd0 = Drift.forwardSI( 0 );
            double gTo = gAlpha * ( nSub + 1 ) + gBeta;

            if( nSub > 0 ) {
                if( ( nRet = addForward( nSub, gFwd0 ) ) != OK )
                    return nRet;
            }

            double gFwd1 =
                ( gZero - nSub ) * gFwd0 +
                ( nSub + 1 - gZero ) * Drift.forwardSI( 0.0, gTo );

            if( ( nRet = addForward( ++nSub, gFwd1 ) ) != OK )
                return nRet;
        }

        while( true ) {
            double gFrom = gAlpha * nSub + gBeta;
            double gTo = gFrom + gAlpha;

            if( gFrom >= Drift.maturity() ) {
                    // totally out of range
                if( ( nRet = addForward( nSub + 1,
                        Drift.forwardSI( Drift.maturity() - 1 )  ) ) != OK ) {
                    return nRet;
                }
                break;
            }

            if( gTo >= Drift.maturity() ) {
                    // cover partial interval
                if( ( nRet = addForward( nSub + 1,
                        Drift.forwardSI( gFrom,
                            (double) Drift.maturity() ) ) ) != OK ) {
                    return nRet;
                }
                    // make sure final rate is correct
                if( ( nRet = addForward( nSub + 2,
                        Drift.forwardSI( Drift.maturity() - 1 )  ) ) != OK ) {
                    return nRet;
                }
                break;
            }

            if( ( nRet = addForward( nSub + 1,
                    Drift.forwardSI( gFrom, gTo )  ) ) != OK ) {
                return nRet;
            }
            
            ++nSub;
        }
    }

    return OK;
}


//
//   o p e r a t o r - =
//

tRetCode tStepDrift::operator-=( const tDrift& Drift )

{
    tRetCode nRet;
    const tStepDrift* p = dynamic_cast<const tStepDrift*>( &Drift );

    if( p == 0 )
        throw tException( INTERNAL_ERROR );
    touch();

    if( ( nRet = ( m_TermStruct -= p->m_TermStruct ) ) != OK )
        return nRet;
    return finalize();
}


//
//   o p e r a t o r ^ =
//

tRetCode tStepDrift::operator^=( const tDrift& Drift )

{
    tRetCode nRet;
    const tStepDrift* p = dynamic_cast<const tStepDrift*>( &Drift );

    if( p == 0 )
        throw tException( INTERNAL_ERROR );
    touch();

    if( ( nRet = ( m_TermStruct ^= p->m_TermStruct ) ) != OK )
        return nRet;
    return finalize();
}


//
//   f i n a l i z e
//

tRetCode tStepDrift::finalize()

{
    tRetCode nRet;

    if( isFinalized() )
        return OK;
    if( ( nRet = m_TermStruct.finalize() ) != OK )
        return nRet;
    setNumOfSamples( m_TermStruct.maturity() );
    return super::finalize();
}

MTG_END_NAMESPACE