// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#include "MtgIncl.h"
#include "MtgTclBondMath.h"

MTG_BEGIN_NAMESPACE

#if defined(_MTG_WITH_TCL)


//
//   g e t D a t e
//

bool tTclBondMath::getDate( Tcl_Obj* pObj, tDate& Date )

{
    int nLength;
    char* sString = Tcl_GetStringFromObj( pObj, &nLength );
    const char* sRest = Date.set( sString );

    if( sRest == 0 )
        return false;

    while( isspace( *sRest ) )
        ++sRest;

    return ! *sRest;
}


//
//   g e t A r g s
//

bool tTclBondMath::getArgs( Tcl_Interp *pInterp, int objc,
    Tcl_Obj *CONST objv[], char* sUsage, 
    tDate& Trade, tDate& Maturity, double& gQuantityA, 
    double* pQuantityB, tDayShift*& pShift, tBondMath*& pMath )

{
    int nNextObj = 1;

    pShift = 0;
    pMath = 0;

    int n, m;

    if( ! super::getArgs( pInterp, objc, objv, nNextObj, pShift, pMath ) )
        goto error;

    n = objc - nNextObj;
    m = pQuantityB ? 1 : 0;

    if( n < 2 + m || n > 3 + m ) {
        Tcl_WrongNumArgs( pInterp, 1, objv, sUsage );
        goto error;
    }

    if( n == 3 + m ) {
        if( ! getDate( objv[nNextObj++], Trade ) )
            goto error;
    }
    else {
        Trade = tDate::today();
    }

    if( ! getDate( objv[nNextObj++], Maturity ) )
        goto error;

    if( Tcl_GetDoubleFromObj( pInterp, objv[nNextObj++],
            &gQuantityA ) != TCL_OK ) {
        goto error;
    }

    if( pQuantityB != 0 &&
        Tcl_GetDoubleFromObj( pInterp, objv[nNextObj++],
            pQuantityB ) != TCL_OK ) {
        goto error;
    }

    if( pMath == 0 )
        pMath = &USBondMath;

    return true;

error:

    if( pShift != 0 ) {
        delete pShift;
        pShift = 0;
    }
    pMath = 0;

    return false;
}


//
//   b i l l P r i c e
//

int tTclBondMath::billPrice( ClientData ClientData,
    Tcl_Interp *pInterp, int objc, Tcl_Obj *CONST objv[] )

{
    tDayShift* pShift;
    tBondMath* pMath;
    tDate Trade, Maturity;
    double gDiscount;

    if( ! getArgs( pInterp, objc, objv,
        "?options? ?tradedate? maturity discount",
        Trade, Maturity, gDiscount, 0, pShift, pMath ) ) {
        return TCL_ERROR;
    }

    if( pShift != 0 ) {
        Tcl_SetObjResult( pInterp, Tcl_NewDoubleObj(
            pMath->billPrice( Trade, Maturity, gDiscount, *pShift ) ) );
        delete pShift;
    }
    else {
        Tcl_SetObjResult( pInterp, Tcl_NewDoubleObj(
            pMath->billPrice( Trade, Maturity, gDiscount ) ) );
    }

    return TCL_OK;
}


//
//   b i l l D i s c o u n t
//

int tTclBondMath::billDiscount( ClientData ClientData,
    Tcl_Interp *pInterp, int objc, Tcl_Obj *CONST objv[] )

{
    tDayShift* pShift;
    tBondMath* pMath;
    tDate Trade, Maturity;
    double gPrice;

    if( ! getArgs( pInterp, objc, objv,
        "?options? ?tradedate? maturity price",
        Trade, Maturity, gPrice, 0, pShift, pMath ) ) {
        return TCL_ERROR;
    }

    if( pShift != 0 ) {
        Tcl_SetObjResult( pInterp, Tcl_NewDoubleObj(
            pMath->billDiscount( Trade, Maturity, gPrice, *pShift ) ) );
        delete pShift;
    }
    else {
        Tcl_SetObjResult( pInterp, Tcl_NewDoubleObj(
            pMath->billDiscount( Trade, Maturity, gPrice ) ) );
    }

    return TCL_OK;
}


//
//   b i l l M M Y i e l d
//

int tTclBondMath::billMMYield( ClientData ClientData,
    Tcl_Interp *pInterp, int objc, Tcl_Obj *CONST objv[] )

{
    tDayShift* pShift;
    tBondMath* pMath;
    tDate Trade, Maturity;
    double gPrice;

    if( ! getArgs( pInterp, objc, objv,
        "?options? ?tradedate? maturity price",
        Trade, Maturity, gPrice, 0, pShift, pMath ) ) {
        return TCL_ERROR;
    }

    if( pShift != 0 ) {
        Tcl_SetObjResult( pInterp, Tcl_NewDoubleObj(
            pMath->billMMYield( Trade, Maturity, gPrice, *pShift ) ) );
        delete pShift;
    }
    else {
        Tcl_SetObjResult( pInterp, Tcl_NewDoubleObj(
            pMath->billMMYield( Trade, Maturity, gPrice ) ) );
    }

    return TCL_OK;
}


//
//   b i l l B E Y i e l d
//

int tTclBondMath::billBEYield( ClientData ClientData,
    Tcl_Interp *pInterp, int objc, Tcl_Obj *CONST objv[] )

{
    tDayShift* pShift;
    tBondMath* pMath;
    tDate Trade, Maturity;
    double gPrice;

    if( ! getArgs( pInterp, objc, objv,
        "?options? ?tradedate? maturity price",
        Trade, Maturity, gPrice, 0, pShift, pMath ) ) {
        return TCL_ERROR;
    }

    if( pShift != 0 ) {
        Tcl_SetObjResult( pInterp, Tcl_NewDoubleObj(
            pMath->billBEYield( Trade, Maturity, gPrice, *pShift ) ) );
        delete pShift;
    }
    else {
        Tcl_SetObjResult( pInterp, Tcl_NewDoubleObj(
            pMath->billBEYield( Trade, Maturity, gPrice ) ) );
    }

    return TCL_OK;
}


//
//   b o n d D i r t y P r i c e
//

int tTclBondMath::bondDirtyPrice( ClientData ClientData,
    Tcl_Interp *pInterp, int objc, Tcl_Obj *CONST objv[] )

{
    tDayShift* pShift;
    tBondMath* pMath;
    tDate Trade, Maturity;
    double gCoupon, gYield;

    if( ! getArgs( pInterp, objc, objv,
        "?options? ?tradedate? maturity coupon yield",
        Trade, Maturity, gCoupon, &gYield, pShift, pMath ) ) {
        return TCL_ERROR;
    }

    if( pShift != 0 ) {
        Tcl_SetObjResult( pInterp, Tcl_NewDoubleObj(
            pMath->bondDirtyPrice( Trade, Maturity,
                gCoupon, gYield, *pShift ) ) );
        delete pShift;
    }
    else {
        Tcl_SetObjResult( pInterp, Tcl_NewDoubleObj(
            pMath->bondDirtyPrice( Trade, Maturity,
                gCoupon, gYield ) ) );
    }

    return TCL_OK;
}


//
//   b o n d C l e a n P r i c e
//

int tTclBondMath::bondCleanPrice( ClientData ClientData,
    Tcl_Interp *pInterp, int objc, Tcl_Obj *CONST objv[] )

{
    tDayShift* pShift;
    tBondMath* pMath;
    tDate Trade, Maturity;
    double gCoupon, gYield;

    if( ! getArgs( pInterp, objc, objv,
        "?options? ?tradedate? maturity coupon yield",
        Trade, Maturity, gCoupon, &gYield, pShift, pMath ) ) {
        return TCL_ERROR;
    }

    if( pShift != 0 ) {
        Tcl_SetObjResult( pInterp, Tcl_NewDoubleObj(
            pMath->bondCleanPrice( Trade, Maturity,
                gCoupon, gYield, *pShift ) ) );
        delete pShift;
    }
    else {
        Tcl_SetObjResult( pInterp, Tcl_NewDoubleObj(
            pMath->bondCleanPrice( Trade, Maturity,
                gCoupon, gYield ) ) );
    }

    return TCL_OK;
}


//
//   b o n d D i r t y 2 Y i e l d
//

int tTclBondMath::bondDirty2Yield( ClientData ClientData,
    Tcl_Interp *pInterp, int objc, Tcl_Obj *CONST objv[] )

{
    tDayShift* pShift;
    tBondMath* pMath;
    tDate Trade, Maturity;
    double gCoupon, gPrice;

    if( ! getArgs( pInterp, objc, objv,
        "?options? ?tradedate? maturity coupon price",
        Trade, Maturity, gCoupon, &gPrice, pShift, pMath ) ) {
        return TCL_ERROR;
    }

    if( pShift != 0 ) {
        Tcl_SetObjResult( pInterp, Tcl_NewDoubleObj(
            pMath->bondDirty2Yield( Trade, Maturity,
                gCoupon, gPrice, *pShift ) ) );
        delete pShift;
    }
    else {
        Tcl_SetObjResult( pInterp, Tcl_NewDoubleObj(
            pMath->bondDirty2Yield( Trade, Maturity,
                gCoupon, gPrice ) ) );
    }

    return TCL_OK;
}


//
//   b o n d C l e a n 2 Y i e l d
//

int tTclBondMath::bondClean2Yield( ClientData ClientData,
    Tcl_Interp *pInterp, int objc, Tcl_Obj *CONST objv[] )

{
    tDayShift* pShift;
    tBondMath* pMath;
    tDate Trade, Maturity;
    double gCoupon, gPrice;

    if( ! getArgs( pInterp, objc, objv,
        "?options? ?tradedate? maturity coupon price",
        Trade, Maturity, gCoupon, &gPrice, pShift, pMath ) ) {
        return TCL_ERROR;
    }

    if( pShift != 0 ) {
        Tcl_SetObjResult( pInterp, Tcl_NewDoubleObj(
            pMath->bondClean2Yield( Trade, Maturity,
                gCoupon, gPrice, *pShift ) ) );
        delete pShift;
    }
    else {
        Tcl_SetObjResult( pInterp, Tcl_NewDoubleObj(
            pMath->bondClean2Yield( Trade, Maturity,
                gCoupon, gPrice ) ) );
    }

    return TCL_OK;
}


//
//   c r e a t e E x t e n s i o n
//

tRetCode tTclBondMath::createExtension( tTclKernel& Kernel )

{
    #define MTG_CREATE_CMD( f ) \
        Tcl_CreateObjCommand( Kernel.interp(), "Mtg::" #f, f, 0, 0 );

    MTG_CREATE_CMD( billPrice )
    MTG_CREATE_CMD( billDiscount )
    MTG_CREATE_CMD( billMMYield )
    MTG_CREATE_CMD( billBEYield )

    MTG_CREATE_CMD( bondDirtyPrice )
    MTG_CREATE_CMD( bondCleanPrice )
    MTG_CREATE_CMD( bondDirty2Yield )
    MTG_CREATE_CMD( bondClean2Yield )

    #undef MTG_CREATE_CMD

    return OK;
}

#endif

MTG_END_NAMESPACE
