// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#include "MtgIncl.h"
#include "MtgTclCgi.h"
#include "MtgDateFormat.h"

MTG_BEGIN_NAMESPACE

#if defined(_MTG_WITH_TCL) && defined(_MTG_WITH_CGI)


//
//   g e t O p t i o n s
//

bool tTclCgi::getOptions( Tcl_Interp* pInterp, int objc,
    Tcl_Obj *CONST objv[], int& nNextObj, tVarType& nVarType,
    Tcl_Obj*& pDefault, Tcl_Obj*& pChoices )

{
    static struct {
        const char* m_sName;
        int m_nId;
        tVarType m_nVarType;
    } KeyWord[] = {
        { "-text", 0, xTextVar },
        { "-textarea", 0, xTextVar },
        { "-string", 0, xTextVar },
        { "-date", 0, xDateVar },
        { "-int", 0, xIntVar },
        { "-posint", 0, xPosIntVar },
        { "-nonnegint", 0, xNonNegIntVar },
        { "-double", 0, xDoubleVar },
        { "-posdouble", 0, xPosDoubleVar },
        { "-nonnegdouble", 0, xNonNegDoubleVar },
        { "-default", 1, xTextVar },
        { "-choices", 2, xTextVar },
    };

    nVarType = xTextVar;
    pDefault = 0;
    pChoices = 0;

    bool bHasType = false;

    while( nNextObj < objc ) {
        int nLength;
        char* sString = Tcl_GetStringFromObj( objv[nNextObj], &nLength );

        if( nLength == 0 || *sString != '-' )
            break;

        char* s = StrToLower( sString );
        int k = 0;

        for( k = 0; k < sizeof(KeyWord) / sizeof(KeyWord[0]); ++k ) {
            if( strcmp( s, KeyWord[k].m_sName ) == 0 )
                break;
        }
        delete s;

        if( k == sizeof(KeyWord) / sizeof(KeyWord[0]) )
            break;

        switch( KeyWord[k].m_nId ) {
            case 0 :
                if( bHasType )
                    return false;
                nVarType = KeyWord[k].m_nVarType;
                bHasType = true;
                break;

            case 1 :
                if( pDefault != 0 || nNextObj + 1 == objc )
                    return false;
                pDefault = objv[++nNextObj];
                break;

            case 2 :
                if( pChoices != 0 || nNextObj + 1 == objc )
                    return false;
                pChoices = objv[++nNextObj];
                break;

            default :
                throw tException( INTERNAL_ERROR );
        }
       
        ++nNextObj;
    }

    return true;
}


//
//   c g i G e t T e x t V a r
//

tRetCode tTclCgi::cgiGetTextVar( Tcl_Interp* pInterp, tVarType nVarType,
    char* sCgiVar, Tcl_Obj* pChoices, Tcl_Obj*& pResult )

{
    MTG_ASSERT( nVarType == xTextVar );

    tRetCode nRet;
    char* sValue;

    if( ( nRet = getVar( sCgiVar, sValue ) ) != OK )
        return nRet;

    if( pChoices == 0 ) {
        pResult = Tcl_NewStringObj( sValue, -1 );
        delete sValue;
    }
    else {
        int argc;
        Tcl_Obj **argv;

        if( Tcl_ListObjGetElements( pInterp, pChoices,
                &argc, &argv ) != TCL_OK ) {
            delete sValue;
            return SCRIPT_ERROR;
        }

        int k;
        for( k = 0; k < argc; ++k ) {
            int nLength;
            char* sString = Tcl_GetStringFromObj( argv[k], &nLength );
                    
            if( strcmp( sValue, sString ) == 0 )
                break;
        }

        Tcl_Free( (char*) argv );
        delete sValue;

        if( k == argc )
            return OUT_OF_RANGE;

        pResult = Tcl_NewIntObj( k );
    }

    return OK;
}


//
//   c g i G e t D a t e V a r
//

tRetCode tTclCgi::cgiGetDateVar( Tcl_Interp* pInterp, tVarType nVarType,
    char* sCgiVar, Tcl_Obj* pChoices, Tcl_Obj*& pResult )

{
    MTG_ASSERT( nVarType == xDateVar );

    tRetCode nRet;
    tDate Value;

    if( ( nRet = getVar( sCgiVar, Value ) ) != OK )
        return nRet;

    if( pChoices == 0 ) {
        tDateFormat Format( "%m/%d/%Y", Value );
        pResult = Tcl_NewStringObj( const_cast<char*>( Format() ), -1 );
    }
    else {
        int argc;
        Tcl_Obj **argv;

        if( Tcl_ListObjGetElements( pInterp, pChoices,
                &argc, &argv ) != TCL_OK ) {
            return SCRIPT_ERROR;
        }

        int k;
        for( k = 0; k < argc; ++k ) {
            int nLength;
            char* sString = Tcl_GetStringFromObj( argv[k], &nLength );

            tDate Choice;

            try {
                Choice = sString;
            }
            catch( tException ) {
                return SCRIPT_ERROR;
            }

            if( Value == Choice )
                break;
        }

        Tcl_Free( (char*) argv );

        if( k == argc )
            return OUT_OF_RANGE;

        pResult = Tcl_NewIntObj( k );
    }

    return OK;
}


//
//   c g i G e t I n t V a r
//

tRetCode tTclCgi::cgiGetIntVar( Tcl_Interp* pInterp, tVarType nVarType,
    char* sCgiVar, Tcl_Obj* pChoices, Tcl_Obj*& pResult )

{
    MTG_ASSERT( nVarType == xIntVar ||
                nVarType == xPosIntVar ||
                nVarType == xNonNegIntVar );

    tRetCode nRet;
    int nValue;

    if( ( nRet = getVar( sCgiVar, nValue ) ) != OK )
        return nRet;

    switch( nVarType ) {
        case xPosIntVar :
            if( nValue <= 0 )
                return OUT_OF_RANGE;
            break;

        case xNonNegIntVar :
            if( nValue < 0 )
                return OUT_OF_RANGE;
            break;

        default :
            break;
    }

    if( pChoices == 0 ) {
        pResult = Tcl_NewIntObj( nValue );
    }
    else {
        int argc;
        Tcl_Obj **argv;

        if( Tcl_ListObjGetElements( pInterp, pChoices,
                &argc, &argv ) != TCL_OK ) {
            return SCRIPT_ERROR;
        }

        int k;
        for( k = 0; k < argc; ++k ) {
            int nChoice;
            
            if( Tcl_GetIntFromObj( pInterp, argv[k], &nChoice ) != TCL_OK )
                return SCRIPT_ERROR;
            if( nValue == nChoice )
                break;
        }

        Tcl_Free( (char*) argv );

        if( k == argc )
            return OUT_OF_RANGE;

        pResult = Tcl_NewIntObj( k );
    }

    return OK;
}


//
//   c g i G e t D o u b l e V a r
//

tRetCode tTclCgi::cgiGetDoubleVar( Tcl_Interp* pInterp, tVarType nVarType,
    char* sCgiVar, Tcl_Obj* pChoices, Tcl_Obj*& pResult )

{
    MTG_ASSERT( nVarType == xDoubleVar ||
                nVarType == xPosDoubleVar ||
                nVarType == xNonNegDoubleVar );

    tRetCode nRet;
    double gValue;

    if( ( nRet = getVar( sCgiVar, gValue ) ) != OK )
        return nRet;

    switch( nVarType ) {
        case xPosDoubleVar :
            if( gValue <= 0 )
                return OUT_OF_RANGE;
            break;

        case xNonNegDoubleVar :
            if( gValue < 0 )
                return OUT_OF_RANGE;
            break;

        default :
            break;
    }

    if( pChoices == 0 ) {
        pResult = Tcl_NewDoubleObj( gValue );
    }
    else {
        int argc;
        Tcl_Obj **argv;

        if( Tcl_ListObjGetElements( pInterp, pChoices,
                &argc, &argv ) != TCL_OK ) {
            return SCRIPT_ERROR;
        }

        int k;
        for( k = 0; k < argc; ++k ) {
            double gChoice;
            
            if( Tcl_GetDoubleFromObj( pInterp, argv[k], &gChoice ) != TCL_OK )
                return SCRIPT_ERROR;
            if( gValue == gChoice )
                break;
        }

        Tcl_Free( (char*) argv );

        if( k == argc )
            return OUT_OF_RANGE;

        pResult = Tcl_NewIntObj( k );
    }

    return OK;
}


//
//   c g i I s A c t i v e
//

int tTclCgi::cgiIsActive( ClientData ClientData, Tcl_Interp *pInterp,
    int objc, Tcl_Obj *CONST objv[] )

{
    if( objc > 1 ) {
        Tcl_WrongNumArgs( pInterp, 1, objv, "" );
        return TCL_ERROR;
    }

    Tcl_SetObjResult( pInterp, Tcl_NewBooleanObj( isActive() ) );
    return TCL_OK;
}


//
//   c g i S e r v e r S o f t w a r e
//

int tTclCgi::cgiServerSoftware( ClientData ClientData, Tcl_Interp *pInterp,
    int objc, Tcl_Obj *CONST objv[] )

{
    if( objc != 1 ) {
        Tcl_WrongNumArgs( pInterp, 1, objv, "" );
        return TCL_ERROR;
    }
    Tcl_SetObjResult( pInterp, Tcl_NewStringObj( 
        const_cast<char*>( getServerSoftware() ), -1 ) );
    return TCL_OK;
}


//
//   c g i S e r v e r N a m e
//

int tTclCgi::cgiServerName( ClientData ClientData, Tcl_Interp *pInterp,
    int objc, Tcl_Obj *CONST objv[] )

{
    if( objc != 1 ) {
        Tcl_WrongNumArgs( pInterp, 1, objv, "" );
        return TCL_ERROR;
    }
    Tcl_SetObjResult( pInterp, Tcl_NewStringObj( 
        const_cast<char*>( getServerName() ), -1 ) );
    return TCL_OK;
}


//
//   c g i G a t e w a y I n t e r f a c e
//

int tTclCgi::cgiGatewayInterface( ClientData ClientData, Tcl_Interp *pInterp,
    int objc, Tcl_Obj *CONST objv[] )

{
    if( objc != 1 ) {
        Tcl_WrongNumArgs( pInterp, 1, objv, "" );
        return TCL_ERROR;
    }
    Tcl_SetObjResult( pInterp, Tcl_NewStringObj( 
        const_cast<char*>( getGatewayInterface() ), -1 ) );
    return TCL_OK;
}


//
//   c g i S e r v e r P r o t o c o l
//

int tTclCgi::cgiServerProtocol( ClientData ClientData, Tcl_Interp *pInterp,
    int objc, Tcl_Obj *CONST objv[] )

{
    if( objc != 1 ) {
        Tcl_WrongNumArgs( pInterp, 1, objv, "" );
        return TCL_ERROR;
    }
    Tcl_SetObjResult( pInterp, Tcl_NewStringObj( 
        const_cast<char*>( getServerProtocol() ), -1 ) );
    return TCL_OK;
}


//
//   c g i S e r v e r P o r t
//

int tTclCgi::cgiServerPort( ClientData ClientData, Tcl_Interp *pInterp,
    int objc, Tcl_Obj *CONST objv[] )

{
    if( objc != 1 ) {
        Tcl_WrongNumArgs( pInterp, 1, objv, "" );
        return TCL_ERROR;
    }
    Tcl_SetObjResult( pInterp, Tcl_NewStringObj( 
        const_cast<char*>( getServerPort() ), -1 ) );
    return TCL_OK;
}


//
//   c g i R e q u e s t M e t h o d
//

int tTclCgi::cgiRequestMethod( ClientData ClientData, Tcl_Interp *pInterp,
    int objc, Tcl_Obj *CONST objv[] )

{
    if( objc != 1 ) {
        Tcl_WrongNumArgs( pInterp, 1, objv, "" );
        return TCL_ERROR;
    }
    Tcl_SetObjResult( pInterp, Tcl_NewStringObj( 
        const_cast<char*>( getRequestMethod() ), -1 ) );
    return TCL_OK;
}


//
//   c g i P a t h I n f o
//

int tTclCgi::cgiPathInfo( ClientData ClientData, Tcl_Interp *pInterp,
    int objc, Tcl_Obj *CONST objv[] )

{
    if( objc != 1 ) {
        Tcl_WrongNumArgs( pInterp, 1, objv, "" );
        return TCL_ERROR;
    }
    Tcl_SetObjResult( pInterp, Tcl_NewStringObj( 
        const_cast<char*>( getPathInfo() ), -1 ) );
    return TCL_OK;
}


//
//   c g i P a t h T r a n s l a t e d
//

int tTclCgi::cgiPathTranslated( ClientData ClientData, Tcl_Interp *pInterp,
    int objc, Tcl_Obj *CONST objv[] )

{
    if( objc != 1 ) {
        Tcl_WrongNumArgs( pInterp, 1, objv, "" );
        return TCL_ERROR;
    }
    Tcl_SetObjResult( pInterp, Tcl_NewStringObj( 
        const_cast<char*>( getPathTranslated() ), -1 ) );
    return TCL_OK;
}


//
//   c g i S c r i p t N a m e
//

int tTclCgi::cgiScriptName( ClientData ClientData, Tcl_Interp *pInterp,
    int objc, Tcl_Obj *CONST objv[] )

{
    if( objc != 1 ) {
        Tcl_WrongNumArgs( pInterp, 1, objv, "" );
        return TCL_ERROR;
    }
    Tcl_SetObjResult( pInterp, Tcl_NewStringObj( 
        const_cast<char*>( getScriptName() ), -1 ) );
    return TCL_OK;
}


//
//   c g i Q u e r y S t r i n g
//

int tTclCgi::cgiQueryString( ClientData ClientData, Tcl_Interp *pInterp,
    int objc, Tcl_Obj *CONST objv[] )

{
    if( objc != 1 ) {
        Tcl_WrongNumArgs( pInterp, 1, objv, "" );
        return TCL_ERROR;
    }
    Tcl_SetObjResult( pInterp, Tcl_NewStringObj( 
        const_cast<char*>( getQueryString() ), -1 ) );
    return TCL_OK;
}


//
//   c g i R e m o t e H o s t
//

int tTclCgi::cgiRemoteHost( ClientData ClientData, Tcl_Interp *pInterp,
    int objc, Tcl_Obj *CONST objv[] )

{
    if( objc != 1 ) {
        Tcl_WrongNumArgs( pInterp, 1, objv, "" );
        return TCL_ERROR;
    }
    Tcl_SetObjResult( pInterp, Tcl_NewStringObj( 
        const_cast<char*>( getRemoteHost() ), -1 ) );
    return TCL_OK;
}


//
//   c g i R e m o t e A d d r
//

int tTclCgi::cgiRemoteAddr( ClientData ClientData, Tcl_Interp *pInterp,
    int objc, Tcl_Obj *CONST objv[] )

{
    if( objc != 1 ) {
        Tcl_WrongNumArgs( pInterp, 1, objv, "" );
        return TCL_ERROR;
    }
    Tcl_SetObjResult( pInterp, Tcl_NewStringObj( 
        const_cast<char*>( getRemoteAddr() ), -1 ) );
    return TCL_OK;
}


//
//   c g i A u t h T y p e
//

int tTclCgi::cgiAuthType( ClientData ClientData, Tcl_Interp *pInterp,
    int objc, Tcl_Obj *CONST objv[] )

{
    if( objc != 1 ) {
        Tcl_WrongNumArgs( pInterp, 1, objv, "" );
        return TCL_ERROR;
    }
    Tcl_SetObjResult( pInterp, Tcl_NewStringObj( 
        const_cast<char*>( getAuthType() ), -1 ) );
    return TCL_OK;
}


//
//   c g i R e m o t e U s e r
//

int tTclCgi::cgiRemoteUser( ClientData ClientData, Tcl_Interp *pInterp,
    int objc, Tcl_Obj *CONST objv[] )

{
    if( objc != 1 ) {
        Tcl_WrongNumArgs( pInterp, 1, objv, "" );
        return TCL_ERROR;
    }
    Tcl_SetObjResult( pInterp, Tcl_NewStringObj( 
        const_cast<char*>( getRemoteUser() ), -1 ) );
    return TCL_OK;
}


//
//   c g i R e m o t e I d e n t
//

int tTclCgi::cgiRemoteIdent( ClientData ClientData, Tcl_Interp *pInterp,
    int objc, Tcl_Obj *CONST objv[] )

{
    if( objc != 1 ) {
        Tcl_WrongNumArgs( pInterp, 1, objv, "" );
        return TCL_ERROR;
    }
    Tcl_SetObjResult( pInterp, Tcl_NewStringObj( 
        const_cast<char*>( getRemoteIdent() ), -1 ) );
    return TCL_OK;
}


//
//   c g i C o n t e n t T y p e
//

int tTclCgi::cgiContentType( ClientData ClientData, Tcl_Interp *pInterp,
    int objc, Tcl_Obj *CONST objv[] )

{
    if( objc != 1 ) {
        Tcl_WrongNumArgs( pInterp, 1, objv, "" );
        return TCL_ERROR;
    }
    Tcl_SetObjResult( pInterp, Tcl_NewStringObj( 
        const_cast<char*>( getContentType() ), -1 ) );
    return TCL_OK;
}


//
//   c g i A c c e p t
//

int tTclCgi::cgiAccept( ClientData ClientData, Tcl_Interp *pInterp,
    int objc, Tcl_Obj *CONST objv[] )

{
    if( objc != 1 ) {
        Tcl_WrongNumArgs( pInterp, 1, objv, "" );
        return TCL_ERROR;
    }
    Tcl_SetObjResult( pInterp, Tcl_NewStringObj( 
        const_cast<char*>( getAccept() ), -1 ) );
    return TCL_OK;
}


//
//   c g i U s e r A g e n t
//

int tTclCgi::cgiUserAgent( ClientData ClientData, Tcl_Interp *pInterp,
    int objc, Tcl_Obj *CONST objv[] )

{
    if( objc != 1 ) {
        Tcl_WrongNumArgs( pInterp, 1, objv, "" );
        return TCL_ERROR;
    }
    Tcl_SetObjResult( pInterp, Tcl_NewStringObj( 
        const_cast<char*>( getUserAgent() ), -1 ) );
    return TCL_OK;
}


//
//   c g i G e t V a r
//

int tTclCgi::cgiGetVar( ClientData ClientData, Tcl_Interp *pInterp,
    int objc, Tcl_Obj *CONST objv[] )

{
    int nNextObj = 1;
    tVarType nVarType;
    Tcl_Obj* pDefault;
    Tcl_Obj* pChoices;

    if( ! getOptions( pInterp, objc, objv, nNextObj,
            nVarType, pDefault, pChoices ) ) {
        return TCL_ERROR;
    }

    if( objc - nNextObj != 2 ) {
        Tcl_WrongNumArgs( pInterp, 1, objv, "?options? cgiVar tclVar" );
        return TCL_ERROR;
    }

    int nLength;
    char* sCgiVar = Tcl_GetStringFromObj( objv[nNextObj++], &nLength );
    char* sTclVar = Tcl_GetStringFromObj( objv[nNextObj++], &nLength );

    tRetCode nRet;
    Tcl_Obj* pResult;

    switch( nVarType ) {
        case xTextVar :
            nRet = cgiGetTextVar( pInterp, nVarType,
                sCgiVar, pChoices, pResult );
            break;

        case xDateVar :
            nRet = cgiGetDateVar( pInterp, nVarType,
                sCgiVar, pChoices, pResult );
            break;

        case xIntVar :
        case xPosIntVar :
        case xNonNegIntVar :
            nRet = cgiGetIntVar( pInterp, nVarType,
                sCgiVar, pChoices, pResult );
            break;

        case xDoubleVar :
        case xPosDoubleVar :
        case xNonNegDoubleVar :
            nRet = cgiGetDoubleVar( pInterp, nVarType,
                sCgiVar, pChoices, pResult );
            break;

        default :
            throw tException( INTERNAL_ERROR );
    }

    if( nRet == SCRIPT_ERROR )
        return TCL_ERROR;

    if( nRet == EMPTY && pDefault != 0 ) {
        pResult = pDefault;     // ref count will be incremented
        nRet = OK;              // by Tcl_ObjSetVar2...
    }

    if( nRet == OK ) {
        Tcl_Obj* p = Tcl_NewStringObj( sTclVar, -1 );

        Tcl_IncrRefCount( p );
        Tcl_ObjSetVar2( pInterp, p, 0, pResult, TCL_PARSE_PART1 );
        Tcl_DecrRefCount( p );
    }

    Tcl_SetObjResult( pInterp, Tcl_NewIntObj( (int) nRet ) );
    return TCL_OK;
}


//
//   c g i M a k e H t t p H e a d e r
//

int tTclCgi::cgiMakeHttpHeader( ClientData ClientData, Tcl_Interp *pInterp,
    int objc, Tcl_Obj *CONST objv[] )

{
    char* sType = 0;
    char* sSubType = 0;

    for( int k = 1; k < objc; ++k ) {
        int nLength;
        char* sString = Tcl_GetStringFromObj( objv[k], &nLength );

        if( nLength == 0 || *sString != '-' )
            return TCL_ERROR;

        char* s = StrToLower( sString );

        if( strcmp( s, "-type" ) == 0 ) {
            if( sType != 0 || k + 1 == objc ) {
                delete s;
                return TCL_ERROR;
            }
            sType = Tcl_GetStringFromObj( objv[++k], &nLength );
        }
        else
        if( strcmp( s, "-subtype" ) == 0 ) {
            if( sSubType != 0 || k + 1 == objc ) {
                delete s;
                return TCL_ERROR;
            }
            sSubType = Tcl_GetStringFromObj( objv[++k], &nLength );
        }
        else {
            delete s;
            return TCL_ERROR;
        }

        delete s;
    }

    if( sType == 0 )
        sType = "text";
    if( sSubType == 0 )
        sSubType = "html";

    char* s = new char[strlen( sType ) + strlen( sSubType ) + 256];

    sprintf( s, "Content-length: %s/%s%c%c", sType, sSubType, 10, 10 );
    Tcl_SetObjResult( pInterp, Tcl_NewStringObj( s, -1 ) );
    delete s;

    return TCL_OK;
}


//
//   c g i O u t H t t p H e a d e r
//

int tTclCgi::cgiOutHttpHeader( ClientData ClientData, Tcl_Interp *pInterp,
    int objc, Tcl_Obj *CONST objv[] )

{
    tRetCode nRet;

    if( cgiMakeHttpHeader( ClientData, pInterp, objc, objv ) != TCL_OK )
        return TCL_ERROR;
    if( ( nRet = out( Tcl_GetStringResult( pInterp ) ) ) != OK )
        return TCL_ERROR;

    return TCL_OK;
}


//
//   c g i O u t
//

int tTclCgi::cgiOut( ClientData ClientData, Tcl_Interp *pInterp,
    int objc, Tcl_Obj *CONST objv[] )

{
    tRetCode nRet;

    for( int k = 1; k < objc; ++k ) {
        int nLength;
        if( ( nRet = out(
                Tcl_GetStringFromObj( objv[k], &nLength ) ) ) != OK ) {
            return TCL_ERROR;
        }
    }

    return TCL_OK;
}


//
//   c r e a t e E x t e n s i o n
//

tRetCode tTclCgi::createExtension( tTclKernel& Kernel )

{
    #define MTG_CREATE_CMD( f ) \
        Tcl_CreateObjCommand( Kernel.interp(), "Mtg::" #f, f, 0, 0 );

    MTG_CREATE_CMD( cgiGetVar )
    MTG_CREATE_CMD( cgiServerSoftware )
    MTG_CREATE_CMD( cgiServerName )
    MTG_CREATE_CMD( cgiGatewayInterface )
    MTG_CREATE_CMD( cgiServerProtocol )
    MTG_CREATE_CMD( cgiServerPort )
    MTG_CREATE_CMD( cgiRequestMethod )
    MTG_CREATE_CMD( cgiPathInfo )
    MTG_CREATE_CMD( cgiPathTranslated )
    MTG_CREATE_CMD( cgiScriptName )
    MTG_CREATE_CMD( cgiQueryString )
    MTG_CREATE_CMD( cgiRemoteHost )
    MTG_CREATE_CMD( cgiRemoteAddr )
    MTG_CREATE_CMD( cgiAuthType )
    MTG_CREATE_CMD( cgiRemoteUser )
    MTG_CREATE_CMD( cgiRemoteIdent )
    MTG_CREATE_CMD( cgiContentType )
    MTG_CREATE_CMD( cgiAccept )
    MTG_CREATE_CMD( cgiUserAgent )        
    MTG_CREATE_CMD( cgiMakeHttpHeader )
    MTG_CREATE_CMD( cgiOutHttpHeader )
    MTG_CREATE_CMD( cgiOut )

    #undef MTG_CREATE_CMD

    return init();
}

#endif

MTG_END_NAMESPACE
