// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#include "MtgIncl.h"
#include "MtgTclHtmlReader.h"

MTG_BEGIN_NAMESPACE

#if defined(_MTG_WITH_TCL)

//
//   v a r i a b l e s
//

Tcl_ObjType tTclHtmlReader::m_Type = {
	"HtmlReader",
	freeInternalRepProc,
	dupInternalRepProc,
	updateStringProc,
	setFromAnyProc
};


//
//   t T c l H t m l R e a d e r
//

tTclHtmlReader::tTclHtmlReader( const char* sSource )
    : m_Doc( sSource )

{
    m_pReader = new tHtmlReader( m_Doc );
    m_pReader->readTables();
}


//
//   ~ t T c l H t m l R e a d e r
//

tTclHtmlReader::~tTclHtmlReader()

{
    if( m_pReader != 0 )
        delete m_pReader;
}


//
//   s e t F r o m A n y P r o c
//

int tTclHtmlReader::setFromAnyProc( Tcl_Interp *pInterp, Tcl_Obj *pObj )

{
    Tcl_ObjType *pOldType = pObj->typePtr;

    int nLength;
    char* sString = Tcl_GetStringFromObj( pObj, &nLength );

    if( pOldType != 0 && pOldType->freeIntRepProc != 0 )
        pOldType->freeIntRepProc( pObj );

        // Simply regard the string as an HTML source.

    pObj->internalRep.otherValuePtr = new tTclHtmlReader( sString );
    pObj->typePtr = &m_Type;

    return TCL_OK;
}


//
//   u p d a t e S t r i n g P r o c
//

void tTclHtmlReader::updateStringProc( Tcl_Obj *pObj )

{
        // The string representation is simply the HTML source.

    if( pObj->internalRep.otherValuePtr == 0 ) {
        pObj->bytes = Tcl_Alloc( 1 );
        strcpy( pObj->bytes, "" );
        pObj->length = 0;
        return;
    }

    tTclHtmlReader* R = 
        static_cast<tTclHtmlReader*>( pObj->internalRep.otherValuePtr );

    const char* s = R->m_Doc.source();

    if( s == 0 )
        s = "";

    pObj->length = strlen( s );
    pObj->bytes = Tcl_Alloc( pObj->length + 1 );

    strcpy( pObj->bytes, s );
}


//
//   d u p I n t e r n a l R e p P r o c
//

void tTclHtmlReader::dupInternalRepProc( Tcl_Obj *pSrc, Tcl_Obj *pDup )

{
        // We don't support that yet:
    pDup->internalRep.otherValuePtr = 0;
    pDup->typePtr = &m_Type;
}


//
//   f r e e I n t e r n a l R e p P r o c
//

void tTclHtmlReader::freeInternalRepProc( Tcl_Obj *pObj )

{
    if( pObj->internalRep.otherValuePtr != 0 ) {
        delete static_cast<tTclHtmlReader*>(
            pObj->internalRep.otherValuePtr );
    }
}


//
//   n e w O b j
//

Tcl_Obj* tTclHtmlReader::newObj( const char* sSource )

{
    Tcl_Obj* pObj = Tcl_NewObj();

    pObj->bytes = 0;
    pObj->internalRep.otherValuePtr = new tTclHtmlReader( sSource );
    pObj->typePtr = &m_Type;

    return pObj;
}


//
//   g e t R e a d e r
//

tHtmlReader* tTclHtmlReader::getReader( Tcl_Interp* pInterp,
    Tcl_Obj* pObj )

{
    if( pObj->typePtr != &m_Type || pObj->internalRep.otherValuePtr == 0 ) {
        if( pInterp != 0 ) {
            Tcl_Obj* pObj = Tcl_GetObjResult( pInterp );
            Tcl_AppendToObj( pObj,
                "first arg must be of type htmlReader", -1 );
        }
        return 0;
    }

    return static_cast<tTclHtmlReader*>(
        pObj->internalRep.otherValuePtr )->m_pReader;
}


//
//   d a t a T o L i s t
//

int tTclHtmlReader::dataToList( Tcl_Interp* pInterp, tHeap<int>& Data )

{
    if( pInterp == 0 )
        return TCL_OK;

    if( Data.numOfElems() == 0 ) {
        Tcl_SetObjResult( pInterp, Tcl_NewListObj( 0, 0 ) );
    }
    else {
        Tcl_Obj** objv = new Tcl_Obj*[Data.numOfElems()];

        for( int i = 0; i < Data.numOfElems(); ++i )
            objv[i] = Tcl_NewIntObj( Data[i] );

        Tcl_SetObjResult( pInterp,
            Tcl_NewListObj( Data.numOfElems(), objv ) );

        delete objv;
    }
    return TCL_OK;
}


//
//   d a t a T o L i s t
//

int tTclHtmlReader::dataToList( Tcl_Interp* pInterp,
    tHeap<const char*>& Data )

{
    if( pInterp == 0 )
        return TCL_OK;

    if( Data.numOfElems() == 0 ) {
        Tcl_SetObjResult( pInterp, Tcl_NewListObj( 0, 0 ) );
    }
    else {
        Tcl_Obj** objv = new Tcl_Obj*[Data.numOfElems()];

        for( int i = 0; i < Data.numOfElems(); ++i ) {
            const char* s = Data[i];

            if( s == 0 ) {
                objv[i] = Tcl_NewStringObj( 0, 0 );
            }
            else {
                objv[i] = Tcl_NewStringObj(
                            const_cast<char*>( s ), strlen( s ) );
            }
        }

        Tcl_SetObjResult( pInterp,
            Tcl_NewListObj( Data.numOfElems(), objv ) );

        delete objv;
    }
    return TCL_OK;
}


//
//   d a t a T o L i s t
//

int tTclHtmlReader::dataToList( Tcl_Interp* pInterp,
    tHeap2<const char*>& Data )

{
    if( pInterp == 0 )
        return TCL_OK;

    if( Data.numOfRows() == 0 || Data.numOfCols() == 0 ) {
        Tcl_SetObjResult( pInterp, Tcl_NewListObj( 0, 0 ) );
    }
    else {
        Tcl_Obj** objr = new Tcl_Obj*[Data.numOfRows()];
        Tcl_Obj** objc = new Tcl_Obj*[Data.numOfCols()];

        for( int i = 0; i < Data.numOfRows(); ++i ) {
            for( int j = 0; j < Data.numOfCols(); ++j ) {
                const char* s = Data[i][j];

                if( s == 0 ) {
                    objc[j] = Tcl_NewStringObj( 0, 0 );
                }
                else {
                    objc[j] = Tcl_NewStringObj(
                                const_cast<char*>( s ), strlen( s ) );
                }
            }
            objr[i] = Tcl_NewListObj( Data.numOfCols(), objc );
        }

        Tcl_SetObjResult( pInterp,
            Tcl_NewListObj( Data.numOfRows(), objr ) );

        delete objr;
        delete objc;
    }
    return TCL_OK;
}


//
//   c r e a t e H t m l R e a d e r
//

int tTclHtmlReader::createHtmlReader( ClientData ClientData,
    Tcl_Interp *pInterp, int objc, Tcl_Obj *CONST objv[] )

{
    if( objc != 2 ) {
        Tcl_WrongNumArgs( pInterp, 1, objv, "htmlSource" );
        return TCL_ERROR;
    }

    int nLength;
    char* sString = Tcl_GetStringFromObj( objv[1], &nLength );

    Tcl_SetObjResult( pInterp, newObj( sString ) );
    return TCL_OK;
}


//
//   h r T o p
//

int tTclHtmlReader::hrTop( ClientData ClientData,
    Tcl_Interp *pInterp, int objc, Tcl_Obj *CONST objv[] )

{
    if( objc != 2 ) {
        Tcl_WrongNumArgs( pInterp, 1, objv, "htmlSource" );
        return TCL_ERROR;
    }

    tHtmlReader* R = getReader( pInterp, objv[1] );
    if( R == 0 )
        return TCL_ERROR;

    R->top();
    return TCL_OK;
}


//
//   h r P u s h
//

int tTclHtmlReader::hrPush( ClientData ClientData,
    Tcl_Interp *pInterp, int objc, Tcl_Obj *CONST objv[] )

{
    if( objc != 2 ) {
        Tcl_WrongNumArgs( pInterp, 1, objv, "htmlSource" );
        return TCL_ERROR;
    }

    tHtmlReader* R = getReader( pInterp, objv[1] );
    if( R == 0 )
        return TCL_ERROR;

    R->push();
    return TCL_OK;
}


//
//   h r P o p
//

int tTclHtmlReader::hrPop( ClientData ClientData,
    Tcl_Interp *pInterp, int objc, Tcl_Obj *CONST objv[] )

{
    if( objc != 2 && objc != 3 ) {
        Tcl_WrongNumArgs( pInterp, 1, objv, "htmlReader" );
        return TCL_ERROR;
    }

    tHtmlReader* R = getReader( pInterp, objv[1] );
    if( R == 0 )
        return TCL_ERROR;

    if( objc == 3 ) {
        int nRemove;

        if( Tcl_GetBooleanFromObj( pInterp, objv[2], &nRemove ) != TCL_OK )
            return TCL_ERROR;
        R->pop( nRemove != 0 );
    }
    else {
        R->pop();
    }

    return TCL_OK;
}


//
//   h r S e l e c t T a b l e
//

int tTclHtmlReader::hrSelectTable( ClientData ClientData,
    Tcl_Interp *pInterp, int objc, Tcl_Obj *CONST objv[] )

{
    if( objc != 3 ) {
        Tcl_WrongNumArgs( pInterp, 1, objv, "htmlReader table" );
        return TCL_ERROR;
    }

    tHtmlReader* R = getReader( pInterp, objv[1] );
    if( R == 0 )
        return TCL_ERROR;

    int nTable;
    if( Tcl_GetIntFromObj( pInterp, objv[2], &nTable ) != TCL_OK )
        return TCL_ERROR;
    
    int nOk = R->selectTable( nTable ) ? 1 : 0;
    Tcl_SetObjResult( pInterp, Tcl_NewBooleanObj( nOk ) );

    return TCL_OK;
}


//
//   h r S e l e c t R o w
//

int tTclHtmlReader::hrSelectRow( ClientData ClientData,
    Tcl_Interp *pInterp, int objc, Tcl_Obj *CONST objv[] )

{
    if( objc < 3 || objc > 4 ) {
        Tcl_WrongNumArgs( pInterp, 1, objv, "htmlReader ?table? row" );
        return TCL_ERROR;
    }

    tHtmlReader* R = getReader( pInterp, objv[1] );
    if( R == 0 )
        return TCL_ERROR;

    int nOk, nRow;

    if( objc == 3 ) {
        if( Tcl_GetIntFromObj( pInterp, objv[2], &nRow ) != TCL_OK )
            return TCL_ERROR;
        nOk = R->selectRow( nRow ) ? 1 : 0;
    }
    else {
        int nTable;

        if( Tcl_GetIntFromObj( pInterp, objv[2], &nTable ) != TCL_OK ||
            Tcl_GetIntFromObj( pInterp, objv[3], &nRow ) != TCL_OK ) {
            return TCL_ERROR;
        }
        nOk = R->selectRow( nTable, nRow ) ? 1 : 0;
    }
   
    Tcl_SetObjResult( pInterp, Tcl_NewBooleanObj( nOk ) );
    return TCL_OK;
}


//
//   h r S e l e c t F i e l d
//

int tTclHtmlReader::hrSelectField( ClientData ClientData,
    Tcl_Interp *pInterp, int objc, Tcl_Obj *CONST objv[] )

{
    if( objc < 3 || objc > 5 ) {
        Tcl_WrongNumArgs( pInterp, 1, objv, "htmlReader ?table? ?row? field" );
        return TCL_ERROR;
    }

    tHtmlReader* R = getReader( pInterp, objv[1] );
    if( R == 0 )
        return TCL_ERROR;

    int nOk, nField;

    if( objc == 3 ) {
        if( Tcl_GetIntFromObj( pInterp, objv[2], &nField ) != TCL_OK )
            return TCL_ERROR;
        nOk = R->selectField( nField ) ? 1 : 0;
    }
    else 
    if( objc == 4 ) {
        int nRow;

        if( Tcl_GetIntFromObj( pInterp, objv[2], &nRow ) != TCL_OK ||
            Tcl_GetIntFromObj( pInterp, objv[3], &nField ) != TCL_OK ) {
            return TCL_ERROR;
        }
        nOk = R->selectField( nRow, nField ) ? 1 : 0;
    }
    else {
        int nTable, nRow;

        if( Tcl_GetIntFromObj( pInterp, objv[2], &nTable ) != TCL_OK ||
            Tcl_GetIntFromObj( pInterp, objv[3], &nRow ) != TCL_OK ||
            Tcl_GetIntFromObj( pInterp, objv[4], &nField ) != TCL_OK ) {
            return TCL_ERROR;
        }
        nOk = R->selectField( nTable, nRow, nField ) ? 1 : 0;
    }
   
    Tcl_SetObjResult( pInterp, Tcl_NewBooleanObj( nOk ) );
    return TCL_OK;
}


//
//   h r S e l e c t N e x t I n R o w
//

int tTclHtmlReader::hrSelectNextInRow( ClientData ClientData,
    Tcl_Interp *pInterp, int objc, Tcl_Obj *CONST objv[] )

{
    if( objc != 2 ) {
        Tcl_WrongNumArgs( pInterp, 1, objv, "htmlReader" );
        return TCL_ERROR;
    }

    tHtmlReader* R = getReader( pInterp, objv[1] );
    if( R == 0 )
        return TCL_ERROR;

    int nOk = R->selectNextInRow() ? 1 : 0;
    Tcl_SetObjResult( pInterp, Tcl_NewBooleanObj( nOk ) );

    return TCL_OK;
}


//
//   h r S e l e c t N e x t I n C o l u m n
//

int tTclHtmlReader::hrSelectNextInColumn( ClientData ClientData,
    Tcl_Interp *pInterp, int objc, Tcl_Obj *CONST objv[] )

{
    if( objc != 2 ) {
        Tcl_WrongNumArgs( pInterp, 1, objv, "htmlReader" );
        return TCL_ERROR;
    }

    tHtmlReader* R = getReader( pInterp, objv[1] );
    if( R == 0 )
        return TCL_ERROR;

    int nOk = R->selectNextInColumn() ? 1 : 0;
    Tcl_SetObjResult( pInterp, Tcl_NewBooleanObj( nOk ) );

    return TCL_OK;
}


//
//   h r V a l i d S e l e c t i o n
//

int tTclHtmlReader::hrValidSelection( ClientData ClientData,
    Tcl_Interp *pInterp, int objc, Tcl_Obj *CONST objv[] )

{
    if( objc != 2 ) {
        Tcl_WrongNumArgs( pInterp, 1, objv, "htmlReader" );
        return TCL_ERROR;
    }

    tHtmlReader* R = getReader( pInterp, objv[1] );
    if( R == 0 )
        return TCL_ERROR;

    int nOk = R->validSelection() ? 1 : 0;
    Tcl_SetObjResult( pInterp, Tcl_NewBooleanObj( nOk ) );

    return TCL_OK;
}


//
//   h r R e a d F i e l d
//

int tTclHtmlReader::hrReadField( ClientData ClientData,
    Tcl_Interp *pInterp, int objc, Tcl_Obj *CONST objv[] )

{
    if( objc < 2 || objc > 5 ) {
        Tcl_WrongNumArgs( pInterp, 1, objv, "htmlReader ?table? ?row? ?field?" );
        return TCL_ERROR;
    }

    tHtmlReader* R = getReader( pInterp, objv[1] );
    if( R == 0 )
        return TCL_ERROR;

    const char* s;

    if( objc == 3 ) {
        int nField;

        if( Tcl_GetIntFromObj( pInterp, objv[2], &nField ) != TCL_OK )
            return TCL_ERROR;
        s = R->readField( nField );
    }
    else 
    if( objc == 4 ) {
        int nRow, nField;

        if( Tcl_GetIntFromObj( pInterp, objv[2], &nRow ) != TCL_OK ||
            Tcl_GetIntFromObj( pInterp, objv[3], &nField ) != TCL_OK ) {
            return TCL_ERROR;
        }
        s = R->readField( nRow, nField );
    }
    else
    if( objc == 5 ) {
        int nTable, nRow, nField;

        if( Tcl_GetIntFromObj( pInterp, objv[2], &nTable ) != TCL_OK ||
            Tcl_GetIntFromObj( pInterp, objv[3], &nRow ) != TCL_OK ||
            Tcl_GetIntFromObj( pInterp, objv[4], &nField ) != TCL_OK ) {
            return TCL_ERROR;
        }
        s = R->readField( nTable, nRow, nField );
    }
    else {
        s = R->readField();
    }

    if( s == 0 ) {
        Tcl_SetObjResult( pInterp, Tcl_NewStringObj( "", 0 ) );
    }
    else {
        Tcl_SetObjResult( pInterp,
            Tcl_NewStringObj( const_cast<char*>( s ), strlen( s ) ) );
    }
    return TCL_OK;
}


//
//   h r R e a d R o w
//

int tTclHtmlReader::hrReadRow( ClientData ClientData,
    Tcl_Interp *pInterp, int objc, Tcl_Obj *CONST objv[] )

{
    if( objc < 2 || objc > 4 ) {
        Tcl_WrongNumArgs( pInterp, 1, objv, "htmlReader ?fromField? ?toField?" );
        return TCL_ERROR;
    }

    tHtmlReader* R = getReader( pInterp, objv[1] );
    if( R == 0 )
        return TCL_ERROR;

    tHeap<const char*> Data;

    if( objc == 2 ) {
        R->readRow( Data );
    }
    else 
    if( objc == 3 ) {
        int nTo;

        if( Tcl_GetIntFromObj( pInterp, objv[2], &nTo ) != TCL_OK )
            return TCL_ERROR;
        R->readRow( Data, nTo );
    }
    else {
        int nFrom, nTo;

        if( Tcl_GetIntFromObj( pInterp, objv[2], &nFrom ) != TCL_OK ||
            Tcl_GetIntFromObj( pInterp, objv[3], &nTo ) != TCL_OK ) {
            return TCL_ERROR;
        }
        R->readRow( Data, nFrom, nTo );
    }

    return dataToList( pInterp, Data );
}


//
//   h r R e a d C o l u m n
//

int tTclHtmlReader::hrReadColumn( ClientData ClientData,
    Tcl_Interp *pInterp, int objc, Tcl_Obj *CONST objv[] )

{
    if( objc < 2 || objc > 4 ) {
        Tcl_WrongNumArgs( pInterp, 1, objv, "htmlReader ?fromRow? ?toRow?" );
        return TCL_ERROR;
    }

    tHtmlReader* R = getReader( pInterp, objv[1] );
    if( R == 0 )
        return TCL_ERROR;

    tHeap<const char*> Data;

    if( objc == 2 ) {
        R->readColumn( Data );
    }
    else 
    if( objc == 3 ) {
        int nTo;

        if( Tcl_GetIntFromObj( pInterp, objv[2], &nTo ) != TCL_OK )
            return TCL_ERROR;
        R->readColumn( Data, nTo );
    }
    else {
        int nFrom, nTo;

        if( Tcl_GetIntFromObj( pInterp, objv[2], &nFrom ) != TCL_OK ||
            Tcl_GetIntFromObj( pInterp, objv[3], &nTo ) != TCL_OK ) {
            return TCL_ERROR;
        }
        R->readColumn( Data, nFrom, nTo );
    }

    return dataToList( pInterp, Data );
}


//
//   h r R e a d M a t r i x
//

int tTclHtmlReader::hrReadMatrix( ClientData ClientData,
    Tcl_Interp *pInterp, int objc, Tcl_Obj *CONST objv[] )

{
    if( objc < 2 || objc > 7 ) {
        Tcl_WrongNumArgs( pInterp, 1, objv,
            "htmlReader ?fromRow? ?toRow? ?fromField? ?toField? ?transpose?" );
        return TCL_ERROR;
    }

    tHtmlReader* R = getReader( pInterp, objv[1] );
    if( R == 0 )
        return TCL_ERROR;

    tHeap2<const char*> Data;
    bool bTranspose = false;

    if( objc == 3 || objc == 5 || objc == 7 ) {
        int t;

        if( Tcl_GetBooleanFromObj( pInterp, objv[objc - 1], &t ) != TCL_OK )
            return TCL_ERROR;
        bTranspose = ( t != 0 );
    }

    if( objc <= 3 ) {
        R->readMatrix( Data, bTranspose );
    }
    else 
    if( objc <= 5 ) {
        int nToRow, nToField;

        if( Tcl_GetIntFromObj( pInterp, objv[2], &nToRow ) != TCL_OK ||
            Tcl_GetIntFromObj( pInterp, objv[3], &nToField ) != TCL_OK ) {
            return TCL_ERROR;
        }
        R->readMatrix( Data, nToRow, nToField, bTranspose );
    }
    else {
        int nFromRow, nToRow, nFromField, nToField;

        if( Tcl_GetIntFromObj( pInterp, objv[2], &nFromRow ) != TCL_OK ||
            Tcl_GetIntFromObj( pInterp, objv[3], &nToRow ) != TCL_OK ||
            Tcl_GetIntFromObj( pInterp, objv[4], &nFromField ) != TCL_OK ||
            Tcl_GetIntFromObj( pInterp, objv[5], &nToField ) != TCL_OK ) {
            return TCL_ERROR;
        }
        R->readMatrix( Data, nFromRow, nToRow, nFromField,
            nToField, bTranspose );
    }

    return dataToList( pInterp, Data );
}


//
//   h r F i n d S u b s t r i n g
//

int tTclHtmlReader::hrFindSubstring( ClientData ClientData,
    Tcl_Interp *pInterp, int objc, Tcl_Obj *CONST objv[] )

{
    if( objc != 3 ) {
        Tcl_WrongNumArgs( pInterp, 1, objv, "htmlReader string" );
        return TCL_ERROR;
    }

    tHtmlReader* R = getReader( pInterp, objv[1] );
    if( R == 0 )
        return TCL_ERROR;

    int nLength;
    char* sString = Tcl_GetStringFromObj( objv[2], &nLength );

    tHeap<int> Data;

    R->findSubstring( sString, Data );
    return dataToList( pInterp, Data );
}


//
//   c r e a t e E x t e n s i o n
//

tRetCode tTclHtmlReader::createExtension( tTclKernel& Kernel )

{
    Tcl_RegisterObjType( &m_Type );

    #define MTG_CREATE_CMD( f ) \
        Tcl_CreateObjCommand( Kernel.interp(), "Mtg::" #f, f, 0, 0 );

    MTG_CREATE_CMD( createHtmlReader )

    MTG_CREATE_CMD( hrTop )
    MTG_CREATE_CMD( hrPush )
    MTG_CREATE_CMD( hrPop )

    MTG_CREATE_CMD( hrSelectTable )
    MTG_CREATE_CMD( hrSelectRow )
    MTG_CREATE_CMD( hrSelectField )

    MTG_CREATE_CMD( hrSelectNextInRow )
    MTG_CREATE_CMD( hrSelectNextInColumn )

    MTG_CREATE_CMD( hrValidSelection )

    MTG_CREATE_CMD( hrReadField )
    MTG_CREATE_CMD( hrReadRow )
    MTG_CREATE_CMD( hrReadColumn )
    MTG_CREATE_CMD( hrReadMatrix )

    MTG_CREATE_CMD( hrFindSubstring )

    #undef MTG_CREATE_CMD

    return OK;
}

#endif

MTG_END_NAMESPACE
