// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#include "MtgIncl.h"
#include "MtgTermStruct.h"
#include "MtgParser.h"

MTG_BEGIN_NAMESPACE


//
//   c l e a n u p
//

void tTermStruct::cleanup()

{
    m_Spec.reset();
    m_ImpSpec.reset();
    m_gBand = 0;
    m_bFinalized = false;
}


//
//   d i f f
//

tRetCode tTermStruct::diff( const tTermStruct& TS, int nSign )

{
    tRetCode nRet;
    int k1, k2, n1, n2, t;  
    double gSum1, gSum2, gImp;
    tSpecMap aSpec;
    tImpSpecHeap aImpSpec;

    m_Spec.migrateTo( aSpec );
    m_ImpSpec.migrateTo( aImpSpec );

    cleanup();

    k1 = k2 = 0;
    n1 = aImpSpec.numOfElems();
    n2 = TS.m_ImpSpec.numOfElems();

    while( k1 < n1 && k2 < n2 ) {
        t = aImpSpec[k1].m_nMaturity;
        if( TS.m_ImpSpec[k2].m_nMaturity < t )
            t = TS.m_ImpSpec[k2].m_nMaturity;

        tSpec& Spec1 = aSpec[t - 1];
        tSpec& Spec2 = TS.m_Spec[t - 1];

        gSum1 = Spec1.m_gImp2 + ( t - Spec1.m_nUnit - 1 ) * Spec1.m_gFwd2;
        gSum2 = Spec2.m_gImp2 + ( t - Spec2.m_nUnit - 1 ) * Spec2.m_gFwd2;

        gImp = scaleDown( nSign * ( gSum1 - gSum2 ) / t );
        if( ( nRet = addImplied( t, gImp ) ) != OK ) {
            cleanup();
            return nRet;
        }

        if( aImpSpec[k1].m_nMaturity == t )
            ++k1;
        if( TS.m_ImpSpec[k2].m_nMaturity == t )
            ++k2;
    }

    while( k1 < n1 ) {
        if( ( nRet = addImplied( aImpSpec[k1].m_nMaturity,
                        nSign * aImpSpec[k1].m_gImp ) ) != OK ) {
            cleanup();
            return nRet;
        }
        ++k1;
    }

    while( k2 < n2 ) {
        if( ( nRet = addImplied( TS.m_ImpSpec[k2].m_nMaturity,
                   -nSign * TS.m_ImpSpec[k2].m_gImp ) ) != OK ) {
            cleanup();
            return nRet;
        }
        ++k2;
    }

    aSpec.clear();
    aImpSpec.clear();

    return OK;
}


//
//   b o u n d
//

tRetCode tTermStruct::bound( const tTermStruct& TS, int nSign,
    bool& bAdjusted )

{
    tRetCode nRet;
    int k1, k2, n1, n2, t;  
    double gFwd, gFwd1, gFwd2;
    tImpSpecHeap aImpSpec;

        // Bound from above if nSign > 0, and bound
        // from below if nSign < 0.

    m_ImpSpec.migrateTo( aImpSpec );

    cleanup();

    k1 = k2 = 0;
    n1 = aImpSpec.numOfElems();
    n2 = TS.m_ImpSpec.numOfElems();

    gFwd1 = 0;
    gFwd2 = 0;

    bAdjusted = false;

    while( k1 < n1 && k2 < n2 ) {
        t = aImpSpec[k1].m_nMaturity;
        if( TS.m_ImpSpec[k2].m_nMaturity < t )
            t = TS.m_ImpSpec[k2].m_nMaturity;

        gFwd1 = aImpSpec[k1].m_gFwd;
        gFwd2 = TS.m_ImpSpec[k2].m_gFwd;

        if( nSign < 0 && gFwd1 < gFwd2 || nSign > 0 && gFwd1 > gFwd2 ) {
            gFwd = gFwd2;
            bAdjusted = true;
        }
        else {
            gFwd = gFwd1;
        }

        if( ( nRet = addForward( t, gFwd ) ) != OK ) {
            cleanup();
            return nRet;
        }

        if( aImpSpec[k1].m_nMaturity == t )
            ++k1;
        if( TS.m_ImpSpec[k2].m_nMaturity == t )
            ++k2;
    }

    while( k1 < n1 ) {
        gFwd1 = aImpSpec[k1].m_gFwd;

        if( nSign < 0 && gFwd1 < gFwd2 || nSign > 0 && gFwd1 > gFwd2 ) {
            gFwd = gFwd2;
            bAdjusted = true;
        }
        else {
            gFwd = gFwd1;
        }

        if( ( nRet = addForward( aImpSpec[k1].m_nMaturity, gFwd ) ) != OK ) {
            cleanup();
            return nRet;
        }
        ++k1;
    }

    while( k2 < n2 ) {
        gFwd2 = TS.m_ImpSpec[k2].m_gFwd;

        if( nSign < 0 && gFwd1 < gFwd2 || nSign > 0 && gFwd1 > gFwd2 ) {
            gFwd = gFwd2;
            bAdjusted = true;
        }
        else {
            gFwd = gFwd1;
        }

        if( ( nRet = addForward( TS.m_ImpSpec[k2].m_nMaturity,
                        gFwd ) ) != OK ) {
            cleanup();
            return nRet;
        }
        ++k2;
    }

    aImpSpec.clear();
    return OK;
}


//
//   c m p
//

int tTermStruct::cmp( const tTermStruct& TS,
    int& nEqualBefore, int& nEqualAfter ) const

{
    int k1, k2, n1, n2, t, c;
    double gFwd1, gFwd2;

    k1 = k2 = 0;
    n1 = m_ImpSpec.numOfElems();
    n2 = TS.m_ImpSpec.numOfElems();

    gFwd1 = 0;
    gFwd2 = 0;

    nEqualAfter = 0;
    c = 0;

    while( k1 < n1 && k2 < n2 ) {
        t = m_ImpSpec[k1].m_nMaturity;
        if( TS.m_ImpSpec[k2].m_nMaturity < t )
            t = TS.m_ImpSpec[k2].m_nMaturity;

        gFwd1 = m_ImpSpec[k1].m_gFwd;
        gFwd2 = TS.m_ImpSpec[k2].m_gFwd;

        if( gFwd1 != gFwd2 ) {
            nEqualAfter = t;

            if( c == 0 ) {
                    // Maximum of start dates:
                if( k1 == 0 )
                    nEqualBefore = 0;
                else
                    nEqualBefore = m_ImpSpec[k1 - 1].m_nMaturity;
                if( k2 > 0 &&
                    TS.m_ImpSpec[k2 - 1].m_nMaturity > nEqualBefore ) {
                    nEqualBefore = TS.m_ImpSpec[k2 - 1].m_nMaturity;
                }
                c = ( gFwd1 > gFwd2 ) ? 1 : -1;
            }
        }

        if( m_ImpSpec[k1].m_nMaturity == t )
            ++k1;
        if( TS.m_ImpSpec[k2].m_nMaturity == t )
            ++k2;
    }

    while( k1 < n1 ) {
        gFwd1 = m_ImpSpec[k1].m_gFwd;
        if( gFwd1 != gFwd2 ) {
            nEqualAfter = m_ImpSpec[k1].m_nMaturity;
            if( c == 0 ) {
                if( k1 == 0 )
                    nEqualBefore = 0;
                else
                    nEqualBefore = m_ImpSpec[k1 - 1].m_nMaturity;
                c = ( gFwd1 > gFwd2 ) ? 1 : -1;
            }
        }
        ++k1;
    }

    while( k2 < n2 ) {
        gFwd2 = TS.m_ImpSpec[k2].m_gFwd;
        if( gFwd1 != gFwd2 ) {
            nEqualAfter = m_ImpSpec[k2].m_nMaturity;
            if( c == 0 ) {
                if( k2 == 0 )
                    nEqualBefore = 0;
                else
                    nEqualBefore = TS.m_ImpSpec[k2].m_nMaturity;
                c = ( gFwd1 > gFwd2 ) ? 1 : -1;
            }
        }
        ++k2;
    }

    if( gFwd1 != gFwd2 )
        nEqualAfter = INT_MAX;
    if( c == 0 )
        nEqualBefore = INT_MAX;

    return c;
}


//
//   c o p y F r o m
//

void tTermStruct::copyFrom( const tTermStruct &TS )

{
    if( &TS == this )
        return;

    cleanup();

    m_Spec.copyFrom( TS.m_Spec );
    m_ImpSpec.copyFrom( TS.m_ImpSpec );
    m_gBand = TS.m_gBand;

    m_bFinalized = TS.m_bFinalized;

    if( m_bFinalized ) {
        m_gMin = TS.m_gMin;
        m_gMax = TS.m_gMax;
        m_nConstantUntil = TS.m_nConstantUntil;
    }         
}


//
//   t T e r m S t r u c t
//

tTermStruct::tTermStruct()

{
    m_gBand = 0;
    m_bFinalized = false;
}


//
//   t T e r m S t r u c t
//

tTermStruct::tTermStruct( const tTermStruct &TS )

{
    copyFrom( TS );
}


//
//   ~ t T e r m S t r u c t
//

tTermStruct::~tTermStruct()

{
}


//
//   o p e r a t o r =
//

tTermStruct& tTermStruct::operator=(
    const tTermStruct &TS )

{
    if( &TS != this )
        copyFrom( TS );
    return *this;
}


//
//   a d d F o r w a r d
//

tRetCode tTermStruct::addForward( int nMaturity, double gFwd )

{
    MTG_ASSERT( nMaturity > 0 );

    tRetCode nRet;
    int i, j, n;
    double gImp, gFwd1;

    if( ( nRet = testFwd( gFwd ) ) != OK )
        return nRet;

    n = m_ImpSpec.numOfElems();
    j = n;

    while( j > 0 &&
            m_ImpSpec[j - 1].m_nMaturity >= nMaturity ) {
        if( m_ImpSpec[j - 1].m_nMaturity == nMaturity )
            return INVALID_MATURITY;
        --j;
    }

    if( j == 0 ) {
        gImp = gFwd;
    }
    else {
        int t = m_ImpSpec[j - 1].m_nMaturity;

        gImp = scaleDown( ( t * scaleUp( m_ImpSpec[j - 1].m_gImp ) +
                ( nMaturity - t ) * scaleUp( gFwd ) ) / nMaturity );
    }

    if( j < n ) {
        nRet = calcFwd( nMaturity, gImp, m_ImpSpec[j].m_nMaturity,
                m_ImpSpec[j].m_gImp, gFwd1 );
        if( nRet != OK )
            return nRet;
        m_ImpSpec[j].m_gFwd = gFwd1;
        ++m_ImpSpec;
        for( i = n; i > j; --i )
            m_ImpSpec[i] = m_ImpSpec[i - 1];
    }
    else {
        ++m_ImpSpec;
    }

    m_ImpSpec[j].m_nMaturity = nMaturity;
    m_ImpSpec[j].m_gImp = gImp;
    m_ImpSpec[j].m_gFwd = gFwd;

    m_bFinalized = false;

    return OK;
}


//
//   a d d I m p l i e d
//

tRetCode tTermStruct::addImplied( int nMaturity, double gImp )

{
    MTG_ASSERT( nMaturity > 0 );

    tRetCode nRet;
    int i, j, n;
    double gFwd, gFwd1;

    if( ( nRet = testImp( gImp ) ) != OK )
        return nRet;

    n = m_ImpSpec.numOfElems();
    j = n;

    while( j > 0 &&
            m_ImpSpec[j - 1].m_nMaturity >= nMaturity ) {
        if( m_ImpSpec[j - 1].m_nMaturity == nMaturity )
            return INVALID_MATURITY;
        --j;
    }

    if( j == 0 ) {
        gFwd = gImp;
    }
    else {
        nRet = calcFwd( m_ImpSpec[j - 1].m_nMaturity,
                m_ImpSpec[j - 1].m_gImp, nMaturity, gImp, gFwd );
        if( nRet != OK )
            return nRet;
    }

    if( j < n ) {
        nRet = calcFwd( nMaturity, gImp, m_ImpSpec[j].m_nMaturity,
                m_ImpSpec[j].m_gImp, gFwd1 );
        if( nRet != OK )
            return nRet;
        m_ImpSpec[j].m_gFwd = gFwd1;
        ++m_ImpSpec;
        for( i = n; i > j; --i )
            m_ImpSpec[i] = m_ImpSpec[i - 1];
    }
    else {
        ++m_ImpSpec;
    }

    m_ImpSpec[j].m_nMaturity = nMaturity;
    m_ImpSpec[j].m_gImp = gImp;
    m_ImpSpec[j].m_gFwd = gFwd;

    m_bFinalized = false;

    return OK;
}


//
//   s e t B a n d
//

void tTermStruct::setBand( double gBand )

{
    if( m_gBand != gBand ) {
        m_gBand = gBand;
        m_bFinalized = false;
    }
}


//
//   o p e r a t o r < =
//

bool tTermStruct::operator<=( const tTermStruct& TS ) const

{
    int nEqualBefore, nEqualAfter;
    return cmp( TS, nEqualBefore, nEqualAfter ) <= 0;
}


//
//   b o u n d F r o m B e l o w
//

tRetCode tTermStruct::boundFromBelow( const tTermStruct& TS )

{
    tRetCode nRet;
    bool bAdjusted;

    if( ( nRet = bound( TS, -1, bAdjusted ) ) != OK )
        return nRet;
    if( bAdjusted )
        return ADJUSTMENT;
    return OK;
}


//
//   b o u n d F r o m A b o v e
//

tRetCode tTermStruct::boundFromAbove( const tTermStruct& TS )

{
    tRetCode nRet;
    bool bAdjusted;

    if( ( nRet = bound( TS, 1, bAdjusted ) ) != OK )
        return nRet;
    if( bAdjusted )
        return ADJUSTMENT;
    return OK;
}


//
//   f i n a l i z e
//

tRetCode tTermStruct::finalize()

{
    int i, n;
    tSpec Spec, LastSpec;

    if( m_bFinalized )
        return OK;

    m_nConstantUntil = 0;
    m_gMin = 0;
    m_gMax = 0;

    m_Spec.reset();

    for( i = 0; i < m_ImpSpec.numOfElems(); ++i ) {
        if( i == 0 )
            Spec.m_nUnit = 0;
        else
            Spec.m_nUnit = m_ImpSpec[i - 1].m_nMaturity;

        Spec.m_gFwd = m_ImpSpec[i].m_gFwd;
        if( m_gBand != 0 ) {
            Spec.m_gFwd += m_gBand;
            trim( Spec.m_gFwd );
        }

        Spec.m_gFwd2 = scaleUp( Spec.m_gFwd );
        Spec.m_gImp2 = Spec.m_gFwd2;

        if( i == 0 ) {
            m_gMin = m_gMax = Spec.m_gFwd;
        }
        else {
            n = Spec.m_nUnit - LastSpec.m_nUnit - 1;
            Spec.m_gImp2 += LastSpec.m_gImp2 + n * LastSpec.m_gFwd2;

            if( Spec.m_gFwd < m_gMin )
                m_gMin = Spec.m_gFwd;
            else
                if( Spec.m_gFwd > m_gMax )
                    m_gMax = Spec.m_gFwd;
        }

        if( i == 0 ||
            m_nConstantUntil == Spec.m_nUnit &&
                Spec.m_gFwd == LastSpec.m_gFwd ) {
            m_nConstantUntil = m_ImpSpec[i].m_nMaturity;
        }

        LastSpec = Spec;
        m_Spec.append( m_ImpSpec[i].m_nMaturity - 1, Spec );
    }

    m_bFinalized = true;

#if defined(_DEBUG)
    //for( i = 0; i < m_ImpSpec.numOfElems(); ++i ) {
    //    if( i == 0 )
    //        MTG_TRACE( "TS: " );
    //    else
    //        MTG_TRACE( ", " );
    //    MTG_TRACE( "%d: %lg -> %lg", m_ImpSpec[i].m_nMaturity,
    //        fwd( m_ImpSpec[i].m_nMaturity - 1 ),
    //        imp( m_ImpSpec[i].m_nMaturity - 1 ) );
    //    if( i == m_ImpSpec.numOfElems() - 1 )
    //        MTG_TRACE( "\n" );
    //}
#endif

    return OK;
}


//
//   g e t F w d R a n g e
//

void tTermStruct::getFwdRange( double& gMin, double& gMax ) const

{
    MTG_ASSERT( m_bFinalized );

    gMin = m_gMin;
    gMax = m_gMax;
}


//
//   f o r w a r d
//

double tTermStruct::forward( int nUnit ) const

{
    MTG_ASSERT( m_bFinalized );

    if( nUnit < 0 ) 
        nUnit = 0;

    if( nUnit >= m_Spec.numOfElems() ) {
        if( m_Spec.numOfElems() == 0 )
            return 0;
        nUnit = m_Spec.numOfElems() - 1;
    }

    return m_Spec[nUnit].m_gFwd;
}


//
//   f o r w a r d
//

double tTermStruct::forward( int nFromUnit, int nToUnit ) const

{
    MTG_ASSERT( m_bFinalized );

    if( nFromUnit == nToUnit )
        return forward( nFromUnit );    // simply a lookup

    int t, n;
    double dr1, dr2;

    if( nFromUnit > nToUnit ) {         // out of courtesy
        int h = nFromUnit;
        nFromUnit = nToUnit;
        nToUnit = h;
    }

    if( nToUnit <= 0 )
        return forward( 0 );

    if( ( n = m_Spec.numOfElems() ) == 0 )
        return 0;

    if( nFromUnit < 0 ) {
        dr1 = nFromUnit * m_Spec[0].m_gFwd2;
    }
    else
    if( nFromUnit == 0 ) {
        dr1 = 0;
    }
    else {
        tSpec& Spec =
            ( nFromUnit > n ) ? m_Spec[n - 1] : m_Spec[nFromUnit - 1];
        t = nFromUnit - Spec.m_nUnit - 1;
        dr1 = Spec.m_gImp2 + t * Spec.m_gFwd2;
    }

    tSpec& Spec = ( nToUnit > n ) ? m_Spec[n - 1] : m_Spec[nToUnit - 1];
    dr2 = Spec.m_gImp2 + ( nToUnit - 1 - Spec.m_nUnit ) * Spec.m_gFwd2;

    return scaleDown( ( dr2 - dr1 ) / ( nToUnit - nFromUnit ) );
}


//
//   f o r w a r d
//

double tTermStruct::forward( double gFromUnit, double gToUnit ) const

{
    MTG_ASSERT( m_bFinalized );

    int nCeilFrom = (int) ceil( gFromUnit );
    int nFloorTo = (int) floor( gToUnit );

    if( nCeilFrom > nFloorTo ) {
            // interval is within the same time unit
        return forward( nFloorTo );
    }

    double gRate;

    if( nCeilFrom < nFloorTo ) {
            // interval spans several time units
        gRate = scaleUp( forward( nCeilFrom, nFloorTo ) );
    }
    else {
            // interval spans two adjacent time units
        gRate = scaleUp( forward( nCeilFrom ) );
    }

        // now look at head and tail
    if( nCeilFrom != gFromUnit ) {
        gRate = 
            ( ( nCeilFrom - gFromUnit ) *
                    scaleUp( forward( nCeilFrom - 1 ) ) +
              ( nFloorTo - nCeilFrom ) * gRate ) / 
              ( nFloorTo - gFromUnit );
    }
    if( nFloorTo != gToUnit ) {
        gRate = 
            ( ( gToUnit - nFloorTo ) * scaleUp( forward( nFloorTo ) ) +
              ( nFloorTo - gFromUnit ) * gRate ) / 
              ( gToUnit - gFromUnit );
    }

    return scaleDown( gRate );
}


//
//   i m p l i e d
//

double tTermStruct::implied( int nMaturity ) const

{
    return forward( 0, nMaturity );
}


//
//   m a t u r i t y
//

int tTermStruct::maturity() const

{
    MTG_ASSERT( m_bFinalized );
    return m_Spec.numOfElems();
}


//
//   c o n s t a n t U n t i l
//

int tTermStruct::constantUntil() const

{
    MTG_ASSERT( m_bFinalized );

    if( m_nConstantUntil >= m_Spec.numOfElems() )
        return INT_MAX;
    return m_nConstantUntil;
}


//
//   c e r t a i n U n t i l
//

int tTermStruct::certainUntil( const tTermStruct& TS ) const

{
    MTG_ASSERT( m_bFinalized );

    int nEqualBefore, nEqualAfter;

    if( cmp( TS, nEqualBefore, nEqualAfter ) == 0 )
        return INT_MAX;
    return nEqualBefore;
}


//
//   c e r t a i n A f t e r
//

int tTermStruct::certainAfter( const tTermStruct& TS ) const

{
    MTG_ASSERT( m_bFinalized );

    int nEqualBefore, nEqualAfter;

    if( cmp( TS, nEqualBefore, nEqualAfter ) == 0 )
        return 0;
    return nEqualAfter;
}


//
//   p a r s e P a r a m
//

tRetCode tTermStruct::parseParam( tParser& Parser,
    double& gPlusBand, double& gMinusBand,
    tTermStruct* pMinTS, tTermStruct* pMaxTS )

{
    MTG_ASSERT( ( pMinTS != 0 ) == ( pMaxTS != 0 ) );

    tRetCode nRet;
    bool bImplied;
    int nMaturity, nSign;
    double gPrior, gMin, gMax, gBand;

    switch( Parser.curToken() ) { 
        case xTokForward :
        case xTokImplied :
            bImplied = ( Parser.curToken() == xTokImplied );

            if( ( nRet = Parser.readToken() ) != OK ||
                ( nRet = Parser.scanInteger( nMaturity, 1 ) ) != OK ) {
                return nRet;
            }
            if( pMinTS != 0 || pMaxTS != 0 )
                nRet = Parser.tScanner::scanPercentageRange( gMin, gPrior, gMax );
            else
                nRet = Parser.scanPercentage( gPrior );
            if( nRet != OK )
                return nRet;

            if( bImplied ) {
                if( ( nRet = addImplied( nMaturity, gPrior ) ) != OK )
                    return Parser.setError( nRet );
                if( pMinTS != 0 ) {
                    if( ( nRet = pMinTS->addImplied( nMaturity, gMin ) ) != OK )
                        return Parser.setError( nRet );
                }
                if( pMaxTS != 0 ) {
                    if( ( nRet = pMaxTS->addImplied( nMaturity, gMax ) ) != OK )
                        return Parser.setError( nRet );
                }
            }
            else {
                if( ( nRet = addForward( nMaturity, gPrior ) ) != OK )
                    return Parser.setError( nRet );
                if( pMinTS != 0 ) {
                    if( ( nRet = pMinTS->addForward( nMaturity, gMin ) ) != OK )
                        return Parser.setError( nRet );
                }
                if( pMaxTS != 0 ) {
                    if( ( nRet = pMaxTS->addForward( nMaturity, gMax ) ) != OK )
                        return Parser.setError( nRet );
                }
            }
            break;

        case xTokBand :
            if( ! pMinTS && ! pMaxTS )
                return Parser.setError( INVALID_KEYWORD );
            if( ( nRet = Parser.readToken() ) != OK )
                return nRet;
            nSign = 0;
            if( Parser.curToken() == xTokPlus ) {
                if( ( nRet = Parser.readToken() ) != OK )
                    return nRet;
                if( Parser.curToken() == xTokMinus ) {
                    if( ( nRet = Parser.readToken() ) != OK )
                        return nRet;
                }
                else {
                    nSign = +1;
                }
            }
            else
            if( Parser.curToken() == xTokMinus ) {
                if( ( nRet = Parser.readToken() ) != OK )
                    return nRet;
                if( Parser.curToken() == xTokPlus ) {
                    if( ( nRet = Parser.readToken() ) != OK )
                        return nRet;
                }
                else {
                    nSign = -1;
                }
            }
            if( nSign >= 0 && gPlusBand > 0 ||
                nSign <= 0 && gMinusBand > 0 ) {
                return Parser.setError( "band defined more than once" );
            }
            if( ( nRet = Parser.scanPosPercentage( gBand ) ) != OK )
                return nRet;
            if( nSign >= 0 )
                gPlusBand = gBand;
            if( nSign <= 0 )
                gMinusBand = gBand;
            break;

        default :
            return GO;  // no access at this point to other parseParam()
    }                   // functions in the hierachry

    return OK;
}

    
//
//   p a r s e P o s t f i x
//

tRetCode tTermStruct::parsePostfix( tParser& Parser,
    double gPlusBand, double gMinusBand,
    tTermStruct* pMinTS, tTermStruct* pMaxTS )

{
    MTG_ASSERT( ( pMinTS != 0 ) == ( pMaxTS != 0 ) );

    if( pMinTS != 0 ) {
        if( ! ( *pMinTS <= *this ) )
            return Parser.setError( "crossing in term structure" );
        if( gMinusBand > 0 )
            *pMinTS -= gMinusBand;
    }

    if( pMaxTS != 0 ) {
        if( ! ( *this <= *pMaxTS ) )
            return Parser.setError( "crossing in term structure" );
        if( gPlusBand > 0 )
            *pMaxTS += gPlusBand;
    }

    return OK;
}

MTG_END_NAMESPACE