// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#include "MtgIncl.h"
#include "MtgEvaluate.h"
#include "MtgHeap2.h"
#include "MtgParser.h"
#include "MtgProfile.h"
#include "MtgRepository.h"
#include "MtgSource.h"
#include "MtgSummary.h"
#include "MtgTimer.h"

#if defined(_WIN32)
    #include <direct.h>    
	#include "windows.h"
#endif

MTG_USING_NAMESPACE


//
//   d e c l a r a t i o n s
//

static void mathError( char* sMessage );
static char* unquote( const char* sQuoted );
static void setDirectory();
static bool open( char* sFileName, tSummary& Summary, MLINK& MathLocal );

static void sendInfo( char* sFileName,
    tRetCode (tSummary::* f)( MLINK MathLocal ) );

static void sendLayerInfo( char* sFileName, int nLayer,
    tRetCode (tSummary::* f)( MLINK MathLocal, int nLayer ) );

static void sendLayerAggrInfo( char* sFileName, int* Layer,
    long nNumOfLayers, tRetCode (tSummary::* f)( MLINK MathLocal,
        const tHeap<int>& LayerSign ) );

static void sendLayerClaimInfo( char* sFileName, int nLayer, int* Claim,
    long nNumOfClaims, tRetCode (tSummary::* f)( MLINK MathLocal, int nLayer,
        const tHeap<int>& ClaimSign ) );

static char* xlatString( const char* sString );


//
//   m a t h E r r o r
//

static void mathError( char* sMessage )

{
    MLPutString( stdlink, sMessage );
	MLEndPacket( stdlink );

#if defined(_WIN32)
        // make sure floating point support is loaded
        // (this is a good place to do it):
    double a = 1;
    a = a * a;
#endif
}


//
//   u n q u o t e
//

static char* unquote( const char* sQuoted )
{
    char *s = StrCopy( sQuoted );
    int i = 0;
    int j = 0;

    while( sQuoted[i] ) {
        if( sQuoted[i] == '\\' && sQuoted[i + 1] == '\\' )
            ++i;
        s[j++] = sQuoted[i++];
    }
    s[j] = 0;

	return s;
}


//
//   s e t D i r e c t o r y
//

static void setDirectory()

{
    long nArgs;
    const char* sDir;

        // If the current directory is changed in Mathematica,
        // we don't know that. That's why we have to investigate
        // here.

    MLPutFunction( stdlink, "EvaluatePacket", 1 );
    MLPutFunction( stdlink, "Directory", 0 );
    MLEndPacket( stdlink );
    
    if( MLCheckFunction( stdlink, "ReturnPacket", &nArgs ) != 0 && 
        MLGetString( stdlink, &sDir ) != 0 ) {

        char *s = unquote( sDir );
        MLDisownString( stdlink, sDir );

        chdir( s );
        delete s;
    }
}


//
//   o p e n
//

static bool open( char* sFileName, tSummary& Summary, MLINK& MathLocal )

{
    tRetCode nRet;
    long nErrno;

    setDirectory();

	char *s = unquote( sFileName );

    if( ( nRet = Summary.openRead( s ) ) != OK ) {
		delete s;
        mathError( "unable to open file" );
        return false;
    }
	delete s;

    MathLocal = MLLoopbackOpen( stdenv, &nErrno );
    /*if( nErrno != 0 ) {
        mathError( "unable to create loopback link" );
        Summary.close();
        return false;
    }*/

    return true;
}


//
//   s e n d I n f o
//

static void sendInfo( char* sFileName,
    tRetCode (tSummary::* f)( MLINK MathLocal ) )

{
    tRetCode nRet;
    tSummary Summary;
    MLINK MathLocal;

    if( ! open( sFileName, Summary, MathLocal ) )
        return;

    if( ( nRet = (Summary.*f)( MathLocal ) ) == OK ) {
        MLTransferExpression( stdlink, MathLocal );
	    MLEndPacket( stdlink );
    }
    else {
        mathError( "unable to get information" );
    }

    MLClose( MathLocal );
    Summary.close();
}


//
//   s e n d L a y e r I n f o
//

static void sendLayerInfo( char* sFileName, int nLayer,
    tRetCode (tSummary::* f)( MLINK MathLocal, int nLayer ) )

{
    tRetCode nRet;
    tSummary Summary;
    MLINK MathLocal;

    if( ! open( sFileName, Summary, MathLocal ) )
        return;

    if( ( nRet = (Summary.*f)( MathLocal, nLayer ) ) == OK ) {
        MLTransferExpression( stdlink, MathLocal );
	    MLEndPacket( stdlink );
    }
    else {
        mathError( "unable to get information" );
    }

    MLClose( MathLocal );
    Summary.close();
}


//
//   s e n d L a y e r A g g r I n f o
//

static void sendLayerAggrInfo( char* sFileName, int* Layer,
    long nNumOfLayers, tRetCode (tSummary::* f)( MLINK MathLocal,
        const tHeap<int>& LayerSign ) )

{
    tRetCode nRet;
    tSummary Summary;
    tHeap<int> LayerSign;
    MLINK MathLocal;

    if( ! open( sFileName, Summary, MathLocal ) )
        return;

    int nMaxLayer = 0;
    for( long i = 0; i < nNumOfLayers; ++i ) {
        if( Layer[i] > 0 && Layer[i] > nMaxLayer )
            nMaxLayer = Layer[i];
        else
        if( Layer[i] < 0 && -Layer[i] > nMaxLayer )
            nMaxLayer = -Layer[i];
    }

        // These are layer id's, not indexes.

    LayerSign.numOfElems( nMaxLayer + 1 );
    for( int j = 0; j < LayerSign.numOfElems(); ++j )
        LayerSign[j] = 0;

    for( MTG_FOR_INIT( long ) i = 0; i < nNumOfLayers; ++i ) {
        if( Layer[i] > 0 )
            LayerSign[Layer[i]] = 1;
        else
        if( Layer[i] < 0 )
            LayerSign[-Layer[i]] = -1;
    }

    if( ( nRet = (Summary.*f)( MathLocal, LayerSign ) ) == OK ) {
        MLTransferExpression( stdlink, MathLocal );
	    MLEndPacket( stdlink );
    }
    else {
        mathError( "unable to get information" );
    }

    MLClose( MathLocal );
    Summary.close();
}


//
//   s e n d L a y e r C l a i m I n f o
//

static void sendLayerClaimInfo( char* sFileName, int nLayer, int* Claim,
    long nNumOfClaims, tRetCode (tSummary::* f)( MLINK MathLocal, int nLayer,
        const tHeap<int>& ClaimSign ) )

{
    tRetCode nRet;
    tSummary Summary;
    tHeap<int> ClaimSign;
    MLINK MathLocal;

    if( ! open( sFileName, Summary, MathLocal ) )
        return;

    int nMaxClaim = 0;
    for( long i = 0; i < nNumOfClaims; ++i ) {
        if( Claim[i] > 0 && Claim[i] > nMaxClaim )
            nMaxClaim = Claim[i];
        else
        if( Claim[i] < 0 && -Claim[i] > nMaxClaim )
            nMaxClaim = -Claim[i];
    }

    ClaimSign.numOfElems( nMaxClaim );
    for( int j = 0; j < ClaimSign.numOfElems(); ++j )
        ClaimSign[j] = 0;

    for( MTG_FOR_INIT( long ) i = 0; i < nNumOfClaims; ++i ) {
        if( Claim[i] > 0 )
            ClaimSign[Claim[i] - 1] = 1;
        else
        if( Claim[i] < 0 )
            ClaimSign[-Claim[i] - 1] = -1;
    }

    if( ( nRet = (Summary.*f)( MathLocal, nLayer, ClaimSign ) ) == OK ) {
        MLTransferExpression( stdlink, MathLocal );
	    MLEndPacket( stdlink );
    }
    else {
        mathError( "unable to get information" );
    }

    MLClose( MathLocal );
    Summary.close();
}


//
//   x l a t S t r i n g
//

static char* xlatString( const char* sString )

{
    char* s = new char[strlen( sString ) + 1];
    char* q = s;

    while( *sString ) {
        if( *sString == '\\' &&
            sString[1] >= '0' && sString[1] <= '7' &&
            sString[2] >= '0' && sString[2] <= '7' &&
            sString[3] >= '0' && sString[3] <= '7' ) {
            int c;
            sscanf( &sString[1], "%3o", &c );
            *q++ = (char) c;
            sString += 4;
        }
        else {
            *q++ = *sString++;
        }
    }
    *q = 0;

    return s;
}


//
//   M t g G e t L a t t i c e
//

extern "C" void MtgGetLattice( char* sFileName )

{
    sendInfo( sFileName, tSummary::sendLattice );
}


//
//   M t g G e t L a y e r s
//

extern "C" void MtgGetLayers( char* sFileName )

{
    sendInfo( sFileName, tSummary::sendSlotLayers );
}


//
//   M t g G e t C o m p l e x i t y
//

extern "C" void MtgGetComplexity( char* sFileName, int nLayer )

{
    sendLayerInfo( sFileName, nLayer, tSummary::sendComplexity );
}


//
//   M t g G e t M a x C o m p l e x i t y
//

extern "C" void MtgGetMaxComplexity( char* sFileName )

{
    sendInfo( sFileName, tSummary::sendMaxComplexity );
}


//
//   M t g G e t D e p e n d e n c y
//

extern "C" void MtgGetDependency( char* sFileName, int nLayer )

{
    sendLayerInfo( sFileName, nLayer, tSummary::sendDependency );
}


//
//   M t g G e t B o u n d a r y
//

extern "C" void MtgGetBoundary( char* sFileName, int nLayer )

{
    sendLayerInfo( sFileName, nLayer, tSummary::sendBoundary );
}


//
//   M t g G e t V o l a t i l i t y
//

extern "C" void MtgGetVolatility( char* sFileName, int nLayer )

{
    sendLayerInfo( sFileName, nLayer, tSummary::sendVolatility );
}


//
//   M t g G e t T o t a l
//

extern "C" void MtgGetTotal( char* sFileName, int nLayer )

{
    sendLayerInfo( sFileName, nLayer, tSummary::sendTotal );
}


//
//   M t g G e t T o t a l L i s t
//

extern "C" void MtgGetTotalList( char* sFileName, int* Layer,
    long nNumOfLayers )

{
    sendLayerAggrInfo( sFileName, Layer, nNumOfLayers, tSummary::sendTotal );
}


//
//   M t g G e t M a y E x P o l i c y
//

extern "C" void MtgGetMayExPolicy( char* sFileName, int nLayer, int* Claim,
    long nNumOfClaims )

{
    sendLayerClaimInfo( sFileName, nLayer, Claim, nNumOfClaims,
        tSummary::sendMayExPolicy );
}


//
//   M t g G e t M a y E x P a y o f f
//

extern "C" void MtgGetMayExPayoff( char* sFileName, int nLayer, int* Claim,
    long nNumOfClaims )

{
    sendLayerClaimInfo( sFileName, nLayer, Claim, nNumOfClaims,
        tSummary::sendMayExPayoff );
}


//
//   M t g G e t F o r c e E x P o l i c y
//

extern "C" void MtgGetForceExPolicy( char* sFileName, int nLayer, int* Claim,
    long nNumOfClaims )

{
    sendLayerClaimInfo( sFileName, nLayer, Claim, nNumOfClaims,
        tSummary::sendForceExPolicy );
}


//
//   M t g G e t F o r c e E x P a y o f f
//

extern "C" void MtgGetForceExPayoff( char* sFileName, int nLayer, int* Claim,
    long nNumOfClaims )

{
    sendLayerClaimInfo( sFileName, nLayer, Claim, nNumOfClaims,
        tSummary::sendForceExPayoff );
}


//
//   M t g R u n S c r i p t
//

extern "C" void MtgRunScript( char* sScript, char* sProfile )

{
    tProfile Profile;
    tParser Parser;
    tRepository Rep;
    tEvaluate* pEvaluate;

    tHeap<tHeap<double>*> Result;
    tHeap<tHeap<double>*> Gradient;
    tHeap<tHeap2<double>*> Front;

    long nErrno;
    char *sErrMsg;
    int k;
    tRetCode nRet;

    MLINK MathLocal = MLLoopbackOpen( stdenv, &nErrno );
    /*if( nErrno != 0 ) {
        mathError( "unable to create loopback link" );
        return;
    }*/

    if( sProfile != 0 && strlen( sProfile ) > 0 ) {
        setDirectory();
        Profile.setFileName( sProfile );
        Profile.activate();
        if( ( nRet = Profile.openWrite() ) != OK ) {
            mathError( "unable to create profile" );
            MLClose( MathLocal );
            return;
        }
    }

    sScript = xlatString( sScript );
    tStringSource Source( sScript );

    Parser.setRepository( Rep );
    if( ( nRet = Parser.setSource( Source ) ) != OK ) {
        mathError( "unable to parse script" );
        goto end;
    }

    while( ( nRet = Parser.parseObjects( pEvaluate ) ) == OK &&
                pEvaluate != 0 ) {

        if( Profile.isActive() )
            pEvaluate->setProfile( Profile );

        tTimer Timer;
        if( ( nRet = pEvaluate->run() ) != OK ) {
            delete pEvaluate;
            break;
        }
        Timer.check();

        Result.append( new tHeap<double> );
        Result.last()->append( pEvaluate->total() );
        Result.last()->append( pEvaluate->delta() );
        Result.last()->append( pEvaluate->gamma() );
        Result.last()->append( Timer.toDouble() / 1000 );
        Result.last()->append( pEvaluate->numOfTasks() );

        Gradient.append( new tHeap<double> );
        pEvaluate->gradient( *Gradient.last() );

        Front.append( new tHeap2<double> );
        pEvaluate->front( *Front.last() );

        delete pEvaluate;
    }

    if( nRet != OK )
        goto error;

    MLPutFunction( MathLocal, "List", Result.numOfElems() );
    for( k = 0; k < Result.numOfElems(); ++k ) {
        MLPutFunction( MathLocal, "List", 3 );
        if( ( nRet = Result[k]->toMathLink( MathLocal ) ) != OK ||
            ( nRet = Gradient[k]->toMathLink( MathLocal ) ) != OK ||
            ( nRet = Front[k]->toMathLink( MathLocal ) ) != OK ) {
            mathError( "unable to send data" );
            goto end;
        }
    }

    MLTransferExpression( stdlink, MathLocal );
    MLEndPacket( stdlink );

    goto end;

error:

    sErrMsg = new char[strlen( Parser.getErrorMsg() ) + 256];

    sprintf( sErrMsg, "error %d in line %d: %s",
        nRet, Source.getCurLine(), Parser.getErrorMsg() );
    mathError( sErrMsg );
    delete sErrMsg;

end:

    if( Profile.isActive() )
        Profile.close();

    MLClose( MathLocal );

    for( int i = 0; i < Result.numOfElems(); ++i )
        delete Result[i];

    for( MTG_FOR_INIT( int ) i = 0; i < Gradient.numOfElems(); ++i )
        delete Gradient[i];

    for( MTG_FOR_INIT( int ) i = 0; i < Front.numOfElems(); ++i )
        delete Front[i];

    delete sScript;
}

#if defined(_WIN32)

//
//   W i n M a i n
//

int WINAPI WinMain( HINSTANCE hinstCurrent, HINSTANCE hinstPrevious,
	LPSTR lpszCmdLine, int nCmdShow )

{
		// Taken from the "addtwo" example in the MathLink documentation:

	char buff[512];
	char FAR* buff_start = buff;
	char FAR* argv[32];
	char FAR* FAR* argv_end = argv + 32;

	if( ! MLInitializeIcon( hinstCurrent, nCmdShow ) )
		return 1;
	MLScanString( argv, &argv_end, &lpszCmdLine, &buff_start );
	return MLMain( argv_end - argv, argv );
}

#else

//
//   m a i n
//

int main( int argc, char* argv[] )
	
{
	return MLMain( argc, argv );
}

#endif