// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

#include "MtgIncl.h"
#include "MtgCgi.h"
#include "MtgEvaluate.h"
#include "MtgLog.h"
#include "MtgNetServer.h"
#include "MtgParser.h"
#include "MtgJobServer.h"
#include "MtgProfile.h"
#include "MtgRepository.h"
#include "MtgService.h"
#include "MtgSocket.h"
#include "MtgSource.h"
#include "MtgTimer.h"
#include "MtgTclKernel.h"
#include "MtgVersion.h"

MTG_BEGIN_NAMESPACE


//
//   c o n s t a n t s
//

static const char* sBanner =
    "\nMtgSvr %d.%d (build %d)\n(C) Copyright Robert Buff 1997,98\n\n";

static const char* sHelp = 
    "Usage: MtgSvr {[option]} [script]\n\n"
    "where [option] is one of\n\n"
    "    -help             Display this message\n"
    "    -exhaustive       Carry all helper combinations\n"
    "    -log <FileName>   Redirect the log to <FileName>\n"
    "    -port <PortNum>   Listen at port <PortNum>\n"
    "    -profile          Create an execution profile\n"
    "\n";

static const char* sCgiMtgScript = "autocgi.mtg";
static const char* sCgiTclScript = "autocgi.tcl";
static const char* sCgiMtgRunTcl = "shell { tcl \"autocgi.tcl\" }";

 
//
//   v a r i a b l e s 
//

static bool bIsInteractive = true;


//
//   d e c l a r a t i o n s
//

static tRetCode useArgs( int& argc, const char* argv[] );
static void report( tEvaluate* pEvaluate, tTimer& Timer );
static tRetCode doFile( const char* sFileName, tProfile& profile );
static tRetCode doCgi();
static int main1( int argc, const char* argv[] );
static int main2( int argc, const char* argv[] );


//
//   u s e A r g s
//

static tRetCode useArgs( int& argc, const char* argv[] )

{
    bool bHelp = false;

    for( int k = 1; k < argc; ++k ) {
        if( strcmp( argv[k], "-help" ) == 0 ||
            strcmp( argv[k], "-?" ) == 0 ) {
            bHelp = true;
            argv[k] = 0;
        }
        else
            if( argv[k][0] == '-' ) {
                printf( "*** Unknown option \'%s\'\n\n", argv[k] );
                return INVALID_KEYWORD;
            }
    }

        // Remove used arguments.

    int j = 1;
    for( MTG_FOR_INIT( int ) k = 1; k < argc; ++k ) {
        if( argv[k] != 0 )
            argv[j++] = argv[k];
    }
    argc = j;

    return bHelp ? HELP_REQUESTED : OK;
}


//
//   r e p o r t
//

static void report( tEvaluate* pEvaluate, tTimer& Timer )

{
    try { 
        double gClean = pEvaluate->cleanTotal();
        double gDirty = pEvaluate->dirtyTotal();

        if( gClean != gDirty ) {
            cout << "Total: "
                 << gDirty << " (dirty), "
                 << gClean << " (clean)" << endl;
        }
        else {
            cout << "Total: " << gDirty << endl;
        }

        cout << "Delta: " << pEvaluate->delta() << endl;
        cout << "Gamma: " << pEvaluate->gamma() << endl;
    }
    catch( tException ) {
    }

    cout << "Time: " << flush;
    Timer.print();
    cout << endl;
}


//
//   d o F i l e
//

static tRetCode doFile( const char* sFileName, tProfile& Profile )

{
    tRetCode nRet;

    tRepository Rep;
    tFileSource Source( sFileName );
    tParser Parser;
    tEvaluate* pEvaluate;

    printf( "Processing script '%s'\n", sFileName );

    if( Profile.isActive() ) {
        Profile.useFileName( sFileName );
        if( ( nRet = Profile.openWrite() ) != OK ) {
            printf( "\n*** Cannot open '%s'\n\n", Profile.fileName() );
            return nRet;
        }
    }

    Parser.setRepository( Rep );
    if( ( nRet = Parser.setSource( Source ) ) != OK ) {
        if( nRet == OPEN_ERROR ) {
            printf( "\n*** Cannot open '%s'\n\n", sFileName );
        }
        else {
            printf( "\n*** Error %d : %s\n\n",
                nRet, Parser.getErrorMsg() );
        }
        goto end;
    }

    while( ( nRet = Parser.parseObjects( pEvaluate ) ) == OK &&
                pEvaluate != 0 ) {
        if( Profile.isActive() )
            pEvaluate->setProfile( Profile );

        tTimer Timer;

        if( ( nRet = pEvaluate->run() ) != OK ) {
            printf( "\n*** Runtime error %d\n\n", nRet );
            delete pEvaluate;
            goto end;
        }
        Timer.check();

        report( pEvaluate, Timer );
        delete pEvaluate;
    }

    if( nRet != OK ) {
        printf( "\n*** Error %d : %s\n\n", nRet, Parser.getErrorMsg() );
        goto end;
    }

    return OK;

end:

    if( Profile.isActive() )
        Profile.close();

    return nRet;
}


//
//   d o C g i
//

static tRetCode doCgi()

{
#if defined(_MTG_WITH_CGI)

    tRetCode nRet;

    if( ( nRet = tCgi::init() ) != OK ||
        ( nRet = tCgi::outHeaderContentType() ) != OK ) {
        return nRet;
    }

    //tCgi::readState();
    //tCgi::writeState();

    FILE* fp;

    tParser Parser;
    tRepository Rep;
    tEvaluate* pEvaluate;

    tFileSource FileSource;
    tStringSource StringSource;
    tSource* pSource = 0;

        // First check whether the mtg script exists; then
        // check whether a Tcl script exists; if neither
        // of them exists, report an error.

    if( ( fp = fopen( sCgiMtgScript, "r" ) ) != 0 ) {
        fclose( fp );
        FileSource.setInput( sCgiMtgScript );
        pSource = &FileSource;
    }

#if defined(_MTG_WITH_TCL)
    else
    if( ( fp = fopen( sCgiTclScript, "r" ) ) != 0 ) {
        fclose( fp );
        StringSource.setInput( sCgiMtgRunTcl ); 
        pSource = &StringSource;
    }
#endif

    if( pSource == 0 )
        goto error;

    Parser.setRepository( Rep );
    if( Parser.setSource( pSource ) )
        goto error;

        // Notice that the result of evaluation is nowhere reported;
        // this must be handled in the mtg or Tcl scripts directly.

    while( ( nRet = Parser.parseObjects( pEvaluate ) ) == OK &&
            pEvaluate != 0 ) {
        if( pEvaluate->run() != OK ) {
            delete pEvaluate;
            break;
        }
        delete pEvaluate;
    }

    if( nRet != OK ) {
        tCgi::out( "<p>Script error</p>" );
    }

    tCgi::exit();
    return nRet;

error:

    tCgi::out( "<p>Internal Server Error</p>" );
    tCgi::exit();

    return INTERNAL_ERROR;

#else

	return NOT_IMPLEMENTED;

#endif
}


//
//   m a i n 1
//

int main1( int argc, const char* argv[] )

{
#if defined(_MTG_WITH_CGI)

        // If this is a cgi request, cut all subsequent tests.

    if( tCgi::isActive() || argc == 2 && strcmp( argv[1], "-cgi" ) == 0 ) {
        bIsInteractive = false;

#if defined(_MTG_WITH_TCL)
        tTclKernel::init( argv[0], bIsInteractive );
#endif

        if( doCgi() != OK )
            return 1;
        return 0;
    }

#endif
        // Ok, we're started on the command line.

    tService Service;
    tRetCode nRet;

    printf( sBanner, VersionMajor, VersionMinor, VersionBuild );

    if( ( nRet = Service.useArgs( argc, argv ) ) != OK )
        return 1;

    int nErrCode = 0;

    switch( ( nRet = Service.perform( main2, argc, argv,
                        nErrCode, bIsInteractive ) ) ) {
        case GO :
            break;

        case OK :
            return nErrCode;

        default :
            printf( "*** Service request failed\n\n" );
            return 1;
    }

    return main2( argc, argv );
}


//
//   m a i n 2
//

int main2( int argc, const char* argv[] )

{
    tLog Log;
    tProfile Profile;
    tNetServer NetServer;
    tJobServer JobServer;

    tRetCode nRet;

#if defined(_MTG_WITH_TCL)
    tTclKernel::init( argv[0], bIsInteractive );
#endif

    if( ( nRet = Log.useArgs( argc, argv ) ) != OK ||
        ( nRet = Profile.useArgs( argc, argv ) ) != OK ||
        ( nRet = NetServer.useArgs( argc, argv ) ) != OK ||
        ( nRet = JobServer.useArgs( argc, argv ) ) != OK ) {
        return 1;
    }

    if( ( nRet = useArgs( argc, argv ) ) != OK ) {
        if( nRet == HELP_REQUESTED )
            printf( "%s", sHelp );
        return 1;
    }

    if( argc > 1 ) {
            // We also accept file scripts.
        for( int k = 1; k < argc; ++k ) {
            if( ( nRet = doFile( argv[k], Profile ) ) != OK )
                return 1;
        }         
    }               
    else {
            // The normal mode of operation is via the net
            // and through pipes.

        if( Log.fileName() != 0 ) {
            if( ( nRet = Log.open() ) != OK ) {
                printf( "*** Cannot open log '%s'\n\n", Log.fileName() );
                return 1;
            }
            NetServer.setLog( Log );
            JobServer.setLog( Log );
        }

        if( ( nRet = tSocket::startup() ) != OK ) {
            printf( "*** Cannot startup socket interface\n\n" );
            return 1;
        }

		// On Windows 98 there are no pipes; ignore
		// error messages w.r.t. pipes.

        JobServer.start();

//        if( ( nRet = JobServer.start() ) != OK ) {
//            tSocket::cleanup();
//            return 1;
//        }

        if( ( nRet = NetServer.main() ) != OK ) {
            JobServer.stop();
            tSocket::cleanup();
            return 1;
        }

        JobServer.stop();
        tSocket::cleanup();
    }

    return 0;
}

MTG_END_NAMESPACE

MTG_USING_NAMESPACE

//
//   m a i n
//

int main( int argc, const char* argv[] )

{
    InitMemory();
    atexit( ExitMemory );

        // Want all memory to be cleaned up at exit.

    return main1( argc, argv );
}
