// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

package nom.rb.MtgClt;
import nom.rb.common.*;

import java.io.*;
import java.net.*;


///////////////////////////////////////////////////////////////////

//
//   E v a l u a t e r
//

public class Evaluater extends Thread {


//
//   v a r i a b l e s
//

private ServerStub m_server;

private Object m_goLock = "lock";
private boolean m_go;

private String m_request;

private boolean m_paramSeller;
private boolean m_paramBuyer;
private boolean m_paramIsUncertain;
private int[] m_paramTimestep;
private double m_paramSpotPrice;

private int m_curTimestep;

private EvaluaterItem m_firstItem;
private EvaluaterItem m_lastItem;
private int m_numOfItems;
private int m_numOfSeqs;

private double[][] m_sellSamples;
private double[][] m_buySamples;


//
//   E v a l u a t e r
//

public Evaluater()

{
    m_server = new ServerStub( Pool.m_report,
        Pool.m_serverHost, Pool.m_serverPort );
}


//
//   s t a r t E x e c
//

public void startExec()

{
    if( ( m_request = Pool.m_workSpace.genRequest() ) == null )
        return;

    m_request += "bye\n";

    if( ! m_server.connectUI( Pool.m_frame ) )
        return;

    Pool.m_status.startTimer();

    if( ! request() ) {
        m_server.close();
        Pool.m_status.stopTimer();
        return;
    }
        
    m_go = true;
    Pool.m_menu.beforeRun();

    m_sellSamples = null;
    m_buySamples = null;

    start();
}


//
//   s t o p E x e c
//

public void stopExec()

{
    synchronized( m_goLock ) {
        if( m_go ) {
            m_go = false;
            try {  
                m_goLock.wait();
            }
            catch( InterruptedException e ) {
                // ignore
            }
        }
    }
}


//
//   r u n
//

public void run()

{
    response();

    m_server.close();
    Pool.m_menu.afterRun();

    Pool.m_status.setMsg( "" );
    Pool.m_status.stopTimer();

    synchronized( m_goLock ) {
        if( m_go )
            m_go = false;
        else
            m_goLock.notify();
    }
}


//
//   r e q u e s t
//

private boolean request()

{
    m_paramSeller = Pool.m_workSpace.paramSeller();
    m_paramBuyer = Pool.m_workSpace.paramBuyer();
    m_paramIsUncertain = Pool.m_workSpace.paramIsUncertain();
    m_paramTimestep = Pool.m_workSpace.paramTimestep();
    m_paramSpotPrice = Pool.m_workSpace.paramSpotPrice();

    try{ 
        m_server.write( m_request );
    }
    catch( IOException e ) {
        return showError( e );
    }

    return true;
}


//
//   r e s p o n s e
//

private void response()

{
    boolean on = true;

    m_firstItem = null;
    m_lastItem = null;
    m_numOfItems = 0;
    m_numOfSeqs = 0;
    m_curTimestep = 0;

    status();

    while( on ) {
        synchronized( m_goLock ) {
            if( ! m_go ) {
                on = false;
                Pool.m_report.println( "*** Abort on user request", true );
            }
        }

        if( on ) {
            String s = null;

            try {       
                s = m_server.readString();
            }
            catch( IOException e ) {
                on = false;
                showError( e );
            }

            if( on && s != null ) {
                s = s.trim();
                if( s.equals( "bye" ) )
                    on = false;
                else
                    on = chop( s );
            }
        }
    }

    if( m_sellSamples != null || m_buySamples != null ) {
        if( m_sellSamples == null ) 
            Pool.m_resultGraph.setSamples( m_paramSpotPrice, m_buySamples );
        else
        if( m_buySamples == null ) 
            Pool.m_resultGraph.setSamples( m_paramSpotPrice, m_sellSamples );
        else {
            Pool.m_resultGraph.setSamples( m_paramSpotPrice,
                m_sellSamples, m_buySamples );
        }
    }
}


//
//   c h o p
//

private boolean chop( String text )

{
    int j = 0;
    int k;

    do {
        k = text.indexOf( " ", j );

        int k1 = text.indexOf( "\n", j );
        if( k1 >= 0 && ( k < 0 || k1 < k ) )
            k = k1;

        String s = 
            ( k >= 0 ) ? text.substring( j, k ) : text.substring( j );

        s = s.trim();
        if( s.length() > 0 ) {
            boolean rep = false;

            if( s.equals( "end" ) ) {
                m_lastItem = new EvaluaterItem( null, m_lastItem );
                if( ++m_numOfItems < 4 ) {
                    Pool.m_report.println( "*** Protocol error", true );
                    return false;
                }
                ++m_numOfSeqs;
                rep = true;
            }
            else {
                Double d = null;

                try {
                    d = new Double( s );
                }
                catch( NumberFormatException e ) {
                    Pool.m_report.println( "*** Protocol error", true );
                    return false;
                }
                m_lastItem = new EvaluaterItem( d, m_lastItem );
                ++m_numOfItems;
            }

            if( m_firstItem == null )
                m_firstItem = m_lastItem;

            if( rep ) {
                if( ! report() )
                    return false;
            }
        }

        if( k >= 0 )
            j = k + 1;
    } while( k >= 0 );

    return true;
}


//
//   r e p o r t
//

private boolean report()

{
    if( m_numOfSeqs < 1 )
        return true;

    if( m_paramSeller && m_paramBuyer && m_numOfSeqs < 2 ) {
        status();
        return true;
    }

    String s = "";

    if( m_curTimestep == 0 )
        s += "\n" + header();

    s += formatTimestep( m_paramTimestep[m_curTimestep] );

    String q = null;

    if( m_paramSeller ) {
        q = "  " + formatTriple();
        if( m_firstItem.obj() == null ) {
            eatItem();
        }
        else {
            if( ( m_sellSamples = eatVector() ) == null ) {
                Pool.m_report.println( "*** Protocol error", true );
                return false;
            }
        }
    }

    if( m_paramBuyer ) {
        s += "  " + formatTriple();
        if( m_firstItem.obj() == null ) {
            eatItem();
        }
        else {
            if( ( m_buySamples = eatVector() ) == null ) {
                Pool.m_report.println( "*** Protocol error", true );
                return false;
            }
        }
    }

    if( m_paramSeller )
        s += q;

    if( m_curTimestep == m_paramTimestep.length - 1 )
        s += "\n";

    Pool.m_report.println( s, true );

    ++m_curTimestep;
    status();

    return true;
}


//
//   s t a t u s
//

private void status()

{
    String msg;

    if( m_curTimestep < m_paramTimestep.length ) {
        msg = "Evaluating ";
        if( m_paramIsUncertain ) {
            if( m_paramSeller && m_numOfSeqs < 1 )
                msg += "seller's position, ";
            else
                msg += "buyer's position, ";
        }
        else {
            msg += " position, ";
        }
        if( m_paramTimestep[m_curTimestep] > 0 ) {
            msg += m_paramTimestep[m_curTimestep] + " day timesteps";
        }
        else {
            if( m_paramTimestep[m_curTimestep] == -1 )
                msg += "1 timestep per day";
            else
                msg += -m_paramTimestep[m_curTimestep] + " timesteps per day";
        }
    }
    else {
        msg = "";
    }

    Pool.m_status.setMsg( msg );
}


//
//   h e a d e r
//

private String header()

{
    String s = "";

    if( m_paramIsUncertain ) {
        s += "    ";
        if( m_paramBuyer )
            s += "  - To avoid any risk.. ----------";
        if( m_paramSeller )
            s += "  - To avoid any risk.. ----------";
        s += "\n";
    }

    s += "Step";

    if( m_paramIsUncertain ) {
        if( m_paramBuyer )
            s += "  ..buy at     Delta      Gamma   ";
        if( m_paramSeller )
            s += "  ..sell at    Delta      Gamma";
        s += "\n";
    }
    else {
        s += "    Value      Delta      Gamma\n";
    }

    return s;
}


//
//   e a t I t e m
//

private Object eatItem()

{
    Object obj = m_firstItem.obj();

    m_firstItem = m_firstItem.next();
    if( m_firstItem == null )
        m_lastItem = null;
    --m_numOfItems;

    if( obj == null )
        --m_numOfSeqs;

    return obj;
}


//
//   e a t V e c t o r
//

private double[][] eatVector()

{
    int n = 0;
    EvaluaterItem p = m_firstItem;

    if( p == null )
        return null;

    while( p.obj() != null ) {
        p = p.next();
        if( p == null || p.obj() == null )
            return null;
        if( ( p = p.next() ) == null )
            return null;
        ++n;
    }

    double[][] v = new double[n][2];

    for( int i = 0; i < n; ++i ) {
        v[i][0] = ( (Double) eatItem() ).doubleValue();
        v[i][1] = ( (Double) eatItem() ).doubleValue();
    }
    eatItem();

    return v;
}


//
//   f o r m a t T r i p l e
//

private String formatTriple()

{
    double total = ( (Double) eatItem() ).doubleValue();
    double delta = ( (Double) eatItem() ).doubleValue();
    double gamma = ( (Double) eatItem() ).doubleValue();

    return formatDouble( total, 3, 6 ) + " " +
           formatDouble( delta, 3, 6 ) + " " +
           formatDouble( gamma, 3, 6 );
}


//
//   f o r m a t D o u b l e
//

private String formatDouble( double value, int before, int after )

{
    String s = Double.toString( value );

    if( s.indexOf( "e" ) >= 0 || s.indexOf( "E" ) >= 0 )
        return s;

    int k = s.indexOf( "." );
    int l = s.length() - k - 1;
    
    if( l < after )
        s += replicate( " ", after - l );
    else
        if( l > after ) 
            s = s.substring( 0, k + after + 1 );

    if( before > k ) {
        s = replicate( " ", before - k ) + s;
    }
    else
        if( before < k ) {
            int n = before + after + 1;
            if( n != k + 1 )
                s = s.substring( 0, n );
            else
                s = " " + s.substring( 0, k );
        }

    while( ( k = s.indexOf( "0 " ) ) > 0 && s.charAt( k - 1 ) != '.' )
        s = s.substring( 0, k ) + " " + s.substring( k + 1 );

    return s;
}


//
//   f o r m a t T i m e s t e p
//

private String formatTimestep( int timestep )

{
    String s;

    if( timestep > 0 )
        s = Integer.toString( timestep ) + "/";
    else
        s = "/" + Integer.toString( -timestep );

    return s + replicate( " ", 4 - s.length() );
}


//
//   r e p l i c a t e
//

private String replicate( String text, int times )

{
    String s = "";

    for( int i = 0; i < times; ++i )
        s += text;
    return s;
}


//
//   s h o w E r r o r
//

private boolean showError( IOException e )

{
    MessageBox.ok( Pool.m_frame, "Error", "Lost connection" );
    return false;
}

} // end of class


///////////////////////////////////////////////////////////////////

//
//   E v a l u a t e r I t e m
//

class EvaluaterItem extends Object {


//
//   v a r i a b l e s
//

private EvaluaterItem m_prev;
private EvaluaterItem m_next;

private Object m_obj;

//
//   E v a l u a t e r I t e m
//

public EvaluaterItem( Object obj, EvaluaterItem last )

{
    m_obj = obj;
    m_next = null;
    if( last != null )
        last.m_next = this;
}


//
//   n e x t
//

public EvaluaterItem next()

{
    return m_next;
}


//
//   o b j
//

public Object obj()

{
    return m_obj;
}

} // end of class
