// Copyright 1999, 2002 Robert Buff
// Contact: http://robertbuff.com/uvm
//
// This file is part of Mtg-Book.
//
// Mtg-Book is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation; either version 2 of the License,
// or (at your option) any later version.
//
// Mtg-Book is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Mtg-Book; if not, write to the 
//
// Free Software Foundation, Inc.
// 59 Temple Place, Suite 330
// Boston, MA 02111-1307
// USA

package nom.rb.MtgClt;
import nom.rb.common.*;


///////////////////////////////////////////////////////////////////

//
//   F o r w a r d V o l
//

public class ForwardVol extends Object {


//
//   v a r i a b l e s
//

private int m_maturity;

private double m_prior;
private double m_min;
private double m_max;


//
//   F o r w a r d V o l
//

public ForwardVol( ImpliedVol imp )
    throws DataException

{
    this( imp, 0, 0 );
}


//
//   F o r w a r d V o l
//

public ForwardVol( ImpliedVol imp, double plusBand, double minusBand )
    throws DataException

{
    m_maturity = imp.maturity();
    m_prior = imp.prior();
    m_min = imp.min() + minusBand;
    m_max = imp.max() + plusBand;

    if( m_min <= 0 ) {
        throw new DataException(
            "Resulting forward volatility is not positive " );
    }
}


//
//   F o r w a r d V o l
//

public ForwardVol( ImpliedVol imp1, ImpliedVol imp2 )
    throws DataException

{
    this( imp1, imp2, 0, 0 );
}


//
//   F o r w a r d V o l
//

public ForwardVol( ImpliedVol imp1, ImpliedVol imp2,
    double plusBand, double minusBand )
    throws DataException

{
    if( imp1.maturity() >= imp2.maturity() )
        throw new IllegalArgumentException();

    m_maturity = imp2.maturity();

    m_prior = calcForward( imp1.maturity(), imp1.prior(), imp2.prior() );
    m_min = calcForward( imp1.maturity(), imp1.min(), imp2.min() );
    m_max = calcForward( imp1.maturity(), imp1.max(), imp2.max() );

    if( m_prior < m_min || m_prior > m_max )
        throw new DataException( "Resulting forward band collapses" );

    m_max += plusBand;
    m_min += minusBand;

    if( m_min <= 0 ) {
        throw new DataException(
            "Resulting forward volatility is not positive " );
    }
}


//
//   i s U n c e r t a i n
//

public boolean isUncertain()

{
    return m_min < m_max;
}


//
//   t o S t r i n g
//

public String toString()

{
    String s = "forward " + m_maturity + " ";

    if( m_min == m_prior && m_max == m_prior )
        s += m_prior + "%";
    else
        s += m_min + "%.." + m_prior + "%.." + m_max + "%";

    return s;
}


//
//   c a l c F o r w a r d
//

private double calcForward( int mat1, double imp1, double imp2 )
    throws DataException

{
    double fwd = ( m_maturity * imp2 * imp2 - mat1 * imp1 * imp1 ) /
                    ( m_maturity - mat1 );
    if( fwd <= 0 ) {
        throw new DataException(
            "Resulting forward volatility is not positive " );
    }
    return Math.sqrt( fwd );
}

} // end of class