///////////////////////////////////////////////////////////////
//
// MosaicLayout.java 1.0 97/02/05 Robert Buff buff@cs.nyu.edu
//                   1.1 97/06/19 Robert Buff buff@cs.nyu.edu
//                   1.2 97/07/10 Robert Buff buff@cs.nyu.edu
//                   1.3 97/07/30 Robert Buff buff@cs.nyu.edu
//                   1.4 98/01/30 Robert Buff buff@cs.nyu.edu
//
// Copyright (c) 1997, 98 Robert Buff. All Rights Reserved.
//
// You may use, copy, modify, and distribute this software
// for NON-COMMERCIAL purposes and without fee provided that
// this copyright notice appears in all copies. If you would
// like to use this software for COMMERCIAL purposes, contact
// the author and we will work something out.
//
// It would also be nice to give credit to the author in the
// Help/About dialog window of any application that uses this
// software.
//
///////////////////////////////////////////////////////////////
//
// This class defines a layout manager that a) organizes its
// components in a hierarchical manner, and b) allows the user
// to resize its components by clicking on and dragging of sashes.
//
// The layout manager mirrors the subdivision of the panel space 
// in a tree whose leaf nodes point to the components, and whose
// interior nodes correspond to rectangular grids.  The columns
// and rows of each interior grid can be resized by the user by
// dragging the sashes that are inserted between adjacent columns
// and rows.
//
// In addition, the layout manager follows a dynamic resizing
// policy that is activated when the underlying panel changes
// its size.  Weights can be assigned to each component; these
// weights, together with the minimum and preferred size of
// the components, determine how the actual size of each
// component changes.
//
// Supporting classes:
//
//   MosaicConstraints: its relation to MosaicLayout is similar
//       to the relation of GridBagConstraints to GridBagLayout.
//       For each component, weights and gravity parameters can
//       be specified.
//
//   MosaicCoordStack: contains the position that is assigned to
//       a component.  Basically, a sequence of (column/row)
//       pairs whose length corresponds to the level of the
//       component in the hierarchy.  Describes the path to
//       the component in the tree.
//
//   MosaicPanel: a subclass of Panel that notifies the layout
//       manager of mouse events and paint requests, for it is
//       the layout manager's responsibility to draw the sashes
//       and handle resize requests delivered by mouse events.
//       If the parent of the layout manager is NOT a MosaicPanel,
//       the layout manager is still functioning, but does not
//       draw sashes any more. The possibility of manually
//       rearranging the components is then no longer given.
//       The layout manager checks for the type of its parent
//       and acts accordingly.
//
// The sashes are painted directly on the parent MosaicPanel,
// without additional components.  The look of the sashes comes
// close to the look of sashes in Windows environments.  I didn't
// bother to make them look nice on black-and-white monitors.
//
// Resizing is done online - while dragging - by default, but this
// can be switched off if repainting the components is expensive
// (see m_liveDrag).
//
// No component is ever made smaller than its minimum size,
// even if the underlying panel gets smaller.  However, components
// can be manually made larger than their preferred size even if
// their assigned weight is zero.
//
// It is possible to create named constraints and load them into
// a dictionary that the layout manager maintains.  Later on,
// these constraints can be assigned to any number of components
// by specifying their name in the add( name, comp ) version of
// the add function of the underlying container panel.
//
// In general, the features of the layout manager can be discovered
// best by studying the commented public member functions of the
// class MosaicLayout, listed in the top section of the class
// definition.  Other features and strategies are only sparsely
// commented.
//
// The following is an example of how to use MosaicLayout. Note
// that the position of components can be indicated be submitting
// MosaicCoordStack objects directly to the layout manager, or
// by calling the layout manager's setPos() function which
// essentially creates a MosaicCoordStack object for the next
// component that is registered with the add( name, comp )
// function of the underlying panel (you must use the version
// that supplies a name in this case, as the other version
// delays transfering the data to the layout manager, thus
// disrupting the order).
//
// The example without using MosaicCoordStack directly (order
// is crucial):
//
//   MosaicPanel m_panel = new MosaicPanel();
//   MosaicLayout m_layout = new MosaicLayout();
//
//   setLayout( new BorderLayout() );
//   add( "Center", m_panel );
//   m_panel.setLayout( m_layout );
//
//   Button button1 = new Button( "Monday" );
//   Button button2 = new Button( "Tuesday" );
//   Button button3 = new Button( "Wednesday" );
//   Button button4 = new Button( "Thursday" );
//   Button button5 = new Button( "Friday" );
//   Button button6 = new Button( "Saturday" );
//   Button button7 = new Button( "Sunday" );
//
//   m_layout.setPos( 0, 0, 0, 0 );
//   m_panel.add( "", button1 );
//
//   m_layout.setPos( 0, 0, 1, 0, 0, 0 );
//   m_panel.add( "", button2 );
//
//   m_layout.setPos( 0, 0, 1, 0, 1, 0 );
//   m_panel.add( "", button3 );
//
//   m_layout.setPos( 0, 0, 1, 0, 0, 1 );
//   m_panel.add( "", button4 );
//
//   m_layout.setPos( 0, 0, 1, 0, 1, 1 );
//   m_panel.add( "", button5 );
//
//   m_layout.setPos( 0, 1, 0, 0 );
//   m_panel.add( "", button6 );
//
//   m_layout.setPos( 0, 1, 1, 0 );
//   m_panel.add( "", button7 );
//
//   m_layout.setConstraints( button3, new MosaicConstraints(
//     MosaicConstraints.CENTER, MosaicConstraints.HORIZONTAL ) );
//
//   m_layout.setConstraints( button5, new MosaicConstraints(
//     MosaicConstraints.CENTER, MosaicConstraints.HORIZONTAL ) );
//
// The example when MosaicCoordStack is used (the order
// does not matter any more; only the middle section of
// the code is repeated):
//
//   m_panel.add( button1 );
//   m_panel.add( button2 );
//   m_panel.add( button3 );
//   m_panel.add( button4 );
//   m_panel.add( button5 );
//   m_panel.add( button6 );
//   m_panel.add( button7 );
//
//   MosaicCoordStack c;
//	 
//   c.setPos( 0, 0, 0, 0 );
//   m_layout.setCoordStack( button1, c );
//   c.setPos( 0, 0, 1, 0, 0, 0 );
//   m_layout.setCoordStack( button2, c );
//   c.setPos( 0, 0, 1, 0, 1, 0 );
//   m_layout.setCoordStack( button3, c );
//   c.setPos( 0, 0, 1, 0, 0, 1 );
//   m_layout.setCoordStack( button4, c );
//   c.setPos( 0, 0, 1, 0, 1, 1 );
//   m_layout.setCoordStack( button5, c );
//   c.setPos( 0, 1, 0, 0 );
//   m_layout.setCoordStack( button6, c );
//   c.setPos( 0, 1, 1, 0 );
//   m_layout.setCoordStack( button7, c );
//
// No guarantee is made that this code actually does what
// has been described above.  This is a first version that
// surely contains bugs.
//
///////////////////////////////////////////////////////////////
//
// History:
//
//
//  1.1  Default cursor is activated in empty tiles.
//       layout() is now called recursively for all components.
//
//  1.2  checkPanel() is now called in minimumLayoutSize() and
//       preferredLayoutSize. These functions are called before
//       layoutContainer() by parent layout managers.
//
//  1.3  After dragging the sashes, the event MOUSE_UP triggers
//       a final recursion in the component tree, to determine the
//       correct cursor shape. Note, however, that (at least in
//       my version of VJ++) the default shape becomes only 
//       active again after the mouse is moved. Don't know why.
//
//  1.4  Minor bugfix: now calling validate() instead of
//       layout() in 1.1.
//
///////////////////////////////////////////////////////////////

package nom.rb.common;

import java.awt.*;
import java.util.*;

import nom.rb.common.MosaicConstraints;
import nom.rb.common.MosaicCoordStack;
import nom.rb.common.MosaicPanel;

///////////////////////////////////////////////////////////////
//
//   M o s a i c L a y o u t
//
///////////////////////////////////////////////////////////////

public class MosaicLayout extends Object implements LayoutManager
{
  // The layout manager maintains a repository of reusable
  // constraint objects, that can be assigned to components
  // many times.  They are identified by their unique name
  // and stored in the following hashtable.

  protected Hashtable m_name2Constraints = null;

  // Each component is represented by a leaf node in the tree
  // that mirrors the hierarchy.  The leaf node that corresponds
  // to a particular component can be found in this hashtable.

  protected Hashtable m_comp2Node = null;

  // Whenever a MosaicCoordStack object is passed to the layout
  // manager, it is copied and the column counter at its lowest
  // level is advanced to the next column to the right.  When a
  // new component is registered, this generated MosaicCoordStack
  // object is taken as its default position.

  protected MosaicCoordStack m_lastCoordStack = null;
  protected MosaicCoordStack m_nextCoordStack = null;

  // This is the root node of the hierarchy tree.

  protected MosaicNode m_root = null;

  // If the parent of the layout manager is a MosaicPanel,
  // then sashes are displayed between components that can
  // be grabbed by the user in order to resize the components.
  // In addition to that, the layout manager tries to detect
  // the frame that contains the panel, because cursor types
  // can only be changed for frames.
  // If the parent of the layout manager is not a MosaicPanel,
  // m_panel is set to null.

  protected MosaicPanel m_panel = null;
  protected Frame m_frame = null;

  // Dragging does always refer to a particular interior
  // node in the hierarchy.  Which one?  This one:

  protected MosaicInteriorNode m_activeNode = null;
  
  // The cursor changes its shape on sashes (if the frame of
  // the parent could be found) according to the following
  // values.  On Windows systems, a different shape is chosen
  // for column and row sashes and for intersections.  On all
  // other systems, the cursor is uniformly changed to
  // MOVE_CURSOR.

  protected int colSashCursor = Frame.DEFAULT_CURSOR;
  protected int rowSashCursor = Frame.DEFAULT_CURSOR;
  protected int crossSashCursor = Frame.DEFAULT_CURSOR;

  // The size of sashes can be set for top level sashes and
  // sashes on all other levels of the hierarchy separately.

  protected int m_topLevelSashSize = 7;
  protected int m_medLevelSashSize = 7;

  // The following variable controls whether the repositioning
  // that results from dragging operations is done while the
  // dragging is going on, thus giving online feedback,
  // or if it should be delayed until the mouse button is
  // released (better for components that take a long time to
  // draw).  Note that if the latter is chosen, there is no
  // hint like outlined sashes.

  protected boolean m_liveDrag = true;

  // Remember if the tree has been constructed (m_initialized)
  // and the components have been reshaped (m_finalized).

  protected boolean m_initialized = false;
  protected boolean m_finalized = false;

  //
  //   M o s a i c L a y o u t
  //

  // The constructor. MosaicConstraints object are registered
  // under the name "Default" and "Centered."  The latter one
  // centers its component at its minimum size.

  public MosaicLayout()
  {
    m_name2Constraints = new Hashtable();
    m_comp2Node = new Hashtable();
    setConstraints( "Default", new MosaicConstraints() );
    setConstraints( "Centered",
      new MosaicConstraints( MosaicConstraints.CENTER,
        MosaicConstraints.NONE ) );
  }

  //
  //   a d d L a y o u t C o m p o n e n t
  //

  // One of the required callbacks of a layout manager.
  // The component comp is added with the constraints that have
  // previously been registered under the specified name.  Its
  // position in the hierarchy is initialized as being to
  // the right of the last position that has been registered
  // or generated.  If the component has already been
  // added (meaning it is found in the m_comp2Node hashtable),
  // only the constraints are changed; its position remains
  // unchanged.

  public void addLayoutComponent( String name, Component comp )
  {
    if( m_comp2Node.containsKey( comp ) ) {
      lookupNode( comp ).m_constraints = newConstraints( name );
    }
    else {
      m_comp2Node.put( comp, new MosaicLeafNode( this, comp,
        newConstraints( name ), newCoordStack() ) );
    }
    m_initialized = m_finalized = false;
  }

  //
  //   l a y o u t C o n t a i n e r
  //

  // One of the required callbacks of a layout manager.
  // The hierarchy tree is generated from all the components
  // that belong to the container parent.  This contains all
  // the components that have been registered with the
  // addLayoutComponent() function, but also others.  For
  // those others that cannot be found in the hashtable
  // m_comp2Node, leaf nodes are generated on the fly.
  // (Their constraints and positions - coord stacks - are
  // generated on the fly as well.)  Note that one can
  // very well register the constraints or position of
  // a component that has not been added with
  // addLayoutComponent(); as a general rule, leaf nodes
  // are created whenever necessary.

  public void layoutContainer( Container parent )
  {
    checkPanel( parent );
    initializeGeometry( parent );
    if( m_root != null ) {
      Insets insets = parent.insets();
      Dimension size = parent.size();
      m_root.finalizeGeometry( insets.left, insets.top,
        size.width - insets.left - insets.right,
        size.height - insets.top - insets.bottom, true );
    }
    m_finalized = true;
  }

  //
  //   m i n i m u m L a y o u t S i z e
  //

  // One of the required callbacks of a layout manager.
  // Returns the overall minimum size derived from the
  // minimum sizes of all components.  The tree is
  // generated first if it doesn't exist yet.

  public Dimension minimumLayoutSize( Container parent )
  {
    checkPanel( parent );
    initializeGeometry( parent );
    if( m_root == null )
      return new Dimension( 0, 0 );
    return m_root.m_minSize;
  }

  //
  //   p r e f e r r e d L a y o u t S i z e
  //

  // One of the required callbacks of a layout manager.
  // Returns the overall preferred size derived from the
  // preferred sizes of all components.  The tree is
  // generated first if it doesn't exist yet.

  public Dimension preferredLayoutSize( Container parent )
  {
    checkPanel( parent );
    initializeGeometry( parent );
    if( m_root == null )
      return new Dimension( 0, 0 );
    return m_root.m_prefSize;
  }

  //
  //   r e m o v e L a y o u t C o m p o n e n t
  //

  // One of the required callbacks of a layout manager.
  // The component comp is removed from the hashtable
  // m_comp2Node, and is thus no longer part of the
  // construction.  The hierachy tree is marked as 
  // invalid; it is rebuilt when layoutContainer() is
  // called later on.

  public void removeLayoutComponent( Component comp )
  {
    if( m_comp2Node.containsKey( comp ) ) {
      m_comp2Node.remove( comp );
      m_initialized = m_finalized = false;
    }
  }

  //
  //   s e t C o n s t r a i n t s
  //

  // This method defines the default constraints.  They
  // are stored under the name "Default" in the constraints
  // repository m_name2Constraints.

  public void setConstraints( MosaicConstraints constraints )
  {
    setConstraints( "Default", constraints );
  }

  //
  //   s e t C o n s t r a i n t s
  //

  // Inserts constraints into the repository of constraints,
  // under the given name.  These constraints can then later
  // be reused and assigned to components.  Existing
  // constraints with the same name are overwritten (only
  // in the repository!).

  public void setConstraints( String name, 
    MosaicConstraints constraints )
  {
    m_name2Constraints.put( name, constraints.clone() );
  }

  //
  //   s e t C o n s t r a i n t s
  //

  // Defines the constraints for component comp.  If no
  // leaf node for component comp exists yet, a new one
  // is created, and its position in the tree (its coord
  // stack) is guessed from the last position encountered.

  public void setConstraints( Component comp, 
    MosaicConstraints constraints )
  {
    if( ! m_comp2Node.containsKey( comp ) ) {
      m_comp2Node.put( comp, new MosaicLeafNode( this, comp,
        (MosaicConstraints) constraints.clone(), newCoordStack() ) );
    }
    else {
      MosaicLeafNode N = lookupNode( comp );
      N.m_constraints = (MosaicConstraints) constraints.clone();
      N.m_initialized = false;
    }
    m_initialized = m_finalized = false;
  }

  //
  //   g e t C o n s t r a i n t s
  //

  // Retrieves a copy of the default constraints.

  public MosaicConstraints getConstraints()
  {
    return getConstraints( "Default" );
  }

  //
  //   g e t C o n s t r a i n t s
  //

  // Retrieves a copy of the constraints with the specified
  // name.  If no constraints with that name exist,
  // null is returned.

  public MosaicConstraints getConstraints( String name )
  {
    MosaicConstraints c = lookupConstraints( name );
    if( c != null )
      return (MosaicConstraints) c.clone();
    return null;
  }

  //
  //   g e t C o n s t r a i n t s
  //

  // Retrieves a copy of the constraints of the component
  // comp.  The component comp has to be known to the
  // layout manager.

  public MosaicConstraints getConstraints( Component comp )
    throws NoSuchElementException
  {
    MosaicLeafNode N = lookupNode( comp );
    if( N == null )
      throw new NoSuchElementException();
    return (MosaicConstraints) N.m_constraints.clone();
  }

  //
  //   s e t C o o r d S t a c k
  //

  // Sets the position (coord stack) of the next component
  // for which a leaf node is generated (that is the next
  // component that is registered with addLayoutComponent(),
  // or that is not known yet to the layout manager and for
  // which a leaf node is generated when its constraints
  // are submitted).  One of the ways to build a hierarchy
  // is to have pairs like
  //
  //    m_layout.setCoordStack( ... );
  //    add( "", comp );
  //
  // in the code as shown in the example above.  Note that
  // the version of add() that takes a name as its first
  // argument calls addLayoutComponent(), whereas the version
  // that doesn't take a name does not.

  public void setCoordStack( MosaicCoordStack coordStack )
  {
    m_nextCoordStack = (MosaicCoordStack) coordStack.clone();
  }

  //
  //   s e t C o o r d S t a c k
  //

  // An alternative way to specify the position of each component
  // is to define it directly.  This works even if the component
  // has not been added to the parent container.  The example
  // in the introduction does not demonstrates this method, however,
  // but it is straightforward to apply.

  public void setCoordStack( Component comp, MosaicCoordStack coordStack )
  {
    m_lastCoordStack = (MosaicCoordStack) coordStack.clone();
    m_nextCoordStack = null;
    if( ! m_comp2Node.containsKey( comp ) ) {
      m_comp2Node.put( comp, new MosaicLeafNode( this, comp,
        newConstraints( "Default" ), m_lastCoordStack ) );
    }
    else {
      lookupNode( comp ).m_coordStack = m_lastCoordStack;
    }
    m_initialized = m_finalized = false;
  }

  //
  //   g e t C o o r d S t a c k
  //

  // Retrieves a copy of the position or coord stack of the
  // given component.

  public MosaicCoordStack getCoordStack( Component comp )
    throws NoSuchElementException
  {
    MosaicLeafNode N = lookupNode( comp );
    if( N == null )
      throw new NoSuchElementException();
    return (MosaicCoordStack) N.m_coordStack.clone();
  }

  //
  //   s e t P o s
  //

  // The layout manager maintains a default coord stack that
  // is used whenever a new leaf node is generated and no
  // other coord stack is available.  (The coord stack of 
  // components that have been positioned in this default manner
  // can be overwritten at any time, of course.)  This default
  // coord stack can be manipulated with the following functions.
  // The first function sets the position to the specified
  // row and column at the top level of the hierarchy.

  public void setPos( int col, int row )
  {
    m_nextCoordStack = new MosaicCoordStack();
    m_nextCoordStack.setPos( col, row );
  }

  //
  //   s e t P o s
  //

  // This function creates a default position at the second
  // level; col1/row1 and col2/row2 form the path that has
  // to be traversed while visiting the top and second level
  // interior grid nodes in order to arrive at the destination.

  public void setPos( int col1, int row1, int col2, int row2 )
  {
    m_nextCoordStack = new MosaicCoordStack();
    m_nextCoordStack.setPos( col1, row1, col2, row2 );
  }

  //
  //   s e t P o s
  //

  // This function creates a default position at the third
  // level; col1/row1, col2/row2 and col3/row3 form the path
  // that has to be traversed while visiting the top, second
  // and third level interior grid nodes in order to arrive
  // at the destination.

  public void setPos( int col1, int row1, int col2, int row2,
    int col3, int row3 )
  {
    m_nextCoordStack = new MosaicCoordStack();
    m_nextCoordStack.setPos( col1, row1, col2, row2 );
    m_nextCoordStack.pushPos( col3, row3 );
  }

  //
  //   s e t P o s
  //

  // This function creates a default position at the forth
  // level, analogous to the preceding functions.

  public void setPos( int col1, int row1, int col2, int row2,
    int col3, int row3, int col4, int row4 )
  {
    m_nextCoordStack = new MosaicCoordStack();
    m_nextCoordStack.setPos( col1, row1, col2, row2 );
    m_nextCoordStack.pushPos( col3, row3 );
    m_nextCoordStack.pushPos( col4, row4 );
  }

  //
  //   g e t S a s h S i z e
  //

  protected int getSashSize( boolean topLevel )
  {
    if( m_panel == null )
      return 0;
    return topLevel ? m_topLevelSashSize : m_medLevelSashSize;
  }

  //
  //   c h e c k P a n e l
  //

  protected void checkPanel( Container parent )
  {
    if( m_panel != null )
		return;

    if( ! ( parent instanceof MosaicPanel ) ) {
      m_panel = null;
      m_frame = null;
      return;
    }

    m_panel = (MosaicPanel) parent;
    while( parent != null && ! ( parent instanceof Frame ) )
      parent = parent.getParent();
    m_frame = (Frame) parent;

    if( m_frame != null ) {
      if( System.getProperty( "os.name" ).startsWith( "Windows" ) ) {
        colSashCursor = Frame.E_RESIZE_CURSOR;
        rowSashCursor = Frame.N_RESIZE_CURSOR;
        crossSashCursor = Frame.MOVE_CURSOR;
      }
      else {
        colSashCursor = Frame.MOVE_CURSOR;
        rowSashCursor = Frame.MOVE_CURSOR;
        crossSashCursor = Frame.MOVE_CURSOR;
      }
    }
  }

  //
  //   h a n d l e E v e n t
  //

  // The events are received from the MosaicPanel parent of
  // the layout manager.

  protected boolean handleEvent( Event evt )
  {
    if( ! m_finalized || m_root == null || evt.target != m_panel )
      return false;

    switch( evt.id ) {
      case Event.MOUSE_EXIT:
		   if( m_activeNode == null )
             defaultCursor();
      	   break;

      case Event.MOUSE_ENTER:
      case Event.MOUSE_MOVE:
      case Event.MOUSE_DOWN:
		   if( m_activeNode == null )
             m_root.handleEvent( evt );
           break;

      case Event.MOUSE_UP:
		   if( m_activeNode != null ) {
             m_activeNode.drag( evt.x, evt.y );
             m_activeNode.releaseSash();
				// take care of cursor
			 m_root.handleEvent( evt );
			 m_panel.validate();
           }
           break;

      case Event.MOUSE_DRAG:
           if( m_activeNode != null )
             m_activeNode.drag( evt.x, evt.y );
           break;
    }

    return true;
  }

  //
  //   p a i n t
  //

  // The paint request comes from the MosaicPanel parent of
  // the layout manager.

  protected void paint( Graphics g )
  {
    if( m_finalized && m_root != null )
      m_root.paint( g );
  }

  //
  //   l o o k u p C o n s t r a i n t s
  //

  protected MosaicConstraints lookupConstraints( String name )
  {
    return (MosaicConstraints) m_name2Constraints.get( name );
  }

  //
  //   l o o k u p N o d e
  //

  protected MosaicLeafNode lookupNode( Component comp )
  {
    return (MosaicLeafNode) m_comp2Node.get( comp );
  }

  //
  //   n e w C o n s t r a i n t s
  //

  protected MosaicConstraints newConstraints( String name )
  {
    MosaicConstraints constraints = getConstraints( name );
    if( constraints == null ) {
      constraints = getConstraints( "Default" );
      if( constraints == null )
        constraints = new MosaicConstraints();
    }
    return constraints;
  }

  //
  //   n e w C o o r d S t a c k
  //

  protected MosaicCoordStack newCoordStack()
  {
    if( m_nextCoordStack != null ) {
      m_lastCoordStack = m_nextCoordStack;
      m_nextCoordStack = null;
    }
    else
      if( m_lastCoordStack != null ) {
        m_lastCoordStack = (MosaicCoordStack) m_lastCoordStack.clone();
        m_lastCoordStack.nextCol();
      }
    else {
      m_lastCoordStack = new MosaicCoordStack();
    }
    return m_lastCoordStack;
  }

  //
  //   i n i t i a l i z e G e o m e t r y
  //

  // This function generates the hierarchy tree.  There are
  // two cases where the tree has to be generated:
  // a) m_initialized == false.  This is the case if new
  // components, constraints or coord stacks have been
  // registered.  b) A new component that is not yet known
  // to the layout manager has been added to the parent panel,
  // by using the add( comp ) function.  Because *all*
  // components that are children of the parent panel are
  // returned by getComponents(), the second case can be
  // discovered as well.

  protected void initializeGeometry( Container parent )
  {
    int i;

    Component comp[] = parent.getComponents();

    for( i = 0; i < comp.length; ++i ) {
      if( ! m_comp2Node.containsKey( comp[i] ) ) {
        addLayoutComponent( "", comp[i] );
        m_initialized = false;
      }
    }

    if( m_initialized )
      return;

    m_root = null;

    for( i = 0; i < comp.length; ++i )
      addNode( lookupNode( comp[i] ) );

    if( m_root != null )
      m_root.initializeGeometry();

    m_initialized = true;
    //printTree();
  }

  //
  //   a d d N o d e
  //

  protected void addNode( MosaicLeafNode N )
  {
    Enumeration coord = N.m_coordStack.elements();

    MosaicCoord c =
      coord.hasMoreElements() ?
        (MosaicCoord) coord.nextElement() : new MosaicCoord( 0, 0 );

    if( m_root == null ) {
      if( ! coord.hasMoreElements() && c.m_col == 0 && c.m_row == 0 ) {
        N.setParent();
        m_root = N;
        return;
      }
      m_root = new MosaicInteriorNode( this );
    }

    MosaicInteriorNode parent = 
      m_root.ensureCapacity( c.m_col, c.m_row );

    m_root = parent;

    while( coord.hasMoreElements() ) {
      MosaicNode child = parent.m_node[c.m_col][c.m_row];
      MosaicCoord c1 = (MosaicCoord) coord.nextElement();

      if( child == null && ! coord.hasMoreElements() &&
          c1.m_col == 0 && c1.m_row == 0 ) {
        break;
      }

      if( child == null ) {
        child = new MosaicInteriorNode( this );
        child.setParent( parent, c.m_col, c.m_row );
      }

      parent = child.ensureCapacity( c1.m_col, c1.m_row );
      c = c1;
    }

    parent.m_node[c.m_col][c.m_row] = N;
    N.setParent( parent, c.m_col, c.m_row );
  }

  //
  //   d e f a u l t C u r s o r
  //

  protected void defaultCursor()
  {
    if( m_frame != null ) {
      if( m_frame.getCursorType() != Frame.DEFAULT_CURSOR )
        m_frame.setCursor( Frame.DEFAULT_CURSOR );
    }
  }

  //
  //   p r i n t T r e e 
  //

  protected void printTree()
  {
    if( m_root != null )
      m_root.print( "" );
  }
}

///////////////////////////////////////////////////////////////
//
//   M o s a i c N o d e
//
///////////////////////////////////////////////////////////////

abstract class MosaicNode extends Object
{
  // For each node, its minimum size, preferred size and actual
  // size are stored separately.  Minimum and preferred sizes are
  // fixed and initialized only once.  The actual size can vary.
  // The minimum size of a grid is the sum of the minimum size 
  // of all its columns/rows, plus the space that is needed for
  // sashes.  The same holds for the preferred and actual size.
  // The minimum/preferred size of a column/row is the maximum
  // of the minimum/preferred size of each box in the column/row,
  // in the corresponding dimension.  The actual size of a node
  // is never less than its minimum size.

  protected Dimension m_minSize = null;
  protected Dimension m_prefSize = null;
  protected Dimension m_size = null;

  // There are two passes.  During initialization, the minimum
  // and preferred sizes and weights are determined and
  // propagated bottom-up.  During finalization, the available
  // space is subdivided into regions according to the weights
  // and passed down (top-down).  Initialization takes place only
  // once.  Finalization takes place whenever the size of the
  // container changes.

  // During finalization, the origin relative to the parent
  // container is determined and stored here.

  protected int m_orgx = 0;
  protected int m_orgy = 0;

  // Each node has a weight assigned to it, separately for each
  // dimension.  The weight determines how much the grid wins or
  // loose when extra space is available or needed.  A weight of
  // zero means the grid never wins or looses any space.  The
  // higher the weight, the more prominent the node when it comes
  // to the distribution of extra space.  The weight of leaf nodes
  // is looked up in the constraints of the corresponding component;
  // the weight of interior nodes is computed recursively by
  // averaging over the weight parameters of its children, giving
  // higher priority to children whose preferred size is higher.

  protected double m_weightx = 0;
  protected double m_weighty = 0;

  // Each node knows its parent node and the position it takes
  // there.

  protected MosaicInteriorNode m_parent = null;
  protected int m_parentCol = 0;
  protected int m_parentRow = 0;

  // Each node also has direct contact to the layout manager,
  // since it has to access some of its parameters.

  protected MosaicLayout m_layout = null;

  //
  //   M o s a i c N o d e
  //

  protected MosaicNode( MosaicLayout layout )
  {
    m_layout = layout;
  }

  //
  //   i n i t i a l i z e G e o m e t r y
  //

  protected abstract void initializeGeometry();

  //
  //   f i n a l i z e G e o m e t r y
  //

  protected abstract void finalizeGeometry( int orgx, int orgy,
    int width, int height, boolean liveDrag );

  //
  //   e n s u r e C a p a c i t y
  //

  protected abstract MosaicInteriorNode ensureCapacity( 
    int col, int row );

  //
  //   s e t P a r e n t
  //

  protected void setParent()
  {
    setParent( null, 0, 0 );
  }

  //
  //   s e t P a r e n t
  //

  protected void setParent( MosaicInteriorNode parent,
    int col, int row )
  {
    m_parent = parent;
    m_parentCol = col;
    m_parentRow = row;
    if( m_parent != null )
      m_parent.m_node[m_parentCol][m_parentRow] = this;
  }

  //
  //   h a n d l e E v e n t
  //

  protected abstract void handleEvent( Event evt );

  //
  //   p a i n t
  //

  protected abstract void paint( Graphics g );

  //
  //   p r i n t
  //

  protected void print( String indent )
  {
    if( m_parent == null ) {
      System.out.println( indent + "no parent" );
    }
    else {
      if( m_parent.m_node[m_parentCol][m_parentRow] != this ) {
        System.out.println( indent +
          "parent (" + m_parentCol + "," + m_parentRow + ") not found" );
      }
      else {
        System.out.println( indent +
          "parent (" + m_parentCol + "," + m_parentRow + ") found" );
      }
    }
  }
}

///////////////////////////////////////////////////////////////
//
//   M o s a i c L e a f N o d e
//
///////////////////////////////////////////////////////////////

final class MosaicLeafNode extends MosaicNode
{
  // There is a one-to-one correspondence between leaf nodes and
  // components.  Leaf nodes are the wrappers of components and their
  // layout constraints.  The position (coord stack) of components in
  // the hierarchy can also be looked up in the corresponding leaf node.
  // The coord stack directly corresponds to the path that has to be
  // followed in the tree in order to arrive at the leaf node (the
  // tree is constructed by consulting the coord stacks).
  // Note that an empty leaf node (comp == null) is allowed, but
  // should be regarded as an oddity that should not occur.

  protected Component m_comp = null;
  protected MosaicConstraints m_constraints = null;
  protected MosaicCoordStack m_coordStack = null;

  // The size of the box of the leaf node is stored in m_size of
  // the super class.  The component itself can be laid out in a
  // slightly different manner, however, according to its layout
  // constraints.  The screen coordinates of the component itself
  // are stored in the following variables (they are computed by
  // checking the m_anchor, m_fill and m_insets fields of the
  // constraints object).

  protected int m_compOrgx = 0;
  protected int m_compOrgy = 0;
  protected int m_compWidth = 0;
  protected int m_compHeight = 0;

  // Indicates if m_minSize and m_prefSize have been determined.

  protected boolean m_initialized = false;

  //
  //   M o s a i c L e a f N o d e
  //

  protected MosaicLeafNode( MosaicLayout layout, Component comp,
    MosaicConstraints constraints, MosaicCoordStack coordStack )
  {
    super( layout );
    m_comp = comp;
    m_constraints = constraints;
    m_coordStack = coordStack;
  }

  //
  //   i n i t i a l i z e G e o m e t r y
  //

  protected void initializeGeometry()
  {
    if( m_initialized )
      return;

    if( m_comp == null ) {
      m_prefSize = m_minSize = new Dimension( 0, 0 );
    }
    else {
      m_minSize = new Dimension( m_comp.minimumSize() );
      m_prefSize = new Dimension( m_comp.preferredSize() );
    }

    m_weightx = m_constraints.m_weightx;
    m_weighty = m_constraints.m_weighty;

    if( m_constraints.m_insets != null ) {
      int w = m_constraints.m_insets.left + m_constraints.m_insets.right;
      int h = m_constraints.m_insets.top + m_constraints.m_insets.bottom;

      m_minSize.width += w;
      m_minSize.height += h;
      m_prefSize.width += w;
      m_prefSize.height += h;
    }

    m_size = new Dimension( m_prefSize );
    m_initialized = true;
  }

  //
  //   f i n a l i z e G e o m e t r y
  //

  protected void finalizeGeometry( int orgx, int orgy,
    int width, int height, boolean liveDrag )
  {	  
    m_orgx = orgx;
    m_orgy = orgy;
    m_size.width = width;
    m_size.height = height;

    if( m_comp != null ) {
      Insets insets = m_constraints.m_insets;

      if( insets != null ) {
        m_compOrgx = orgx + insets.left;
        m_compOrgy = orgy + insets.top;
        m_compWidth = width - ( insets.right + insets.left );
        m_compHeight = height - ( insets.bottom + insets.top );
      }
      else {
        m_compOrgx = orgx;
        m_compOrgy = orgy;
        m_compWidth = width;
        m_compHeight = height;
      }

      adjustForGravity();
	  if( liveDrag ) {
	    m_comp.reshape( m_compOrgx, m_compOrgy, m_compWidth, m_compHeight );
	    m_comp.validate();
	  }
    }
  }

  //
  //   e n s u r e C a p a c i t y
  //

  // Converts the leaf node into an interior node that is
  // guaranteed to be big enough to contain a slot at grid
  // position (col,row).  Parent pointers are taken care of,
  // and the new node is returned.

  protected MosaicInteriorNode ensureCapacity( int col, int row )
  {
    MosaicInteriorNode N = new MosaicInteriorNode( m_layout );

    N.setParent( m_parent, m_parentCol, m_parentRow );
    N.ensureCapacity( col, row );
    setParent( N, 0, 0 );

    return N;
  }

  //
  //   a d j u s t F o r G r a v i t y
  //

  protected void adjustForGravity()
  {
    int dx, dy;

    if( m_constraints.m_fill == MosaicConstraints.BOTH )
      return;

    if( m_constraints.m_fill == MosaicConstraints.HORIZONTAL ) {
      dx = 0;
    }
    else {
      dx = m_compWidth - m_minSize.width;
      m_compWidth = m_minSize.width;
    }

    if( m_constraints.m_fill == MosaicConstraints.VERTICAL ) {
      dy = 0;
    }
    else {
      dy = m_compHeight - m_minSize.height;
      m_compHeight = m_minSize.height;
    }

    switch( m_constraints.m_anchor ) {
      case MosaicConstraints.CENTER:
           m_compOrgx += dx / 2; m_compOrgy += dy / 2; break;
      case MosaicConstraints.EAST:
           m_compOrgx += dx;     m_compOrgy += dy / 2; break;
      case MosaicConstraints.NORTH:
           m_compOrgx += dx / 2;                       break;
      case MosaicConstraints.NORTHEAST:
           m_compOrgx += dx;                           break;
      case MosaicConstraints.NORTHWEST:
                                                       break;
      case MosaicConstraints.SOUTH:
           m_compOrgx += dx / 2; m_compOrgy += dy;     break;
      case MosaicConstraints.SOUTHEAST:
           m_compOrgx += dx;     m_compOrgy += dy;     break;
      case MosaicConstraints.SOUTHWEST:
           m_compOrgy += dy;                           break;
      case MosaicConstraints.WEST:
           m_compOrgy += dy / 2;                       break;
    }
  }

  //
  //   h a n d l e E v e n t
  //

  protected void handleEvent( Event evt )
  {
    m_layout.defaultCursor();
  }

  //
  //   p a i n t
  //

  protected void paint( Graphics g )
  {
    g.clearRect( m_orgx, m_orgy, m_size.width, m_size.height );
  }

  //
  //   p r i n t
  //

  protected void print( String indent )
  {
    System.out.println( indent + "leaf node " + m_coordStack.toString() );
    super.print( indent + "o ");
  }
}

///////////////////////////////////////////////////////////////
//
//   M o s a i c I n t e r i o r N o d e
//
///////////////////////////////////////////////////////////////

final class MosaicInteriorNode extends MosaicNode
{
  // Every interior node forms a rectangular grid with m_cols
  // columns and m_rows rows.  As children get added to the node,
  // the grid grows dynamically; it represents at all times the 
  // smallest rectangular grid that contains all the children 
  // as well as the gridpoint (0,0).

  protected int m_cols = 0;
  protected int m_rows = 0;

  // For each column and row (called "slices"), size and weight
  // information is stored in MosaicSlice objects.

  protected MosaicSlice m_col[] = null;
  protected MosaicSlice m_row[] = null;

  // The grid itself is stored in the following array.

  protected MosaicNode m_node[][] = null;

  // The size of the sash handles is looked up in the layout
  // manager and stored here. A size of zero means sashes are
  // not available.

  protected int m_sashSize = 0;

  // If the node is active (some of its sashes have been grabbed),
  // the position of the relevant sash(es) is/are stored here.

  protected int m_activeColSash = 0;
  protected int m_activeRowSash = 0;

  // The following values are correction factors; their function
  // is to avoid the jumping of the sashes.  They store the
  // relative position of the mouse cursor in the sashes when
  // the mouse button is first pressed (event MOUSE_DOWN).

  protected int m_colSashOfs = 0;
  protected int m_rowSashOfs = 0;

  //
  //   M o s a i c I n t e r i o r N o d e
  //

  protected MosaicInteriorNode( MosaicLayout layout )
  {
    super( layout );
  }

  //
  //   i n i t i a l i z e G e o m e t r y
  //

  protected void initializeGeometry()
  {
    int i, j;

    m_minSize = new Dimension( 0, 0 );
    m_prefSize = new Dimension( 0, 0 );

    m_weightx = 0;
    m_weighty = 0;

    if( m_cols == 0 || m_rows == 0 ) {
      m_size = new Dimension( 0, 0 );
      return;
    }

    m_col = new MosaicSlice[m_cols];
    m_row = new MosaicSlice[m_rows];

    for( j = 0; j < m_rows; ++j )
      m_row[j] = new MosaicSlice();

    for( i = 0; i < m_cols; ++i ) {
      m_col[i] = new MosaicSlice();
      for( j = 0; j < m_rows; ++j ) {
        MosaicNode N = m_node[i][j];

        if( N != null ) {
          N.initializeGeometry();

          m_col[i].expand( N.m_minSize.width, N.m_prefSize.width );
          m_row[j].expand( N.m_minSize.height, N.m_prefSize.height );
        }
      }
    }

    adjustWeight();
    createSashes();

    m_size = new Dimension( m_prefSize );
  }

  //
  //   a d j u s t W e i g h t
  //

  // At this point, the minimum and preferred sizes of each slice
  // is known.  The weight of each slice is determined and normalized
  // such that the weight sums up to one in both dimensions.

  protected void adjustWeight()
  {
    int i, j;

    for( i = 0; i < m_cols; ++i ) {
      for( j = 0; j < m_rows; ++j ) {
        MosaicNode N = m_node[i][j];

        if( N != null ) {
          m_col[i].m_weight += N.m_weightx * N.m_prefSize.width;
          m_row[j].m_weight += N.m_weighty * N.m_prefSize.height;
        }
      }
    }

    for( i = 0; i < m_cols; ++i ) {
      m_col[i].m_size = m_col[i].m_prefSize;
      m_minSize.width += m_col[i].m_minSize;
      m_prefSize.width += m_col[i].m_prefSize;
      if( m_col[i].m_sizeSum > 1 )
        m_col[i].m_weight /= m_col[i].m_sizeSum;
      m_weightx += m_col[i].m_weight;
    }

    if( m_weightx == 0 ) {
      for( i = 0; i < m_cols; ++i )
        m_col[i].m_weight = 1.0 / m_cols;
    }
    else {
      for( i = 0; i < m_cols; ++i )
        m_col[i].m_weight /= m_weightx;
    }

    for( j = 0; j < m_rows; ++j ) {
      m_row[j].m_size = m_row[j].m_prefSize;
      m_minSize.height += m_row[j].m_minSize;
      m_prefSize.height += m_row[j].m_prefSize;
      if( m_row[j].m_sizeSum > 1 )
        m_row[j].m_weight /= m_row[j].m_sizeSum;
      m_weighty += m_row[j].m_weight;
    }

    if( m_weighty == 0 ) {
      for( j = 0; j < m_rows; ++j )
        m_row[j].m_weight = 1.0 / m_rows;
    }
    else {
      for( j = 0; j < m_rows; ++j )
        m_row[j].m_weight /= m_weighty;
    }
  }

  //
  //   c r e a t e S a s h e s
  //

  protected void createSashes()
  {
    int i, j, t;

    m_sashSize = m_layout.getSashSize( m_parent == null );

    if( m_sashSize == 0 )
      return;

    t = ( m_cols - 1 ) * m_sashSize;
    m_minSize.width += t;
    m_prefSize.width += t;

    t = ( m_rows - 1 ) * m_sashSize;
    m_minSize.height += t;
    m_prefSize.height += t;
  }

  //
  //   f i n a l i z e G e o m e t r y
  //

  protected void finalizeGeometry( int orgx, int orgy,
    int width, int height, boolean liveDrag )
  {
    int i, j;

    m_orgx = orgx;
    m_orgy = orgy;

    Dimension newSize = new Dimension( width, height );

    if( m_cols > 0 ) {
      if( width <= m_minSize.width ) {
        for( i = 0; i < m_cols; ++i )
          m_col[i].m_size = m_col[i].m_minSize;
        newSize.width = m_minSize.width;
      }
      else {
        int dx = width - m_size.width;

        if( dx > 0 )
          distributeExtraSpace( dx, m_cols, m_col );
        else
          if( dx < 0 )
            collectExtraSpace( -dx, m_cols, m_col );
      }

      m_col[0].m_org = m_orgx;
      for( i = 1; i < m_cols; ++i ) {
        m_col[i].m_org =
          m_col[i - 1].m_org + m_col[i - 1].m_size + m_sashSize;
      }
    }

    if( m_rows > 0 ) {
      if( height <= m_minSize.height ) {
        for( j = 0; j < m_rows; ++j )
          m_row[j].m_size = m_row[j].m_minSize;
        newSize.height = m_minSize.height;
      }
      else {
        int dy = height - m_size.height;

        if( dy > 0 )
          distributeExtraSpace( dy, m_rows, m_row );
        else
          if( dy < 0 )
            collectExtraSpace( -dy, m_rows, m_row );
      }

      m_row[0].m_org = m_orgy;
      for( j = 1; j < m_rows; ++j ) {
        m_row[j].m_org =
          m_row[j - 1].m_org + m_row[j - 1].m_size + m_sashSize;
      }
    }

    m_size = newSize;
    for( i = 0; i < m_cols; ++i ) {
      for( j = 0; j < m_rows; ++j ) {
        if( m_node[i][j] != null ) {
          m_node[i][j].finalizeGeometry( m_col[i].m_org, m_row[j].m_org,
            m_col[i].m_size, m_row[j].m_size, liveDrag );
        }
      }
    }
  }

  //
  //   d i s t r i b u t e E x t r a S p a c e
  //

  protected void distributeExtraSpace( int extraSpace, int numOfSlices,
    MosaicSlice[] slice )
  {
    int i, d, restSpace;
    double w;

    restSpace = extraSpace;

    // First bring all components whose weight is positive
    // to their preferred size.

    while( restSpace > 0 ) {
      extraSpace = restSpace;

      w = 0;
      for( i = 0; i < numOfSlices; ++i ) {
        if( slice[i].m_size < slice[i].m_prefSize )
          w += slice[i].m_weight;
      }

      if( w == 0 )
        break;

      for( i = 0; i < numOfSlices; ++i ) {
        if( slice[i].m_size < slice[i].m_prefSize &&
            slice[i].m_weight > 0 ) {
          d = (int) Math.ceil( slice[i].m_weight / w * extraSpace );
          if( d > restSpace )
            d = restSpace;
          if( slice[i].m_size + d > slice[i].m_prefSize )
            d = slice[i].m_prefSize - slice[i].m_size;
          slice[i].m_size += d;
          restSpace -= d;
        }
      }
    }

    // Then distribute the rest proportionally among all
    // components whose weight is positive.

    if( restSpace > 0 ) {
      extraSpace = restSpace;
      for( i = 0; i < numOfSlices; ++i ) {
        d = (int) Math.ceil( slice[i].m_weight * extraSpace );
        if( d > restSpace )
          d = restSpace;
        slice[i].m_size += d;
        restSpace -= d;
      }
    }
  }

  //
  //   c o l l e c t E x t r a S p a c e
  //

  protected void collectExtraSpace( int extraSpace, int numOfSlices,
    MosaicSlice[] slice )
  {
    int i, n, d, d1, restSpace;
    double w;

    restSpace = extraSpace;

    // First steal space from all slices that are larger
    // than their preferred size, and whose weight is greater
    // than zero.

    while( restSpace > 0 ) {
      extraSpace = restSpace;

      w = 0;
      for( i = 0; i < numOfSlices; ++i ) {
        if( slice[i].m_size > slice[i].m_prefSize )
          w += slice[i].m_weight;
      }

      if( w == 0 )
        break;

      for( i = 0; i < numOfSlices; ++i ) {
        if( slice[i].m_size > slice[i].m_prefSize &&
            slice[i].m_weight > 0 ) {
          d = (int) Math.ceil( slice[i].m_weight / w * extraSpace );
          if( d > restSpace )
            d = restSpace;
          if( slice[i].m_size - d < slice[i].m_prefSize )
            d = slice[i].m_size - slice[i].m_prefSize;
          slice[i].m_size -= d;
          restSpace -= d;
        }
      }
    }

    // Now collect from all slices that are larger than their
    // minimum size, and whose weight is greater than zero.

    while( restSpace > 0 ) {
      extraSpace = restSpace;

      w = 0;
      for( i = 0; i < numOfSlices; ++i ) {
        if( slice[i].m_size > slice[i].m_minSize )
          w += slice[i].m_weight;
      }

      if( w == 0 )
        break;

      for( i = 0; i < numOfSlices; ++i ) {
        if( slice[i].m_size > slice[i].m_minSize &&
            slice[i].m_weight > 0 ) {
          d = (int) Math.ceil( slice[i].m_weight / w * extraSpace );
          if( d > restSpace )
            d = restSpace;
          if( slice[i].m_size - d < slice[i].m_minSize )
            d = slice[i].m_size - slice[i].m_minSize;
          slice[i].m_size -= d;
          restSpace -= d;
        }
      } 
    }

    // Now we need another round for the case where the slices
    // that have nonzero weight are already at their minimum
    // size, and all other slices have zero weight.

    while( restSpace > 0 ) {
      n = 0;
      for( i = 0; i < numOfSlices; ++i ) {
        if( slice[i].m_size > slice[i].m_minSize )
          ++n;
      }

      d1 = ( restSpace + n - 1 ) / n;
      for( i = 0; i < numOfSlices; ++i ) {
        if( slice[i].m_size > slice[i].m_minSize ) {
          d = Math.min( restSpace, d1 );
          if( slice[i].m_size - d < slice[i].m_minSize )
            d = slice[i].m_size - slice[i].m_minSize;
          slice[i].m_size -= d;
          restSpace -= d;
        }
      } 
    }
  }

  //
  //   e n s u r e C a p a c i t y
  //

  // Makes the interior node big enough to contain grid
  // position (col,row).

  protected MosaicInteriorNode ensureCapacity( int col, int row )
  {
    int h1Col, h2Col, h1Row, h2Row;

    h1Col = 3 * ( ( m_cols + 2 ) / 3 );
    if( m_cols <= col ) {
      int newCols = col + 1;
      m_cols = newCols;
      h2Col = 3 * ( ( newCols + 2 ) / 3 );
    }
    else {
      h2Col = h1Col;
    }

    h1Row = 3 * ( ( m_rows + 2 ) / 3 );

    if( m_rows <= row ) {
      int newRows = row + 1;
      m_rows = newRows;
      h2Row = 3 * ( ( newRows + 2 ) / 3 );
    }
    else {
      h2Row = h1Row;
    }

    if( h1Col < h2Col || h1Row < h2Row ) {
      MosaicNode[][] newNode = new MosaicNode[h2Col][h2Row];

      for( int i = 0; i < h1Col; ++i ) {
        for( int j = 0; j < h1Row; ++j )
          newNode[i][j] = m_node[i][j];
      }
      m_node = newNode;
    }
    return this;
  }

  //
  //   h a n d l e E v e n t
  //

  protected void handleEvent( Event evt )
  {
    int i, j, inCol, afterCol, inRow, afterRow, cursor;

    // Find out whether the mouse is position within
    // a column (inCol >= 0) or between columns, on a
    // sash (afterCol >= 0).

    inCol = afterCol = -1;
    for( i = 1; i < m_cols; ++i ) {
      if( evt.x < m_col[i].m_org - m_sashSize ) {
        inCol = i - 1;
        break;
      }
      if( evt.x < m_col[i].m_org ) {
        afterCol = i - 1;
        break;
      }
    }

    if( inCol < 0 && afterCol < 0 )
      inCol = m_cols - 1;

    inRow = afterRow = -1;
    for( j = 1; j < m_rows; ++j ) {
      if( evt.y < m_row[j].m_org - m_sashSize ) {
        inRow = j - 1;
        break;
      }
      if( evt.y < m_row[j].m_org ) {
        afterRow = j - 1;
        break;
      }
    }

    if( inRow < 0 && afterRow < 0 )
      inRow = m_rows - 1;

    // Recurse if no sash was hit.

    if( inCol >= 0 && inRow >= 0 ) {
      if( m_node[inCol][inRow] != null )
        m_node[inCol][inRow].handleEvent( evt );
	  else
        m_layout.defaultCursor();
      return;
    }

    // Find out what shape the cursor should take, but
    // only if we can indeed change it (frames only).

    if( m_layout.m_frame != null ) {
      if( afterCol >= 0 ) {
        if( afterRow >= 0 )
          cursor = m_layout.crossSashCursor;
        else
          cursor = m_layout.colSashCursor;
      }
      else {
        cursor = m_layout.rowSashCursor;
      }

      if( m_layout.m_frame.getCursorType() != cursor )
        m_layout.m_frame.setCursor( cursor );
    }

    // If the event was MOUSE_DOWN, initiate the dragging
    // operation by making this node the active one.  Note
    // that the events MOUSE_DRAG and MOUSE_UP are directly
    // relayed to the active node, bypassing the recursion
    // via handleEvent().

    if( evt.id == Event.MOUSE_DOWN ) {
      if( afterCol >= 0 ) {
        m_activeColSash = afterCol + 1;
        m_colSashOfs = m_col[m_activeColSash].m_org - evt.x;
      }

      if( afterRow >= 0 ) {
    	m_activeRowSash = afterRow + 1;
        m_rowSashOfs = m_row[m_activeRowSash].m_org - evt.y;
      }

      m_layout.m_activeNode = this;
    }
  }

  //
  //   d r a g
  //

  protected void drag( int dragX, int dragY )
  {
    boolean finalize = false;

    if( m_activeColSash > 0 ) {
      int dx = dragX + m_colSashOfs - m_col[m_activeColSash].m_org;

      if( m_col[m_activeColSash].m_size - dx <
          m_col[m_activeColSash].m_minSize ) {
        dx = m_col[m_activeColSash].m_size -
             m_col[m_activeColSash].m_minSize;
      }
      else
        if( m_col[m_activeColSash - 1].m_size + dx <
            m_col[m_activeColSash - 1].m_minSize ) {
          dx = m_col[m_activeColSash - 1].m_minSize -
               m_col[m_activeColSash - 1].m_size;
        }

      if( dx != 0 ) {
        m_col[m_activeColSash].m_org += dx;
        m_col[m_activeColSash].m_size -= dx;
        m_col[m_activeColSash - 1].m_size += dx;
        finalize = true;
      }
    }

    if( m_activeRowSash > 0 ) {
      int dy = dragY + m_rowSashOfs - m_row[m_activeRowSash].m_org;

      if( m_row[m_activeRowSash].m_size - dy <
          m_row[m_activeRowSash].m_minSize ) {
        dy = m_row[m_activeRowSash].m_size -
             m_row[m_activeRowSash].m_minSize;
      }
      else
        if( m_row[m_activeRowSash - 1].m_size + dy <
            m_row[m_activeRowSash - 1].m_minSize ) {
          dy = m_row[m_activeRowSash - 1].m_minSize -
               m_row[m_activeRowSash - 1].m_size;
        }

      if( dy != 0 ) {
        m_row[m_activeRowSash].m_org += dy;
        m_row[m_activeRowSash].m_size -= dy;
        m_row[m_activeRowSash - 1].m_size += dy;
        finalize = true;
      }
    }

    if( finalize )
      fixDrag( m_layout.m_liveDrag );
  }

  //
  //   r e l e a s e S a s h
  //

  protected void releaseSash()
  {
    if( ! m_layout.m_liveDrag )
      fixDrag( true );

    m_activeColSash = 0;
    m_activeRowSash = 0;
    m_layout.m_activeNode = null;
  }

  //
  //   f i x D r a g
  //

  // The new position of the dragged sash(es) has/have been
  // determined.  This change has to be propagate down the
  // hierarchy, a la finalize().  We have the panel interrupt
  // the invalidation chain, since we don't want the whole
  // container to be laid out again.

  protected void fixDrag( boolean liveDrag )
  {
    m_layout.m_panel.dontPropagateInvalidate();

    for( int i = 0; i < m_cols; ++i ) {
      for( int j = 0; j < m_rows; ++j ) {
        if( m_node[i][j] != null ) {
          m_node[i][j].finalizeGeometry( m_col[i].m_org,
	      m_row[j].m_org, m_col[i].m_size, m_row[j].m_size, liveDrag );
        }
      }
    }

    m_layout.m_panel.propagateInvalidate();
    m_layout.m_panel.repaint( m_orgx, m_orgy,
      m_size.width, m_size.height );
  }


  //
  //   p a i n t
  //

  // Paint the sashes of the grid of this interior node.

  protected void paint( Graphics g )
  {
    int i, j;

    if( m_sashSize > 0 ) {
      for( i = 1; i < m_cols; ++i )
        paintColSash( m_col[i].m_org - m_sashSize, g );

      for( j = 1; j < m_rows; ++j ) {
        int y = m_row[j].m_org - m_sashSize;
        paintRowSash( y, g );
        if( m_sashSize > 4 ) {
          for( i = 1; i < m_cols; ++i )
            paintCrossSash( m_col[i].m_org - m_sashSize, y, g );
        }
      }
    }

    for( i = 0; i < m_cols; ++i ) {
      for( j = 0; j < m_rows; ++j ) {
        if( m_node[i][j] != null ) {
          m_node[i][j].paint( g );
	}
	else {
          g.clearRect( m_col[i].m_org, m_row[j].m_org,
			m_col[i].m_size, m_row[j].m_size );
	}
      }
    }
  }

  //
  //   p a i n t C o l S a s h
  //

  protected void paintColSash( int x, Graphics g )
  {
    if( m_sashSize <= 4 ) {
      g.setColor( Color.gray );
      g.fillRect( x, m_orgy, m_sashSize, m_size.height );
      return;
    }

    int x2 = x + m_sashSize - 1;
    int y2 = m_orgy + m_size.height - 1;

    g.setColor( Color.lightGray );
    g.fillRect( x, m_orgy + 1, m_sashSize - 2, m_size.height - 1 );
    g.setColor( Color.white );
    g.drawLine( x + 1, m_orgy + 1, x + 1, y2 - 1 );
    g.drawLine( x, m_orgy, x2 - 2, m_orgy );
    g.setColor( Color.gray );
    g.drawLine( x2 - 1, m_orgy, x2 - 1, y2 );
    g.drawLine( x, y2, x2 - 2, y2 );
    g.setColor( Color.black );
    g.drawLine( x2, m_orgy, x2, y2 );
  }

  //
  //   p a i n t R o w S a s h
  //

  protected void paintRowSash( int y, Graphics g )
  {
    if( m_sashSize <= 4 ) {
      g.setColor( Color.gray );
      g.fillRect( m_orgx, y, m_size.width, m_sashSize );
      return;
    }

    int x2 = m_orgx + m_size.width - 1;
    int y2 = y + m_sashSize - 1;

    g.setColor( Color.lightGray );
    g.fillRect( m_orgx + 1, y, m_size.width - 1, m_sashSize - 2 );
    g.setColor( Color.white );
    g.drawLine( m_orgx + 1, y + 1, x2 - 1, y + 1 );
    g.drawLine( m_orgx, y, m_orgx, y2 - 2 );
    g.setColor( Color.gray );
    g.drawLine( m_orgx, y2 - 1, x2, y2 - 1 );
    g.drawLine( x2, y, x2, y2 - 2 );
    g.setColor( Color.black );
    g.drawLine( m_orgx, y2, x2, y2 );
  }

  //
  //   p a i n t C r o s s S a s h
  //

  protected void paintCrossSash( int x, int y, Graphics g )
  {
    int x2 = x + m_sashSize - 1;
    int y2 = y + m_sashSize - 1;

    g.setColor( Color.white );
    g.drawLine( x + 1, y, x + 1, y );
    g.drawLine( x + 1, y2 - 1, x + 1, y2 );
    g.setColor( Color.lightGray );
    g.drawLine( x + 2, y + 1, x2 - 2, y + 1 );
    g.fillRect( x + 2, y2 - 1, m_sashSize - 4, 2 );
    g.drawLine( x, y2, x, y2 );
    g.setColor( Color.gray );
    g.drawLine( x2 - 1, y, x2 - 1, y );
    g.drawLine( x2 - 1, y2, x2 - 1, y2 );
  }

  //
  //   p r i n t
  //

  protected void print( String indent )
  {
    System.out.println( indent + "interior node cols=" +
      m_cols + " rows=" + m_rows );
    super.print( indent + "o ");
    for( int i = 0; i < m_cols; ++i ) {
      for( int j = 0; j < m_rows; ++j ) {
        if( m_node[i][j] != null ) {
          System.out.println( indent +
            "o child (" + i + "," + j + ")" );
          m_node[i][j].print( indent + "  " );
        }
      }
    }
  }
}

///////////////////////////////////////////////////////////////
//
//   M o s a i c S l i c e
//
///////////////////////////////////////////////////////////////

final class MosaicSlice
{
  // A slice describes the shape of a column or row of an
  // interior node.  m_org is the coordinate at which the
  // column/row starts; m_minSize and m_prefSize are the
  // minimum and preferred size, derived from the minimum
  // and preferred sizes of all child nodes that are part
  // of the column/row.  m_size is the actual size.
  // Note that all these values are one-dimensional.

  protected int m_org = 0;
  protected int m_minSize = 0;
  protected int m_prefSize = 0;
  protected int m_size = 0;

  // m_sizeSum is used temporarily to compute the weight of
  // the slice.  The preferred size of all children is
  // summed up in this variable; it is used to normalize
  // the weight such that the sum over all weight parameters
  // of the child nodes in the slice is equal to one.

  protected int m_sizeSum = 0;
  protected double m_weight = 0;

  //
  //   e x p a n d
  //

  protected void expand( int minSize, int prefSize )
  {
    if( minSize > m_minSize )
      m_minSize = minSize;
    if( prefSize > m_prefSize )
      m_prefSize = prefSize;
    m_sizeSum += prefSize;
  }
}
