// Copyright 1999, 2002 Robert Buff

// Contact: http://robertbuff.com/uvm

//

// This file is part of Mtg-Book.

//

// Mtg-Book is free software; you can redistribute it and/or modify

// it under the terms of the GNU General Public License as published

// by the Free Software Foundation; either version 2 of the License,

// or (at your option) any later version.

//

// Mtg-Book is distributed in the hope that it will be useful,

// but WITHOUT ANY WARRANTY; without even the implied warranty of

// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the

// GNU General Public License for more details.

//

// You should have received a copy of the GNU General Public License

// along with Mtg-Book; if not, write to the 

//

// Free Software Foundation, Inc.

// 59 Temple Place, Suite 330

// Boston, MA 02111-1307

// USA



#include "MtgIncl.h"

#include "MtgArrayBounds.h"



MTG_BEGIN_NAMESPACE





//

//   i n i t

//



void tArrayBounds::init()



{

    reset();

}





//

//   p a r t i a l P o s

//



int tArrayBounds::partialPos( int nSamples, ... ) const



{

    va_list Marker;



    va_start( Marker, nSamples );



    int nPos = 0;

    int k = 0;



    for( ; k < nSamples; ++k )

        nPos = nPos * m_Dim[k].m_nSize + va_arg( Marker, int );



    for( ; k < m_Dim.numOfElems(); ++k )

        nPos *= m_Dim[k].m_nSize;



    return nPos;

}





//

//   r e d u c e D i m

//



void tArrayBounds::reduceDim( int nShift )



{

    for( int k = nShift; k < m_Dim.numOfElems(); ++k )

        m_Dim[k - nShift] = m_Dim[k];

    m_Dim.numOfElems( m_Dim.numOfElems() - nShift );

}





//

//   t A r r a y B o u n d s

//



tArrayBounds::tArrayBounds()



{

    init();

}





//

//   t A r r a y B o u n d s

//



tArrayBounds::tArrayBounds( const tArrayBounds& Bounds )



{

    init();

    copyFrom( Bounds );

}





//

//   t A r r a y B o u n d s

//



tArrayBounds::tArrayBounds( const tArrayBounds& Bounds, int x1 )



{

    init();

    copyFrom( Bounds );

    reduce( x1 );

}





//

//   t A r r a y B o u n d s

//



tArrayBounds::tArrayBounds( const tArrayBounds& Bounds, int x1, int x2 )



{

    init();

    copyFrom( Bounds );

    reduce( x1, x2 );

}





//

//   t A r r a y B o u n d s

//



tArrayBounds::tArrayBounds( const tArrayBounds& Bounds, int x1, int x2, int x3 )



{

    init();

    copyFrom( Bounds );

    reduce( x1, x2, x3 );

}





//

//   o p e r a t o r =

//



tArrayBounds& tArrayBounds::operator=( const tArrayBounds& Bounds )



{

    if( &Bounds != this )

        copyFrom( Bounds );

    return *this;

}





//

//   r e s e t

//



void tArrayBounds::reset()



{

    m_nOfs = 0;

    m_Dim.reset();

}





//

//   c o p y F r o m

//



void tArrayBounds::copyFrom( const tArrayBounds& Bounds )



{

    if( &Bounds == this )

        return;



    m_nOfs = Bounds.m_nOfs;

    m_Dim = Bounds.m_Dim;

}





//

//   c l o n e

//



tArrayBounds* tArrayBounds::clone() const



{

    return new tArrayBounds( *this );

}





//

//   a p p e n d

//



void tArrayBounds::append( int nFrom, int nTo )



{

    MTG_ASSERT( nFrom <= nTo );



    ++m_Dim;

    m_Dim.last().m_nFrom = nFrom;

    m_Dim.last().m_nTo = nTo;

    m_Dim.last().m_nSize = nTo - nFrom + 1;

    m_nOfs = m_nOfs * m_Dim.last().m_nSize - m_Dim.last().m_nFrom;

}





//

//   a p p e n d

//



void tArrayBounds::append( const tArrayBounds& Bounds )



{

    for( int k = 0; k < Bounds.dimension(); ++k )

        append( Bounds.from( k ), Bounds.to( k ) );

}





//

//   r e d u c e

//



void tArrayBounds::reduce( int x1 )



{

    MTG_ASSERT( m_Dim.numOfElems() > 1 );

    MTG_ASSERT( m_Dim[0].m_nFrom <= x1 && x1 <= m_Dim[0].m_nTo );



    m_nOfs += partialPos( 1, x1 );

    reduceDim( 1 );

}





//

//   r e d u c e

//



void tArrayBounds::reduce( int x1, int x2 )



{

    MTG_ASSERT( m_Dim.numOfElems() > 2 );

    MTG_ASSERT( m_Dim[0].m_nFrom <= x1 && x1 <= m_Dim[0].m_nTo );

    MTG_ASSERT( m_Dim[1].m_nFrom <= x2 && x2 <= m_Dim[1].m_nTo );



    m_nOfs += partialPos( 2, x1, x2 );

    reduceDim( 2 );

}





//

//   r e d u c e

//



void tArrayBounds::reduce( int x1, int x2, int x3 )



{

    MTG_ASSERT( m_Dim.numOfElems() > 3 );

    MTG_ASSERT( m_Dim[0].m_nFrom <= x1 && x1 <= m_Dim[0].m_nTo );

    MTG_ASSERT( m_Dim[1].m_nFrom <= x2 && x2 <= m_Dim[1].m_nTo );

    MTG_ASSERT( m_Dim[2].m_nFrom <= x3 && x3 <= m_Dim[2].m_nTo );



    m_nOfs += partialPos( 3, x1, x2, x3 );

    reduceDim( 3 );

}



#if defined(_DEBUG)



//

//   p o s

//



int tArrayBounds::pos( int x1 ) const



{

    MTG_ASSERT( m_Dim.numOfElems() == 1 );

    MTG_ASSERT( m_Dim[0].m_nFrom <= x1 && x1 <= m_Dim[0].m_nTo );



    return x1 + m_nOfs;

}





//

//   p o s

//



int tArrayBounds::pos( int x1, int x2 ) const



{

    MTG_ASSERT( m_Dim.numOfElems() == 2 );

    MTG_ASSERT( m_Dim[0].m_nFrom <= x1 && x1 <= m_Dim[0].m_nTo );

    MTG_ASSERT( m_Dim[1].m_nFrom <= x2 && x2 <= m_Dim[1].m_nTo );



    return x1 * m_Dim[1].m_nSize + x2 + m_nOfs;

}





//

//   p o s

//



int tArrayBounds::pos( int x1, int x2, int x3 ) const



{

    MTG_ASSERT( m_Dim.numOfElems() == 3 );

    MTG_ASSERT( m_Dim[0].m_nFrom <= x1 && x1 <= m_Dim[0].m_nTo );

    MTG_ASSERT( m_Dim[1].m_nFrom <= x2 && x2 <= m_Dim[1].m_nTo );

    MTG_ASSERT( m_Dim[2].m_nFrom <= x3 && x3 <= m_Dim[2].m_nTo );



    return ( x1 * m_Dim[1].m_nSize + x2 ) * m_Dim[2].m_nSize + x3 + m_nOfs;

}





//

//   r o w

//



int tArrayBounds::row( int x1 ) const



{

    MTG_ASSERT( m_Dim.numOfElems() == 2 );

    return pos( x1, m_Dim[1].m_nFrom );

}





//

//   r o w

//



int tArrayBounds::row( int x1, int x2 ) const



{

    MTG_ASSERT( m_Dim.numOfElems() == 3 );

    return pos( x1, x2, m_Dim[2].m_nFrom );

}





//

//   r o w

//



int tArrayBounds::row( int x1, int x2, int x3 ) const



{

    MTG_ASSERT( m_Dim.numOfElems() == 4 );

    return pos( x1, x2, x3, m_Dim[3].m_nFrom );

}



#endif



//

//   p o s

//



int tArrayBounds::pos( int x1, int x2, int x3, int x4, va_list Marker ) const



{

    MTG_ASSERT( m_Dim.numOfElems() >= 4 );

    MTG_ASSERT( m_Dim[0].m_nFrom <= x1 && x1 <= m_Dim[0].m_nTo );

    MTG_ASSERT( m_Dim[1].m_nFrom <= x2 && x2 <= m_Dim[1].m_nTo );

    MTG_ASSERT( m_Dim[2].m_nFrom <= x3 && x3 <= m_Dim[2].m_nTo );

    MTG_ASSERT( m_Dim[3].m_nFrom <= x4 && x4 <= m_Dim[3].m_nTo );



    int nPos = ( ( x1 * m_Dim[1].m_nSize + x2 ) * m_Dim[2].m_nSize + 

                   x3 ) * m_Dim[3].m_nSize + x4;



    for( int k = 4; k < m_Dim.numOfElems(); ++k ) {

        int x = va_arg( Marker, int );

        MTG_ASSERT( m_Dim[k].m_nFrom <= x && x <= m_Dim[k].m_nTo );

        nPos = nPos * m_Dim[k].m_nSize + x;

    }



    return nPos + m_nOfs;

}





//

//   p o s

//



int tArrayBounds::pos( int x1, int x2, int x3, int x4, ... ) const



{

    va_list Marker;

    va_start( Marker, x4 );

    return pos( x1, x2, x3, x4, Marker );

}





//

//   p o s

//



int tArrayBounds::pos( const tHeap<int>& x ) const



{

    MTG_ASSERT( m_Dim.numOfElems() == x.numOfElems() );



    int nPos = 0;



    for( int k = 0; k < m_Dim.numOfElems(); ++k ) {

        MTG_ASSERT( m_Dim[k].m_nFrom <= x[k] && x[k] <= m_Dim[k].m_nTo );

        nPos = nPos * m_Dim[k].m_nSize + x[k];

    }



    return nPos + m_nOfs;

}





//

//   p o s

//



int tArrayBounds::pos( const int x[] ) const



{

    int nPos = 0;



    for( int k = 0; k < m_Dim.numOfElems(); ++k ) {

        MTG_ASSERT( m_Dim[k].m_nFrom <= x[k] && x[k] <= m_Dim[k].m_nTo );

        nPos = nPos * m_Dim[k].m_nSize + x[k];

    }



    return nPos + m_nOfs;

}





//

//   p o s

//



int tArrayBounds::pos( int x1, const tHeap<int>& x ) const



{

    MTG_ASSERT( m_Dim.numOfElems() == x.numOfElems() + 1 );

    MTG_ASSERT( m_Dim[0].m_nFrom <= x1 && x1 <= m_Dim[0].m_nTo );



    int nPos = x1;



    for( int k = 1; k < m_Dim.numOfElems(); ++k ) {

        MTG_ASSERT( m_Dim[k].m_nFrom <= x[k - 1] && x[k - 1] <= m_Dim[k].m_nTo );

        nPos = nPos * m_Dim[k].m_nSize + x[k - 1];

    }



    return nPos + m_nOfs;

}





//

//   r o w

//



int tArrayBounds::row( int x1, int x2, int x3, int x4, va_list Marker ) const



{

    MTG_ASSERT( m_Dim.numOfElems() >= 5 );

    MTG_ASSERT( m_Dim[0].m_nFrom <= x1 && x1 <= m_Dim[0].m_nTo );

    MTG_ASSERT( m_Dim[1].m_nFrom <= x2 && x2 <= m_Dim[1].m_nTo );

    MTG_ASSERT( m_Dim[2].m_nFrom <= x3 && x3 <= m_Dim[2].m_nTo );

    MTG_ASSERT( m_Dim[3].m_nFrom <= x4 && x4 <= m_Dim[3].m_nTo );



    int nPos = ( ( x1 * m_Dim[1].m_nSize + x2 ) * m_Dim[2].m_nSize + 

                   x3 ) * m_Dim[3].m_nSize + x4;



    for( int k = 4; k < m_Dim.numOfElems() - 1; ++k )

        nPos = nPos * m_Dim[k].m_nSize + va_arg( Marker, int );



    nPos *= m_Dim.last().m_nSize;



    return nPos + m_Dim.last().m_nFrom + m_nOfs;

}





//

//   r o w

//



int tArrayBounds::row( int x1, int x2, int x3, int x4, ... ) const



{

    va_list Marker;

    va_start( Marker, x4 );

    return row( x1, x2, x3, x4, Marker );

}





//

//   r o w

//



int tArrayBounds::row( const tHeap<int>& x ) const



{

    MTG_ASSERT( m_Dim.numOfElems() == x.numOfElems() ||

                m_Dim.numOfElems() == x.numOfElems() + 1 );



    int nPos = 0;



    for( int k = 0; k < m_Dim.numOfElems() - 1; ++k ) {

        MTG_ASSERT( m_Dim[k].m_nFrom <= x[k] && x[k] <= m_Dim[k].m_nTo );

        nPos = nPos * m_Dim[k].m_nSize + x[k];

    }



    nPos *= m_Dim.last().m_nSize;

    return nPos + m_Dim.last().m_nFrom + m_nOfs;

}





//

//   r o w

//



int tArrayBounds::row( const int x[] ) const



{

    int nPos = 0;



    for( int k = 0; k < m_Dim.numOfElems() - 1; ++k ) {

        MTG_ASSERT( m_Dim[k].m_nFrom <= x[k] && x[k] <= m_Dim[k].m_nTo );

        nPos = nPos * m_Dim[k].m_nSize + x[k];

    }



    nPos *= m_Dim.last().m_nSize;

    return nPos + m_Dim.last().m_nFrom + m_nOfs;

}



#if defined(_MTG_WITH_MATHLINK)



//

//   t o M a t h L i n k

//



tRetCode tArrayBounds::toMathLink( MLINK MathLink ) const



{

    if( MLPutFunction( MathLink, "List", dimension() ) == 0 )

        return MATHLINK_ERROR;



    for( int i = 0; i < dimension(); ++i ) {

        if( MLPutFunction( MathLink, "List", 2 ) == 0 ||

            MLPutInteger( MathLink, from( i ) ) == 0 ||

            MLPutInteger( MathLink, to( i ) ) == 0 ) {

            return MATHLINK_ERROR;

        }

    }

    return OK;

}



#endif



MTG_END_NAMESPACE



#if defined(_TEST)



#if defined(_WIN32)

    #include <conio.h>

#endif 





//

//   m a i n

//



void main( int argc, char *argv[] )



{

    bool bGo;

    int k, cCmd, nFrom, nTo, nPos, nNum, aSample[16];

    tRetCode nRet;

    tArrayBounds B1, B2;



    printf( "\nTest tArrayBounds\n\n" );



    bGo = true;

    while( bGo ) {

        printf( "<R>eset <A>ppend <P>os Re<d>uce E<x>it: " );

        cCmd = getche();

        printf( "\n" );



        nRet = OK;

        switch( cCmd ) {

            case 'r' :

            case 'R' :

                B1.reset();

                break;



            case 'a' :

            case 'A' :

                printf( "  From To: " );

                scanf( "%d%d", &nFrom, &nTo );

                B1.append( nFrom, nTo );

                break;



            case 'p' :

            case 'P' :

                printf( "  Pos%d: ", B1.dimension() );

                for( k = 0; k < B1.dimension(); ++k )

                    scanf( "%d", &aSample[k] );

                switch( B1.dimension() ) {

                    case 1 :

                        nPos = B1.pos( aSample[0] ); 

                        break;



                    case 2 :

                        nPos = B1.pos( aSample[0], aSample[1] ); 

                        break;



                    case 3 :

                        nPos = B1.pos( aSample[0], aSample[1], aSample[2] ); 

                        break;



                    case 4 :

                        nPos = B1.pos( aSample[0], aSample[1], aSample[2],

                                       aSample[3] ); 

                        break;



                    case 5 :

                        nPos = B1.pos( aSample[0], aSample[1], aSample[2],

                                       aSample[3], aSample[4] ); 

                        break;

                }

                printf( "Pos: %d\n", nPos );

                break;



            case 'd' :

            case 'D' :

                printf( "  Num Pos...: " );

                scanf( "%d", &nNum );

                for( k = 0; k < nNum; ++k )

                    scanf( "%d", &aSample[k] );

                B2 = B1;

                switch( k ) {

                    case 1 :

                        B2.reduce( aSample[0] ); 

                        break;



                    case 2 :

                        B2.reduce( aSample[0], aSample[1] ); 

                        break;



                    case 3 :

                        B2.reduce( aSample[0], aSample[1], aSample[2] ); 

                        break;

                }

                B1 = B2;

                break;



            case 'x' :

            case 'X' :

                bGo = false;

                break;

        }



        if( nRet != OK )

            printf( "  Error=%d\n", nRet );

    }

}



#endif

