// Copyright 1999, 2002 Robert Buff

// Contact: http://robertbuff.com/uvm

//

// This file is part of Mtg-Book.

//

// Mtg-Book is free software; you can redistribute it and/or modify

// it under the terms of the GNU General Public License as published

// by the Free Software Foundation; either version 2 of the License,

// or (at your option) any later version.

//

// Mtg-Book is distributed in the hope that it will be useful,

// but WITHOUT ANY WARRANTY; without even the implied warranty of

// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the

// GNU General Public License for more details.

//

// You should have received a copy of the GNU General Public License

// along with Mtg-Book; if not, write to the 

//

// Free Software Foundation, Inc.

// 59 Temple Place, Suite 330

// Boston, MA 02111-1307

// USA



#include "MtgIncl.h"

#include "MtgBondEquation.h"



#include <float.h>



MTG_BEGIN_NAMESPACE





//

//   v a r i a b l e s

//



const int tBondEquation::m_nMaxIter = 50;





//

//   i n i t

//



void tBondEquation::init( tType nType, tDayCount DayCount )



{

    m_gRedemption = 100;

    m_nType = nType;

    m_DayCount = DayCount;

    m_pCompounder = 0;

}





//

//   c l e a n u p

//



void tBondEquation::cleanup()



{

    if( m_pCompounder != 0 ) {

        delete m_pCompounder;

        m_pCompounder = 0;

    }

}





//

//   c o p y F r o m

//



void tBondEquation::copyFrom( const tBondEquation& Equation )



{

    if( &Equation == this )

        return;



    cleanup();



    m_gRedemption = Equation.m_gRedemption;

    m_nType = Equation.m_nType;

    m_DayCount = Equation.m_DayCount;

    

    if( Equation.m_pCompounder != 0 )

        m_pCompounder = Equation.m_pCompounder->clone();

}





//

//   f i n d F r a c t i o n s

//



void tBondEquation::findFractions( tDate Settlement, tDate Maturity,

    const tCompounder* pCompounder, double gFractionFactor,     

    int& nNumOfRegularCoupons, double& gFractionToNextCoupon,

    bool& bHasAccruedInterest, double& gFractionSincePrevCoupon,

    bool& bHasOddLastCoupon, double& gFractionOfLastPeriod ) const



{

    nNumOfRegularCoupons =

        pCompounder->numOfCompoundingDates( Settlement, Maturity );



    MTG_ASSERT( nNumOfRegularCoupons >= 0 );



    tDate PeriodStart, PeriodEnd;



    if( ! pCompounder->isCompoundingDate( Maturity ) ) {

            // we have an odd-last-coupon period

        bHasOddLastCoupon = true;

        pCompounder->getCompoundingPeriod( Maturity, PeriodStart, PeriodEnd );



            // day count fraction of entire last coupon period:

        gFractionOfLastPeriod = gFractionFactor * 

            m_DayCount.fraction( PeriodStart, Maturity,

                PeriodStart, PeriodEnd );



        if( nNumOfRegularCoupons == 0 ) {

                // day count fraction from settlement to

                // last coupon date:

            gFractionToNextCoupon = gFractionFactor * 

                m_DayCount.fraction( Settlement, Maturity,

                    PeriodStart, PeriodEnd ) /

                gFractionOfLastPeriod;

        }

        else {

                // there's a regular coupon between settlement

                // and maturity:

            pCompounder->getCompoundingPeriod( Settlement,

                PeriodStart, PeriodEnd );

            gFractionToNextCoupon = gFractionFactor * 

                m_DayCount.fraction( Settlement, PeriodEnd,

                    PeriodStart, PeriodEnd );

        }

    }

    else {

        bHasOddLastCoupon = false;

        pCompounder->getCompoundingPeriod( Settlement,

            PeriodStart, PeriodEnd );



        if( nNumOfRegularCoupons > 0 ) {

            gFractionToNextCoupon = gFractionFactor * 

                m_DayCount.fraction( Settlement, PeriodEnd,

                    PeriodStart, PeriodEnd );

        }

        else {

            MTG_ASSERT( Settlement == Maturity );

            gFractionToNextCoupon = 0;

        }

    }



    bHasAccruedInterest = true;

    gFractionSincePrevCoupon = gFractionFactor * 

        m_DayCount.fraction( PeriodStart, Settlement,

            PeriodStart, PeriodEnd );



    if( bHasOddLastCoupon && nNumOfRegularCoupons == 0 ) {

            // this has to be normalized if the only coupon

            // left is the odd-last-coupon

        gFractionSincePrevCoupon /= gFractionOfLastPeriod;

    }

}





//

//   p r i c e

//



double tBondEquation::price( double gCoupon, double gLastCoupon,

    double gYield, int nNumOfRegularCoupons, double gFractionToNextCoupon,

    bool bHasOddLastCoupon, double gFractionOfLastPeriod,

    double* pGradient ) const



{

        // For a summary of these equations, see Stigum and

        // Robinson, Money Market and Bond Calculations, pp. 281.

        

    if( m_nType == xBraessFangmeyer )

        throw tException( NOT_IMPLEMENTED );



        // Discount factor:



    double V = 1 / ( 1 + gYield );

    int n1 = nNumOfRegularCoupons - 1;



    double gPrice = 0;



    if( pGradient )

        *pGradient = 0;



        // Add regular coupon contribution:



    if( nNumOfRegularCoupons > 0 ) {

        if( fabs( gYield ) > DBL_EPSILON ) {

            double h = pow( V, n1 );

            double t = ( 1 - h ) / gYield;



            gPrice += gCoupon * ( 1 + t );

            if( pGradient && n1 > 0 )

                *pGradient += gCoupon * ( n1 * h * V - t ) / gYield;

        }

        else {

                // Use de l'Hospital for gradient.

            gPrice += gCoupon * ( 1 + n1 );

            if( *pGradient && n1 > 0 ) 

                *pGradient += gCoupon * n1 * ( n1 + 1 ) / ( -2 ); 

        }

    }



        // Add redemption component, plus possibly

        // odd last coupon:



    if( bHasOddLastCoupon ) {

            // This is the irregular payment:

        double x = m_gRedemption + gLastCoupon;



        if( nNumOfRegularCoupons > 0 ) {

                // Discount x back to next coupon date:

            double h = pow( V, n1 + gFractionOfLastPeriod );



            gPrice += x * h;

            if( pGradient )

                *pGradient -= x * ( n1 + gFractionOfLastPeriod ) * h * V;

        }

        else {

                // The odd last coupon is the next coupon,

                // there's nothing to discount. No gradient either.

            gPrice += x;

        }

    }

    else {

        if( nNumOfRegularCoupons > 0 ) {

            double h = pow( V, n1 );



            gPrice += m_gRedemption * h;

            if( pGradient )

                *pGradient -= m_gRedemption * n1 * h * V;

        }

    }



        // Now discount everything back to the settlement date.

        // The standard and Moosmueller approaches differ in

        // that Moosmueller uses money market discounting

        // from the next coupon payment back to the settlement

        // date.

        

    switch( m_nType ) {

        case xStandard : {

                double h = pow( V, gFractionToNextCoupon );



                gPrice *= h;

                if( pGradient ) {

                    *pGradient =

                        h * *pGradient - gFractionToNextCoupon * gPrice * V;

                }

            }

            break;



        case xMoosmueller : {

                double h = 1 / ( 1 + gFractionToNextCoupon * gYield );



                gPrice *= h;

                if( pGradient ) {

                    *pGradient = 

                        h * *pGradient - gFractionToNextCoupon * gPrice * h; 

                }

            }

            break;



        default :

            throw tException( INTERNAL_ERROR );

    }



    return gPrice;

}





//

//   p r i c e

//



double tBondEquation::price( tDate Settlement, tDate Maturity, double gCoupon,

    double gYield, const tCompounder* pCompounder, bool bClean ) const



{

    MTG_ASSERT( pCompounder != 0 && pCompounder->numOfPeriodsPerYear() > 0 );

    MTG_ASSERT( gCoupon >= 0 && gYield > 0 );



    if( Settlement >= Maturity )

        return 0;   // bond has expired



    double gYieldFactor, gFractionFactor;



        // Normalization of yield and coupon:



    if( ! pCompounder->getNormFactor( m_DayCount,

            gYieldFactor, gFractionFactor ) ) {

            // illegal combination

        throw tException( INTERNAL_ERROR );

    }



    gYield *= gYieldFactor;

    gCoupon /= pCompounder->numOfPeriodsPerYear();



    int nNumOfRegularCoupons;

    double gFractionToNextCoupon;

    bool bHasAccruedInterest;

    double gFractionSincePrevCoupon;

    bool bHasOddLastCoupon;

    double gFractionOfLastPeriod;



    findFractions( Settlement, Maturity, pCompounder, gFractionFactor,

        nNumOfRegularCoupons, gFractionToNextCoupon,

        bHasAccruedInterest, gFractionSincePrevCoupon,

        bHasOddLastCoupon, gFractionOfLastPeriod );



    double gLastCoupon = 

        bHasOddLastCoupon ? gCoupon * gFractionOfLastPeriod : gCoupon;



    double gPrice = 

        price( gCoupon, gLastCoupon, gYield,

            nNumOfRegularCoupons, gFractionToNextCoupon,

            bHasOddLastCoupon, gFractionOfLastPeriod );



        // Adjust for accrued interest:



    if( bClean && bHasAccruedInterest ) {

        if( nNumOfRegularCoupons == 0 )

            gPrice -= gLastCoupon * gFractionSincePrevCoupon;

        else

            gPrice -= gCoupon * gFractionSincePrevCoupon;

    }



    return gPrice;

}





//

//   y i e l d

//



double tBondEquation::yield( tDate Settlement, tDate Maturity, double gCoupon,

    double gPrice, const tCompounder* pCompounder, bool bClean ) const



{

    MTG_ASSERT( pCompounder != 0 && pCompounder->numOfPeriodsPerYear() > 0 );

    MTG_ASSERT( gCoupon >= 0 );

    MTG_ASSERT( Settlement < Maturity );



    double gYieldFactor, gFractionFactor;



        // Normalization of yield and coupon:



    if( ! pCompounder->getNormFactor( m_DayCount,

            gYieldFactor, gFractionFactor ) ) {

            // illegal combination

        throw tException( INTERNAL_ERROR );

    }



    gCoupon /= pCompounder->numOfPeriodsPerYear();



    int nNumOfRegularCoupons;

    double gFractionToNextCoupon;

    bool bHasAccruedInterest;

    double gFractionSincePrevCoupon;

    bool bHasOddLastCoupon;

    double gFractionOfLastPeriod;



    findFractions( Settlement, Maturity, pCompounder, gFractionFactor,

        nNumOfRegularCoupons, gFractionToNextCoupon,

        bHasAccruedInterest, gFractionSincePrevCoupon,

        bHasOddLastCoupon, gFractionOfLastPeriod );



    double gLastCoupon = 

        bHasOddLastCoupon ? gCoupon * gFractionOfLastPeriod : gCoupon;



        // Adjust for accrued interest - we need the dirty price:



    if( bClean && bHasAccruedInterest ) {

        if( nNumOfRegularCoupons == 0 )

            gPrice += gLastCoupon * gFractionSincePrevCoupon;

        else

            gPrice += gCoupon * gFractionSincePrevCoupon;

    }



        // Now refine initial guess with Newton-Raphson.



    double gYield = gCoupon / ( 1 + gCoupon );



    int nIter = 0;

    double gLastError = 0;



    while( true ) {

        double gGradient;

        double gNewPrice =

            price( gCoupon, gLastCoupon, gYield,

                nNumOfRegularCoupons, gFractionToNextCoupon,

                bHasOddLastCoupon, gFractionOfLastPeriod, &gGradient );



        double gNewYield = gYield - ( gNewPrice - gPrice ) / gGradient;

        double gError = fabs( gNewPrice / gPrice - 1 );

        

        ++nIter;



        if( gLastError > 0 &&

            gError >= gLastError &&

            nIter > m_nMaxIter / 2 ) {

                // No improvement; be content with last guess.

                // (This escape only after a considerable number

                // of iterations.)

            break;

        }



        if( gError < 1e-10 ) {

                // Be content with 10 digits accuracy.

            gYield = gNewYield;

            break;

        }

        

        if( nIter >= m_nMaxIter )

            return false;



        gLastError = gError;

        gYield = gNewYield;

    }



    return gYield / gYieldFactor;

}





//

//   t B o n d E q u a t i o n

//



tBondEquation::tBondEquation()



{

    init( xStandard, DayCountACT_ACT );

}





//

//   t B o n d E q u a t i o n

//



tBondEquation::tBondEquation( tType nType )



{

    init( nType, DayCountACT_ACT );

}





//

//   t B o n d E q u a t i o n

//



tBondEquation::tBondEquation( tType nType, tDayCount DayCount )



{

    init( nType, DayCount );

}





//

//   t B o n d E q u a t i o n

//



tBondEquation::tBondEquation( const tBondEquation& Equation )



{

    init( xStandard, DayCountACT_ACT );

    copyFrom( Equation );

}





//

//   ~ t B o n d E q u a t i o n

//



tBondEquation::~tBondEquation()



{

    cleanup();

}





//

//   o p e r a t o r =

//



tBondEquation& tBondEquation::operator=( const tBondEquation& Equation )



{

    if( &Equation != this )

        copyFrom( Equation );

    return *this;

}





//

//   s e t

//



void tBondEquation::set( tType nType, tDayCount DayCount )



{

    m_nType = nType;

    m_DayCount = DayCount;

}





//

//   s e t

//



void tBondEquation::set( tType nType )



{

    m_nType = nType;

}





//

//   s e t

//



void tBondEquation::set( tDayCount DayCount )



{

    m_DayCount = DayCount;

}





//

//   s e t

//



void tBondEquation::set( const tCompounder& Compounder )



{

    cleanup();

    m_pCompounder = Compounder.clone();

}





//

//   a c c r u e d I n t e r e s t

//



double tBondEquation::accruedInterest( tDate Settlement, tDate Maturity,

    double gCoupon ) const



{

    return accruedInterest( Settlement, Maturity, gCoupon, m_pCompounder );

}





//

//   a c c r u e d I n t e r e s t

//



double tBondEquation::accruedInterest( tDate Settlement, tDate Maturity,

    double gCoupon, const tCompounder* pCompounder ) const



{

    MTG_ASSERT( pCompounder != 0 && pCompounder->numOfPeriodsPerYear() > 0 );

    MTG_ASSERT( gCoupon >= 0 );



    if( Settlement >= Maturity )

        return 0;   // bond has expired



    double gYieldFactor, gFractionFactor;



        // Normalization of yield and coupon:



    if( ! pCompounder->getNormFactor( m_DayCount,

            gYieldFactor, gFractionFactor ) ) {

            // illegal combination

        throw tException( INTERNAL_ERROR );

    }



    gCoupon /= pCompounder->numOfPeriodsPerYear();



    int nNumOfRegularCoupons;

    double gFractionToNextCoupon;

    bool bHasAccruedInterest;

    double gFractionSincePrevCoupon;

    bool bHasOddLastCoupon;

    double gFractionOfLastPeriod;



    findFractions( Settlement, Maturity, pCompounder, gFractionFactor,

        nNumOfRegularCoupons, gFractionToNextCoupon,

        bHasAccruedInterest, gFractionSincePrevCoupon,

        bHasOddLastCoupon, gFractionOfLastPeriod );



    if( ! bHasAccruedInterest )

        return 0;



    if( nNumOfRegularCoupons == 0 && bHasOddLastCoupon )

        return gCoupon * gFractionOfLastPeriod * gFractionSincePrevCoupon;

    return gCoupon * gFractionSincePrevCoupon;

}





//

//   a c c r u e d I n t e r e s t

//



double tBondEquation::accruedInterest( tDate Settlement, double gCoupon,

    const tCouponCompounder* pCompounder ) const



{

    MTG_ASSERT( pCompounder != 0 );

    return accruedInterest( Settlement, pCompounder->maturity(), gCoupon,

        pCompounder );

}





//

//   c l e a n P r i c e

//



double tBondEquation::cleanPrice( tDate Settlement, tDate Maturity,

    double gCoupon, double gYield ) const



{

    return cleanPrice( Settlement, Maturity, gCoupon, gYield, m_pCompounder );

}





//

//   c l e a n P r i c e

//



double tBondEquation::cleanPrice( tDate Settlement, tDate Maturity,

    double gCoupon, double gYield, const tCompounder* pCompounder ) const



{

    return price( Settlement, Maturity, gCoupon, gYield,

        pCompounder, true );

}





//

//   c l e a n P r i c e

//



double tBondEquation::cleanPrice( tDate Settlement, double gCoupon,

    double gYield, const tCouponCompounder* pCompounder ) const



{

    MTG_ASSERT( pCompounder != 0 );

    return cleanPrice( Settlement, pCompounder->maturity(), gCoupon,

        gYield, pCompounder );

}





//

//   d i r t y P r i c e

//



double tBondEquation::dirtyPrice( tDate Settlement, tDate Maturity,

    double gCoupon, double gYield ) const



{

    return dirtyPrice( Settlement, Maturity, gCoupon, gYield, m_pCompounder );

}





//

//   d i r t y P r i c e

//



double tBondEquation::dirtyPrice( tDate Settlement, tDate Maturity,

    double gCoupon, double gYield, const tCompounder* pCompounder ) const



{

    return price( Settlement, Maturity, gCoupon, gYield,

        pCompounder, false );

}





//

//   d i r t y P r i c e

//



double tBondEquation::dirtyPrice( tDate Settlement, double gCoupon, 

    double gYield, const tCouponCompounder* pCompounder ) const



{

    MTG_ASSERT( pCompounder != 0 );

    return dirtyPrice( Settlement, pCompounder->maturity(), gCoupon,

        gYield, pCompounder );

}





//

//   c l e a n 2 Y i e l d

//



double tBondEquation::clean2Yield( tDate Settlement, tDate Maturity,

    double gCoupon, double gPrice ) const



{

    return clean2Yield( Settlement, Maturity, gCoupon, gPrice, m_pCompounder );

}





//

//   c l e a n 2 Y i e l d

//



double tBondEquation::clean2Yield( tDate Settlement, tDate Maturity,

    double gCoupon, double gPrice, const tCompounder* pCompounder ) const



{

    return yield( Settlement, Maturity, gCoupon, gPrice,

        pCompounder, true );

}





//

//   c l e a n 2 Y i e l d

//



double tBondEquation::clean2Yield( tDate Settlement, double gCoupon,

    double gPrice, const tCouponCompounder* pCompounder ) const



{

    MTG_ASSERT( pCompounder != 0 );

    return clean2Yield( Settlement, pCompounder->maturity(), gCoupon,

        gPrice, pCompounder );

}





//

//   d i r t y 2 Y i e l d

//



double tBondEquation::dirty2Yield( tDate Settlement, tDate Maturity,

    double gCoupon, double gPrice ) const



{

    return dirty2Yield( Settlement, Maturity, gCoupon, gPrice, m_pCompounder );

}





//

//   d i r t y 2 Y i e l d

//



double tBondEquation::dirty2Yield( tDate Settlement, tDate Maturity,

    double gCoupon, double gPrice, const tCompounder* pCompounder ) const



{

    return yield( Settlement, Maturity, gCoupon, gPrice,

        pCompounder, false );

}





//

//   d i r t y 2 Y i e l d

//



double tBondEquation::dirty2Yield( tDate Settlement, double gCoupon,

    double gPrice, const tCouponCompounder* pCompounder ) const



{

    MTG_ASSERT( pCompounder != 0 );

    return dirty2Yield( Settlement, pCompounder->maturity(), gCoupon,

        gPrice, pCompounder );

}



MTG_END_NAMESPACE





//#define _TEST

#if defined(_TEST)



#if defined(_WIN32)

    #include <conio.h>

#else

    #define getche getchar

#endif



MTG_USING_NAMESPACE



//

//   m a i n

//



void main( int argc, char *argv[] )



{

    printf( "\nTest tBondEquation\n\n" );



    char sBuf[256];

    int cCmd;



    tDayCount DayCount = DayCountACT_ACT;

    tCouponCompounder Compounder;

    tDate Settlement = System.base();



    double gCoupon = 5;

    double gYield = 0.05;

    double gPrice = 100;



    tBondEquation Eq;

    tBondEquation::tType nType = tBondEquation::xStandard;



    tDate D1, D2;



    bool bGo = true;



    while( bGo ) { 

        printf( "<D>aycount <B>ond <T>ype <S>ettle <P>rice <Y>ield E<x>it: " );

        cCmd = getche();

        printf( "\n" );



        switch( cCmd ) {

            case 'd' :

            case 'D' :

                printf( "1=ACT/ACT     2=ACT/365   3=ACT/365NL\n"

                        "4=ACT/365ISDA 5=ACT/360   6=30/360ISDA\n"

                        "7=30/360PSA   8=30/360SIA 9=30E/360\n" 

                        "Choose: " );

                cCmd = getche();

                printf( "\n" );

                switch( cCmd ) {

                    case '1' : DayCount = DayCountACT_ACT;      break;

                    case '2' : DayCount = DayCountACT_365;      break;

                    case '3' : DayCount = DayCountACT_365_NL;   break;

                    case '4' : DayCount = DayCountACT_365_ISDA; break;

                    case '5' : DayCount = DayCountACT_360;      break;

                    case '6' : DayCount = DayCount30_360_ISDA;  break;

                    case '7' : DayCount = DayCount30_360_PSA;   break;

                    case '8' : DayCount = DayCount30_360_SIA;   break;

                    case '9' : DayCount = DayCount30E_360;      break;

                }

                break;



            case 'b' :

            case 'B' :

                printf( "Date of maturity: " );

                gets( sBuf );

                if( D1.set( sBuf ) == 0 ) {

                    printf( "Format error\n" );

                    break;

                }

                printf( "Coupon date or <Return>: " );

                gets( sBuf );

                if( sBuf[0] != 0 ) {

                    if( D2.set( sBuf ) == 0 ) {

                        printf( "Format error\n" );

                        break;

                    }

                }

                else {

                    D2 = D1;

                }

                Compounder.set( D1, D2, 2 );

                break;



            case 't' :

            case 'T' :

                if( nType == tBondEquation::xStandard ) {

                    nType = tBondEquation::xMoosmueller;

                    printf( "Type is now Moosmueller\n" );

                }

                else {

                    nType = tBondEquation::xStandard;

                    printf( "Type is now Standard\n" );

                }

                break;



            case 's' :

            case 'S' :

                printf( "Settlement date: " );

                gets( sBuf );

                if( Settlement.set( sBuf ) == 0 )

                    printf( "Format error\n" );

                break;



            case 'p' :

            case 'P' :

                printf( "Coupon: " );

                gets( sBuf );

                sscanf( sBuf, "%lg", &gCoupon );

                printf( "Yield: " );

                gets( sBuf );

                sscanf( sBuf, "%lg", &gYield );



                Eq.set( nType );

                Eq.set( DayCount );

                                

                printf( "Clean price is %lg\n",

                    Eq.cleanPrice( Settlement, Compounder.maturity(),

                        gCoupon, gYield, &Compounder ) );

                printf( "Dirty price is %lg\n",

                    Eq.dirtyPrice( Settlement, Compounder.maturity(),

                        gCoupon, gYield, &Compounder ) );

                break;



            case 'y' :

            case 'Y' :

                printf( "Coupon: " );

                gets( sBuf );

                sscanf( sBuf, "%lg", &gCoupon );

                printf( "Clean price: " );

                gets( sBuf );

                sscanf( sBuf, "%lg", &gPrice );



                Eq.set( nType );

                Eq.set( DayCount );

                                

                printf( "Yield is %lg\n",

                    Eq.clean2Yield( Settlement, Compounder.maturity(),

                        gCoupon, gPrice, &Compounder ) );

                break;



            case 'x' :

            case 'X' :

                bGo = false;

                break;

        }

    }



#if ! defined(_WIN32)

    printf( "\n" );

#endif

}



#endif