// Copyright 1999, 2002 Robert Buff

// Contact: http://robertbuff.com/uvm

//

// This file is part of Mtg-Book.

//

// Mtg-Book is free software; you can redistribute it and/or modify

// it under the terms of the GNU General Public License as published

// by the Free Software Foundation; either version 2 of the License,

// or (at your option) any later version.

//

// Mtg-Book is distributed in the hope that it will be useful,

// but WITHOUT ANY WARRANTY; without even the implied warranty of

// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the

// GNU General Public License for more details.

//

// You should have received a copy of the GNU General Public License

// along with Mtg-Book; if not, write to the 

//

// Free Software Foundation, Inc.

// 59 Temple Place, Suite 330

// Boston, MA 02111-1307

// USA



#if ! defined(_MTG_BOND_EQUATION_)

#define _MTG_BOND_EQUATION_



#include "MtgCompounder.h"



MTG_BEGIN_NAMESPACE





//

//   t B o n d E q u a t i o n

//



class tBondEquation {



public:



        // The bond equation shouldn't be used directly, but

        // rather via appropriate tBondMath objects which know

        // how to use it. (For instance, a tUSBondMath object

        // for US treasury bonds.)



        // Reference: Stigum and Robinson, Money Market and

        // Bond Calculations, pp. 281.





    enum tType {

        xStandard,          

        xMoosmueller,       // Money-market discounting in the

                            // current coupon period

        xBraessFangmeyer

    };



private:



    double m_gRedemption;   // 100, fixed throughout



    tType m_nType;          // Default: standard

    tDayCount m_DayCount;   // Default: ACT/ACT



        // Can be 0, then it has to be supplied every time

        // a price-yield calculation is made:

        

    tCompounder* m_pCompounder;



        // Max number of iterations in Newton-Raphson method

        // for value to yield calculations:



    static const int m_nMaxIter;



    void init( tType nType, tDayCount DayCount );

    void cleanup();



    void copyFrom( const tBondEquation& Equation );



    void findFractions( tDate Settlement, tDate Maturity,

        const tCompounder* pCompounder, double gFractionFactor,

        int& nNumOfRegularCoupons, double& gFractionToNextCoupon,

        bool& bHasAccruedInterest, double& gFractionSincePrevCoupon,

        bool& bHasOddLastCoupon, double& gFractionOfLastPeriod ) const;



    double price( double gCoupon, double gLastCoupon, double gYield,

        int nNumOfRegularCoupons, double gFractionToNextCoupon,

        bool bHasOddLastCoupon, double gFractionOfLastPeriod,

        double* pGradient = 0 ) const;



    double price( tDate Settlement, tDate Maturity, double gCoupon,

        double gYield, const tCompounder* pCompounder, bool bClean ) const;



    double yield( tDate Settlement, tDate Maturity,double gCoupon,

        double gPrice, const tCompounder* pCompounder, bool bClean ) const;



public:



    tBondEquation();

    tBondEquation( tType nType );

    tBondEquation( tType nType, tDayCount DayCount );

    tBondEquation( const tBondEquation& Equation );



    ~tBondEquation();



    tBondEquation& operator=( const tBondEquation& Equation );



    void set( tType nType, tDayCount DayCount );



    void set( tType nType );

    void set( tDayCount DayCount );

    void set( const tCompounder& Compounder );



        // If only the accrued interest is desired:



    double accruedInterest( tDate Settlement, tDate Maturity,

        double gCoupon ) const;



    double accruedInterest( tDate Settlement, tDate Maturity,

        double gCoupon, const tCompounder* pCompounder ) const;



    double accruedInterest( tDate Settlement, double gCoupon,

        const tCouponCompounder* pCompounder ) const;



        // Yield-to-maturity to clean price:



    double cleanPrice( tDate Settlement, tDate Maturity,

        double gCoupon, double gYield ) const;



    double cleanPrice( tDate Settlement, tDate Maturity,

        double gCoupon, double gYield,

        const tCompounder* pCompounder ) const;



    double cleanPrice( tDate Settlement, double gCoupon, double gYield,

        const tCouponCompounder* pCompounder ) const;



        // Yield-to-maturity to dirty price:



    double dirtyPrice( tDate Settlement, tDate Maturity,

        double gCoupon, double gYield ) const;



    double dirtyPrice( tDate Settlement, tDate Maturity,

        double gCoupon, double gYield,

        const tCompounder* pCompounder ) const;



    double dirtyPrice( tDate Settlement, double gCoupon, double gYield,

        const tCouponCompounder* pCompounder ) const;



        // Clean price to yield-to-maturity:



    double clean2Yield( tDate Settlement, tDate Maturity,

        double gCoupon, double gPrice ) const;



    double clean2Yield( tDate Settlement, tDate Maturity,

        double gCoupon, double gPrice,

        const tCompounder* pCompounder ) const;



    double clean2Yield( tDate Settlement, double gCoupon,

        double gPrice, const tCouponCompounder* pCompounder ) const;



        // Dirty price to yield-to-maturity:



    double dirty2Yield( tDate Settlement, tDate Maturity,

        double gCoupon, double gPrice ) const;



    double dirty2Yield( tDate Settlement, tDate Maturity,

        double gCoupon, double gPrice,

        const tCompounder* pCompounder ) const;



    double dirty2Yield( tDate Settlement, double gCoupon,

        double gPrice, const tCouponCompounder* pCompounder ) const;

};



MTG_END_NAMESPACE



#endif

