// Copyright 1999, 2002 Robert Buff

// Contact: http://robertbuff.com/uvm

//

// This file is part of Mtg-Book.

//

// Mtg-Book is free software; you can redistribute it and/or modify

// it under the terms of the GNU General Public License as published

// by the Free Software Foundation; either version 2 of the License,

// or (at your option) any later version.

//

// Mtg-Book is distributed in the hope that it will be useful,

// but WITHOUT ANY WARRANTY; without even the implied warranty of

// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the

// GNU General Public License for more details.

//

// You should have received a copy of the GNU General Public License

// along with Mtg-Book; if not, write to the 

//

// Free Software Foundation, Inc.

// 59 Temple Place, Suite 330

// Boston, MA 02111-1307

// USA



#include "MtgIncl.h"

#include "MtgBootstrap.h"

#include "MtgClaim.h"

#include "MtgParser.h"

#include "MtgPortfolio.h"



#include <float.h>



MTG_BEGIN_NAMESPACE





//

//   c o p y F r o m

//



void tBootstrap::copyFrom( const tBootstrap& Boot )



{

    if( &Boot == this )

        return;



    m_Spline = Boot.m_Spline;

    m_OutFile = Boot.m_OutFile;

    m_OutLegFile = Boot.m_OutLegFile;

    m_OutGnuDataFile = Boot.m_OutGnuDataFile;

    m_OutGnuTicsFile = Boot.m_OutGnuTicsFile;



    super::copyFrom( Boot );

}





//

//   p a r s e I t e m

//



tRetCode tBootstrap::parseItem( tParser& Parser )



{

    MTG_ASSERT( Parser.curToken() == xTokClaim ||

                Parser.curToken() == xTokPortfolio ||

                Parser.curToken() == xTokId ||

                Parser.beginOfObj() );



    tRetCode nRet;



    char* sName = 0;

    tClaim* pClaim = 0;

    tPortfolio* pPortfolio = 0;



    bool bExpectClaim = false;

    bool bExpectPortfolio = false;

    bool bNew = false;



    switch( Parser.curToken() ) {

        case xTokClaim :

            if( ( nRet = Parser.readToken() ) != OK )

                return nRet;

            bExpectClaim = true;

            break;



        case xTokPortfolio :

            if( ( nRet = Parser.readToken() ) != OK )

                return nRet;

            bExpectPortfolio = true;

            break;



        default:

            break;

    }



    if( Parser.curToken() == xTokId ) {

        sName = StrCopy( Parser.curText() );

        if( ( nRet = Parser.readToken() ) != OK ) {

            delete sName;

            return nRet;

        }

    }



    if( Parser.beginOfObj() ) {

        if( bExpectPortfolio ) {

            if( sName != 0 )

                delete sName;

            return Parser.setError( "portfolio expected" );

        }



        tObject *pObj;



        if( ( nRet = tClaim::parse( Parser, system(), pObj ) ) != OK ) {

            if( sName != 0 )

                delete sName;

            return nRet;

        }

        pClaim = static_cast<tClaim*>( pObj );

        bNew = true;

    } 

    else {

        if( sName == 0 )

            return Parser.setError( "claim/portfolio name expected" );



        tObject* pObj = Parser.findObject( sName );



        if( pObj == 0 ) {

            delete sName;

            return Parser.setError( NOT_FOUND );

        }



        if( ( pClaim = dynamic_cast<tClaim*>( pObj ) ) != 0 ) {

            if( bExpectPortfolio )

                return Parser.setError( "portfolio expected" );

        }

        else

        if( ( pPortfolio = dynamic_cast<tPortfolio*>( pObj ) ) != 0 ) {

            if( bExpectClaim )

                return Parser.setError( "claim expected" );

        }

        else {

            delete sName;

            return Parser.setError( "claim/portfolio expected" );

        }

    }



    if( sName != 0 )

        delete sName;



    if( pClaim != 0 ) {

        if( ! pClaim->addPaymentStream( m_Spline ) ) {

            if( bNew )

                delete pClaim;

            return Parser.setError( "claim cannot be used in bootstrap" );

        }

    }



    if( pPortfolio != 0 )

        pPortfolio->addPaymentStream( m_Spline );



    if( bNew )

        delete pClaim;



    return OK;

}





//

//   p a r s e P a r a m

//



tRetCode tBootstrap::parseParam( tParser& Parser, tParseInfoStub& Info )



{

    tRetCode nRet;

    tDate Start, End;

    double gYield;



    if( Parser.curToken() == xTokImport ||

        Parser.curToken() == xTokClaim ||

        Parser.curToken() == xTokPortfolio ||

        Parser.curToken() == xTokId ||

        Parser.beginOfObj() ) {



        if( Parser.curToken() == xTokImport ) {

            if( ( nRet = Parser.readToken() ) != OK ) 

                return nRet;

            if( Parser.curToken() != xTokClaim &&

                Parser.curToken() != xTokPortfolio &&

                Parser.curToken() != xTokId &&

                ! Parser.beginOfObj() ) {

                return Parser.setError( INVALID_KEYWORD );

            }

        }



        if( ( nRet = parseItem( Parser ) ) != OK )

            return nRet;

    }

    else {

        switch( Parser.curToken() ) {

            case xTokForward :

                if( ( nRet = Parser.readToken() ) != OK ||

                    ( nRet = Parser.scanDate( Start ) ) != OK ) {

                    return nRet;

                }

                if( Parser.curToken() == xTokString ) {

                    if( ( nRet = Parser.scanDate( End ) ) != OK )

                        return nRet;

                }

                else {

                        // Start date not necessary for forwards.

                    End = Start;

                    Start = End - 1;

                }

                if( Start >= End )

                    return Parser.setError( INVALID_DATE );

                if( ( nRet = Parser.scanPosPercentage( gYield ) ) != OK )

                    return nRet;

                m_Spline.addForward( Start, End, gYield );

                break;



            case xTokImplied :

                if( ( nRet = Parser.readToken() ) != OK ||

                    ( nRet = Parser.scanDate( Start ) ) != OK ||

                    ( nRet = Parser.scanDate( End ) ) != OK ||

                    ( nRet = Parser.scanPosPercentage( gYield ) ) != OK ) {

                    return nRet;

                }

                if( Start >= End )

                    return Parser.setError( INVALID_DATE );

                m_Spline.addImplied( Start, End, gYield );

                break;



            case xTokSave :

                if( ( nRet = Parser.readToken() ) != OK )

                    return nRet;



                switch( Parser.curToken() ) {

                    case xTokLeg :

                        if( m_OutLegFile.isValid() )

                            return Parser.setError( ATTR_REDEFINITION );

                        if( ( nRet = Parser.readToken() ) != OK ||

                            ( nRet = m_OutLegFile.parseWrite(

                                        Parser ) ) != OK ) {

                            return nRet;

                        }

                        break;



                    case xTokGnuData :

                        if( m_OutGnuDataFile.isValid() )

                            return Parser.setError( ATTR_REDEFINITION );

                        if( ( nRet = Parser.readToken() ) != OK ||

                            ( nRet = m_OutGnuDataFile.parseWrite(

                                        Parser ) ) != OK ) {

                            return nRet;

                        }

                        break;



                    case xTokGnuTics :

                        if( m_OutGnuTicsFile.isValid() )

                            return Parser.setError( ATTR_REDEFINITION );

                        if( ( nRet = Parser.readToken() ) != OK ||

                            ( nRet = m_OutGnuTicsFile.parseWrite(

                                        Parser ) ) != OK ) {

                            return nRet;

                        }

                        break;



                    default :

                        if( m_OutFile.isValid() )

                            return Parser.setError( ATTR_REDEFINITION );

                        if( Parser.curToken() == xTokBootstrap ) {

                            if( ( nRet = Parser.readToken() ) != OK )

                                return nRet;

                        }

                        if( ( nRet = m_OutFile.parseWrite( Parser ) ) != OK )

                            return nRet;

                        break;

                }

                break;



            default :

                return super::parseParam( Parser, Info );

        }

    }



    return OK;

}





//

//   p a r s e P o s t f i x

//



tRetCode tBootstrap::parsePostfix( tParser& Parser, tParseInfoStub& Info )



{

    if( m_Spline.numOfLegs() == 0 ) 

        return Parser.setError( MISSING_ITEM );



    m_Spline.set( tInterest( tInterest::xExponential,

        tInterest::xYield, system().dayCount() ) );



    m_Spline.merge();



#if defined(_DEBUG)

    //tDate End;

    //double gYield;



    //m_Spline.getFirstLeg( End, gYield );

    //do {

    //    MTG_TRACE( "%02d/%02d/%04d %lg\n", 

    //        End.month(), End.day(), End.year(), gYield );

    //} while( m_Spline.getNextLeg( End, gYield ) );

#endif



    return super::parsePostfix( Parser, Info );

}





//

//   t B o o t s t r a p

//



tBootstrap::tBootstrap()



{

}





//

//   t B o o t s t r a p

//



tBootstrap::tBootstrap( const tBootstrap& Boot )



{

    copyFrom( Boot );

}





//

//   ~ t B o o t s t r a p

//



tBootstrap::~tBootstrap()



{

}





//

//   o p e r a t o r =

//



tBootstrap& tBootstrap::operator=( const tBootstrap& Boot )



{

    if( &Boot != this )

        copyFrom( Boot );

    return *this;

}





//

//   c l o n e

//



tObject* tBootstrap::clone() const



{

    return new tBootstrap( *this );

}





//

//   f i n a l i z e

//



tRetCode tBootstrap::finalize()



{

    tRetCode nRet;



    if( ( nRet = super::finalize() ) != OK )

        return nRet;



    return OK;

}





//

//   s a v e

//



tRetCode tBootstrap::save()



{

    MTG_ASSERT( isFinalized() );

                

    tRetCode nRet;



        // output entire object



    if( m_OutFile.isValid() ) {

        ofstream Out;



        if( ( nRet = m_OutFile.openWrite( Out ) ) != OK )

            return nRet;



        Out << setprecision( DBL_DIG )

            << "bootstrap " << name() << " {" << endl;



        tDate End;

        double gYield;



        m_Spline.getFirstLeg( End, gYield );

        do {

            Out << "    forward \"" << End

                << "\" " << gYield << "," << endl;

        } while( m_Spline.getNextLeg( End, gYield ) );



        Out << "}" << endl;



        if( ! Out.good() )

            return WRITE_ERROR;

    }



        // output plain legs



    if( m_OutLegFile.isValid() ) {

        ofstream Out;



        if( ( nRet = m_OutLegFile.openWrite( Out,

                        "bootstrap legs" ) ) != OK ) {

            return nRet;

        }



        tDate End;

        double gYield;



        m_Spline.getFirstLeg( End, gYield );

        do {

            Out << End << " " << gYield << endl;

        } while( m_Spline.getNextLeg( End, gYield ) );



        if( ! Out.good() )

            return WRITE_ERROR;

    }



        // output legs for Gnuplot (step plot expected)



    if( m_OutGnuDataFile.isValid() ) {

        ofstream Out;



        if( ( nRet = m_OutGnuDataFile.openWrite( Out ) ) != OK )

            return nRet;



            // for Gnuplot, we multiply the samples by 100



        tDate End;

        double gYield;

        int nTic = 0;



        m_Spline.getFirstLeg( End, gYield );

        do {

            Out << nTic++ << " " << 100 * gYield << endl;

        } while( m_Spline.getNextLeg( End, gYield ) );

        Out << nTic << " " << 100 * gYield << endl;



        if( ! Out.good() )

            return WRITE_ERROR;

    }



    if( m_OutGnuTicsFile.isValid() ) {

        ofstream Out;



        static const char* M[] = {

            "", "Jan", "Feb", "Mar", "Apr", "May", "Jun",

                "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"

        };



        if( ( nRet = m_OutGnuTicsFile.openWrite( Out ) ) != OK )

            return nRet;



        tDate End, Now = system().base();

        int nTic = 0;

        double gYield;



        m_Spline.getFirstLeg( End, gYield );



        Out.fill( '0' );



        if( Now < End ) {

            Out << "set xtics (\"" << M[Now.month()] << " "

                << setw( 2 ) << Now.year() % 100 << "\" 0";

        }

        else {

            Out << "set xtics (\"Past\" 0";

        }



        do {

            Out << ", \"" << M[End.month()] << " "

                << setw( 2 ) << End.year() % 100 << "\" " << ++nTic;

        } while( m_Spline.getNextLeg( End, gYield ) );



        Out << ")" << endl; 



        if( ! Out.good() )

            return WRITE_ERROR;

    }



    return OK;

}





//

//   p a r s e

//



tRetCode tBootstrap::parse( tParser& Parser, tSystem& System,

    tObject*& pObj )



{

    tRetCode nRet;



    if( ( nRet = Parser.scanBeginOfObj() ) != OK )

        return nRet;



    tBootstrap* pBoot = new tBootstrap;

    pBoot->setSystem( System );



    if( ( nRet = pBoot->super::parse( Parser ) ) != OK ) {

        delete pBoot;

        return nRet;

    }



    pObj = pBoot;

    return OK;

}



#if defined(_MTG_WITH_TCL)





//

//   t c l T e s t

//



bool tBootstrap::tclTest( const tObject* pObj )



{

    return dynamic_cast<const tBootstrap*>( pObj ) != 0;

}





//

//   t c l C o m m a n d

//



int tBootstrap::tclCommand( ClientData clientData, Tcl_Interp *pInterp,

    int objc, Tcl_Obj *CONST objv[] )



{

    tBootstrap* pBoot = static_cast<tBootstrap*>( clientData );



    if( objc < 4 ) {

        Tcl_WrongNumArgs( pInterp, 1, objv,

            "command [options] startdate enddate" );

        return TCL_ERROR;

    }



    int nLength;

    char* sCmd =

        StrToLower( Tcl_GetStringFromObj( objv[1], &nLength ) );



    int nCmd = -1;



    if( strcmp( sCmd, "presentvalue" ) == 0 )

        nCmd = 0;

    else

    if( strcmp( sCmd, "futurevalue" ) == 0 )

        nCmd = 1;

    else

    if( strcmp( sCmd, "swaprate" ) == 0 )

        nCmd = 2;

    else

    if( strcmp( sCmd, "forward" ) == 0 || strcmp( sCmd, "yield" ) == 0 )

        nCmd = 3;



    delete sCmd;

    if( nCmd < 0 )

        return TCL_ERROR;



    int nNextObj = 2;

    int nPeriods = 2;

    tBondMath* pMath = 0;



    if( nCmd == 2 ) {

        if( ! getBondMath( pInterp, objc, objv, nNextObj, pMath ) )

            return TCL_ERROR;



        if( pMath == 0 ) {

            char* sString = Tcl_GetStringFromObj( objv[nNextObj], &nLength );



            if( nLength > 0 && *sString == '-' ) { 

                sString = StrToLower( sString );



                if( strcmp( sString, "-periods" ) == 0 ) {

                    delete sString;

                    if( ++nNextObj == objc ||

                        Tcl_GetIntFromObj( pInterp,

                            objv[nNextObj], &nPeriods ) != TCL_OK ) {

                        return TCL_ERROR;

                    }

                    ++nNextObj;

                }

                else {

                    delete sString;

                    return TCL_ERROR;

                }

            }

        }

    }



    if( objc - nNextObj != 2 ) {

        Tcl_WrongNumArgs( pInterp, 1, objv,

            "command [options] startdate enddate" );

        return TCL_ERROR;

    }



    tDate Start, End;



    if( ! getDate( pInterp, objc, objv, nNextObj, Start ) ||

        ! getDate( pInterp, objc, objv, nNextObj, End ) ) {

        return TCL_ERROR;

    }



    if( Start > End ) {

        tDate T = Start;

        Start = End;

        End = T;

    }



    double gValue;



    switch( nCmd ) {

        case 0 :    // presentvalue

            try { 

                gValue = pBoot->spline().presentValue( Start, End );

            }

            catch( tException ) {

                return TCL_ERROR;

            }

            break;



        case 1 :    // futurevalue

            try { 

                gValue = pBoot->spline().futureValue( Start, End );

            }

            catch( tException ) {

                return TCL_ERROR;

            }

            break;



        case 2 :    // swaprate

            try { 

                if( pMath == 0 )

                    gValue = pBoot->spline().swapRate( Start, End, nPeriods );

                else

                    gValue = pBoot->spline().swapRate( Start, End, *pMath );

            }

            catch( tException ) {

                return TCL_ERROR;

            }

            break;



        case 3 :    // forward

            try { 

                gValue = pBoot->spline().yield( Start, End );

            }

            catch( tException ) {

                return TCL_ERROR;

            }

            break;



        default :

            throw tException( INTERNAL_ERROR );

    }



    Tcl_SetObjResult( pInterp, Tcl_NewDoubleObj( gValue ) );

    return TCL_OK;

}





//

//   t c l D e l e t e

//



void tBootstrap::tclDelete( ClientData clientData )



{

    tBootstrap* pBoot = static_cast<tBootstrap*>( clientData );



    if( pBoot != 0 )

        delete pBoot;

}



#endif



MTG_END_NAMESPACE