// Copyright 1999, 2002 Robert Buff

// Contact: http://robertbuff.com/uvm

//

// This file is part of Mtg-Book.

//

// Mtg-Book is free software; you can redistribute it and/or modify

// it under the terms of the GNU General Public License as published

// by the Free Software Foundation; either version 2 of the License,

// or (at your option) any later version.

//

// Mtg-Book is distributed in the hope that it will be useful,

// but WITHOUT ANY WARRANTY; without even the implied warranty of

// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the

// GNU General Public License for more details.

//

// You should have received a copy of the GNU General Public License

// along with Mtg-Book; if not, write to the 

//

// Free Software Foundation, Inc.

// 59 Temple Place, Suite 330

// Boston, MA 02111-1307

// USA



#include "MtgIncl.h"

#include "MtgCompounder.h"



MTG_BEGIN_NAMESPACE





//

//   i n i t

//



void tCompounder::init()



{

    m_bLeftOpen = false;

}





//

//   s e t L e f t O p e n

//



void tCompounder::setLeftOpen()



{

    m_bLeftOpen = true;

}





//

//   s e t R i g h t O p e n

//



void tCompounder::setRightOpen()



{

    m_bLeftOpen = false;

}





//

//   c o p y F r o m

//



void tCompounder::copyFrom( const tCompounder& Compounder )



{

    if( &Compounder == this )

        return;

    m_bLeftOpen = Compounder.m_bLeftOpen;

}





//

//   t C o m p o u n d e r

//



tCompounder::tCompounder()



{

    init();

}





//

//   t C o m p o u n d e r

//



tCompounder::tCompounder( const tCompounder& Compounder )



{

    init();

    copyFrom( Compounder );

}





//

//   ~ t C o m p o u n d e r

//



tCompounder::~tCompounder()



{

}





//

//   n u m O f P e r i o d s P e r Y e a r

//



int tCompounder::numOfPeriodsPerYear() const



{

    return -1;

}





//

//   g e t N o r m F a c t o r

//



bool tCompounder::getNormFactor( const tDayCount& DayCount,

    double& gYieldFactor, double& gFractionFactor ) const



{

    int n = numOfPeriodsPerYear();



    if( DayCount == DayCountACT_ACT ) {

        if( n < 1 )     // unknown?

            return false;

        gYieldFactor = 1.0 / n;

        gFractionFactor = 1;

    }

    else {

        if( n < 1 ) {

            gYieldFactor = 1;

            gFractionFactor = 1;

        }

        else {

            gYieldFactor = 1.0 / n;

            gFractionFactor = n;

        }

    }



    return true;

}





//

//   g e t Y e a r F a c t o r

//



bool tCompounder::getYearFactor( const tDayCount& DayCount,

    double& gFractionFactor ) const



{

    double gYieldFactor;



    if( ! getNormFactor( DayCount, gYieldFactor, gFractionFactor ) )

        return false;



    int n = numOfPeriodsPerYear();



    if( n > 0 )

        gFractionFactor /= n;



    return true;

}





//

//   i n i t

//



void tPeriodCompounder::init( int nNumOfPeriods )



{

    m_nNumOfPeriods = nNumOfPeriods;

}





//

//   c o p y F r o m

//



void tPeriodCompounder::copyFrom( const tPeriodCompounder& Compounder )



{

    if( &Compounder == this )

        return;



    m_nNumOfPeriods = Compounder.m_nNumOfPeriods;

    super::copyFrom( Compounder );

}





//

//   t P e r i o d C o m p o u n d e r

//



tPeriodCompounder::tPeriodCompounder()



{

    init( 0 );

}





//

//   t P e r i o d C o m p o u n d e r

//



tPeriodCompounder::tPeriodCompounder( int nNumOfPeriods )



{

    init( nNumOfPeriods );

}





//

//   t P e r i o d C o m p o u n d e r

//



tPeriodCompounder::tPeriodCompounder( const tPeriodCompounder& Compounder )



{

    init( 0 );

    copyFrom( Compounder );

}





//

//   ~ t P e r i o d C o m p o u n d e r

//



tPeriodCompounder::~tPeriodCompounder()



{

}





//

//   o p e r a t o r =

//



tPeriodCompounder& tPeriodCompounder::operator=(

    const tPeriodCompounder& Compounder )



{

    if( &Compounder != this )

        copyFrom( Compounder );

    return *this;

}





//

//   c l o n e

//



tCompounder* tPeriodCompounder::clone() const



{

    return new tPeriodCompounder( *this );

}





//

//   s e t

//



void tPeriodCompounder::set( int nNumOfPeriods )



{

    m_nNumOfPeriods = nNumOfPeriods;

}





//

//   n u m O f P e r i o d s P e r Y e a r

//



int tPeriodCompounder::numOfPeriodsPerYear() const



{

    return m_nNumOfPeriods;

}





//

//   i s C o m p o u n d i n g D a t e

//



bool tPeriodCompounder::isCompoundingDate( tDate Date ) const



{

    throw tException( NOT_IMPLEMENTED );

    return false;

}





//

//   g e t C o m p o u n d i n g P e r i o d

//



void tPeriodCompounder::getCompoundingPeriod( tDate Date,

    tDate& Start, tDate& End ) const



{

    throw tException( NOT_IMPLEMENTED );

}





//

//   g e t N e x t C o m p o u n d i n g P e r i o d

//



void tPeriodCompounder::getNextCompoundingPeriod( tDate& Start, 

    tDate& End ) const



{

    throw tException( NOT_IMPLEMENTED );

}





//

//   n u m O f C o m p o u n d i n g D a t e s

//



int tPeriodCompounder::numOfCompoundingDates( tDate Start, tDate End ) const



{

    throw tException( NOT_IMPLEMENTED );

    return 0;

}





//

//   i n i t

//



void tCyclicCompounder::init()



{

}





//

//   n e x t C o m p o u n d i n g D a t e

//



int tCyclicCompounder::nextCompoundingDate( const tDate& Date,

    tDate& Compounding ) const



{

    MTG_ASSERT( m_DateTemplate.numOfElems() > 0 );



    int k = 0;

    int n = m_DateTemplate.numOfElems();



    while( k < n ) {

        Compounding = prepCompoundingDate( k, Date.year() );

        if( Compounding > Date || leftOpen() && Compounding == Date )

            break;

        ++k;

    }



    if( k == n ) {

            // wrap-around

        k = 0;

        Compounding = prepCompoundingDate( k, Date.year() + 1 );

    }



    return k;

}





//

//   p r e p C o m p o u n d i n g D a t e

//



tDate tCyclicCompounder::prepCompoundingDate( int nIndex, int nYear ) const



{

    const tDateTemplate& C = m_DateTemplate[nIndex];

    int nDay = C.m_nDay;



    if( C.m_nMonth == 2 ) {

        int t = tDate::lastDayInFebruary( nYear );



        if( nDay > t )

            nDay = t;

    }

    else 

    if( nDay == 31 &&

        ( C.m_nMonth == 4 || C.m_nMonth == 6 ||

          C.m_nMonth == 9 || C.m_nMonth == 11 ) ) {

        nDay = 30;

    }



    tDate d( nDay, C.m_nMonth, nYear );

    

    if( C.m_pShift != 0 )

        return C.m_pShift->shift( d );

    return d;

}





//

//   c o p y F r o m

//



void tCyclicCompounder::copyFrom( const tCyclicCompounder& Compounder )



{

    if( &Compounder == this )

        return;



    resetDateTemplates();



    for( int i = 0; i < Compounder.m_DateTemplate.numOfElems(); ++i ) {

        const tDateTemplate& T = Compounder.m_DateTemplate[i];



        if( T.m_pShift == 0 )

            addDateTemplate( T.m_nDay, T.m_nMonth );

        else

            addDateTemplate( T.m_nDay, T.m_nMonth, *T.m_pShift );

    }



    super::copyFrom( Compounder );

}





//

//   r e s e t D a t e T e m p l a t e s

//



void tCyclicCompounder::resetDateTemplates()



{

    for( int i = 0; i < m_DateTemplate.numOfElems(); ++i ) {

        if( m_DateTemplate[i].m_pShift != 0 )

            delete m_DateTemplate[i].m_pShift;

    }



    m_DateTemplate.reset();

}





//

//   t C y c l i c C o m p o u n d e r

//



tCyclicCompounder::tCyclicCompounder()



{

    init();

}





//

//   t C y c l i c C o m p o u n d e r

//



tCyclicCompounder::tCyclicCompounder( const tCyclicCompounder& Compounder )



{

    init();

    copyFrom( Compounder );

}





//

//   ~ t C y c l i c C o m p o u n d e r

//



tCyclicCompounder::~tCyclicCompounder()



{

    resetDateTemplates();

}





//

//   o p e r a t o r =

//



tCyclicCompounder& tCyclicCompounder::operator=(

    const tCyclicCompounder& Compounder )



{

    if( &Compounder != this )

        copyFrom( Compounder );

    return *this;

}





//

//   c l o n e

//



tCompounder* tCyclicCompounder::clone() const



{

    return new tCyclicCompounder( *this );

}





//

//   n u m O f P e r i o d s P e r Y e a r

//



int tCyclicCompounder::numOfPeriodsPerYear() const



{

    return m_DateTemplate.numOfElems();

}





//

//   i s V a l i d D a t e T e m p l a t e

//



bool tCyclicCompounder::isValidDateTemplate( int nDay, int nMonth ) const



{

    if( nDay < 1 || nDay > 31 || nMonth < 1 || nMonth > 12 )

        return false;

        // everything else gets adjusted

    return true;

}





//

//   a d d D a t e T e m p l a t e

//



bool tCyclicCompounder::addDateTemplate( int nDay, int nMonth )



{

    return addDateTemplate( nDay, nMonth, tIdentityDayShift() );

}





//

//   a d d D a t e T e m p l a t e

//



bool tCyclicCompounder::addDateTemplate( int nDay, int nMonth, 

    const tDayShift& Shift )



{

    if( ! isValidDateTemplate( nDay, nMonth ) )

        return false;



    int i = 0;



    for( ; i < m_DateTemplate.numOfElems(); ++i ) {

        tDateTemplate& C = m_DateTemplate[i];



            // Check if compounding day exists already:

        if( C.m_nDay == nDay && C.m_nMonth == nMonth ) {

            if( C.m_pShift != 0 )

                delete C.m_pShift;

            C.m_pShift = Shift.clone();

            return true;

        }



            // Keep sequence sorted.

        if( C.m_nMonth > nMonth ||

            C.m_nMonth == nMonth && C.m_nDay > nDay ) {

            ++m_DateTemplate;

            for( int j = m_DateTemplate.numOfElems() - 1; j > i; --j )

                m_DateTemplate[j] = m_DateTemplate[j - 1];

            break;

        }

    }



    if( i == m_DateTemplate.numOfElems() )

        ++m_DateTemplate;



    tDateTemplate& C = m_DateTemplate[i];



    C.m_nDay = nDay;

    C.m_nMonth = nMonth;

    C.m_pShift = Shift.clone();



    return true;

}





//

//   i s C o m p o u n d i n g D a t e

//



bool tCyclicCompounder::isCompoundingDate( tDate Date ) const



{

    int n = m_DateTemplate.numOfElems();



    if( n == 0 )

        return false;



        // there could be a wrap around:

    if( prepCompoundingDate( n - 1, Date.year() - 1 ) == Date )

        return true;



        // now check current year:



    for( int i = 0; i < n; ++i ) {

        tDate d = prepCompoundingDate( i, Date.year() );



        if( d == Date )

            return true;

        if( d > Date )

            break;

    }



    return false;

}





//

//   g e t C o m p o u n d i n g P e r i o d

//



void tCyclicCompounder::getCompoundingPeriod( tDate Date,

    tDate& Start, tDate& End ) const



{

    int k = nextCompoundingDate( Date, End );



        // find previous coupon



    int j = ( k == 0 ) ? m_DateTemplate.numOfElems() - 1 : k - 1;

    Start = prepCompoundingDate( j, Date.year() );

    

    if( Start > Date || leftOpen() && Start == Date ) {

            // wrap-around necessary

        Start = prepCompoundingDate( j, Date.year() - 1 );

    }

}





//

//   g e t N e x t C o m p o u n d i n g P e r i o d

//



void tCyclicCompounder::getNextCompoundingPeriod( tDate& Start,

    tDate& End ) const



{

    if( leftOpen() )

        getCompoundingPeriod( End + 1, Start, End );

    else

        getCompoundingPeriod( End, Start, End );

}





//

//   n u m O f C o m p o u n d i n g D a t e s

//



int tCyclicCompounder::numOfCompoundingDates( tDate Start, tDate End ) const



{

    int n = m_DateTemplate.numOfElems();



    if( n == 0 )

        return 0;



    int c = 0;



    if( Start > End )

        return 0;



    if( Start.year() == End.year() ) {

        for( int i = 0; i < n; ++i ) {

            tDate d = prepCompoundingDate( i, Start.year() );



            if( d > Start || leftOpen() && d == Start ) {

                if( d <= End )

                    ++c;

                else

                    break;

            }

        }

    }

    else {

            // take care of first and last year

        for( int i = 0; i < n; ++i ) {

            tDate d = prepCompoundingDate( i, Start.year() );



            if( d > Start || leftOpen() && d == Start )

                ++c;

            if( prepCompoundingDate( i, End.year() ) <= End )

                ++c;

        }



            // now take care of intermediate years

        c += ( End.year() - Start.year() - 1 ) * n;

    }



    return c;

}





//

//   i n i t

//



void tCouponCompounder::init()



{

    m_nNumOfPeriods = 1;

    m_pShift = 0;

}





//

//   s e t D a t e T e m p l a t e s

//



void tCouponCompounder::setDateTemplates()



{

    MTG_ASSERT( m_nNumOfPeriods >= 1 && m_nNumOfPeriods <= 12 );

    MTG_ASSERT( 12 % m_nNumOfPeriods == 0 );



    resetDateTemplates();



    int m = 12 / m_nNumOfPeriods;



    int nDay = m_Coupon.isLastDayInMonth() ? 31 : m_Coupon.day();

    int nMonth = m_Coupon.month();



    for( int i = 0; i < m_nNumOfPeriods; ++i ) {

        if( m_pShift == 0 )

            addDateTemplate( nDay, nMonth );

        else

            addDateTemplate( nDay, nMonth, *m_pShift );

        nMonth = nMonth + m;

        if( nMonth > 12 )

            nMonth -= 12;

    }

}





//

//   c o p y F r o m

//



void tCouponCompounder::copyFrom( const tCouponCompounder& Compounder )



{

    if( &Compounder == this )

        return;



    if( m_pShift != 0 ) {

        delete m_pShift;

        m_pShift = 0;

    }



    if( Compounder.m_pShift != 0 )

        m_pShift = Compounder.m_pShift->clone();



    m_Coupon = Compounder.m_Coupon;

    m_Maturity = Compounder.m_Maturity;

    m_nNumOfPeriods = Compounder.m_nNumOfPeriods;



    super::copyFrom( Compounder );

}





//

//   t C o u p o n C o m p o u n d e r

//



tCouponCompounder::tCouponCompounder()



{

    init();

}





//

//   t C o u p o n C o m p o u n d e r

//



tCouponCompounder::tCouponCompounder( tDate Maturity, int nNumOfPeriods )



{

    init();

    set( Maturity, nNumOfPeriods );

}





//

//   t C o u p o n C o m p o u n d e r

//



tCouponCompounder::tCouponCompounder( tDate Maturity, const tDayShift& Shift,

    int nNumOfPeriods )



{

    init();

    set( Maturity, Shift, nNumOfPeriods );

}





//

//   t C o u p o n C o m p o u n d e r

//



tCouponCompounder::tCouponCompounder( tDate Maturity, tDate Coupon, 

    int nNumOfPeriods )



{

    init();

    set( Maturity, Coupon, nNumOfPeriods );

}





//

//   t C o u p o n C o m p o u n d e r

//



tCouponCompounder::tCouponCompounder( tDate Maturity, tDate Coupon,

    const tDayShift& Shift, int nNumOfPeriods )



{

    init();

    set( Maturity, Coupon, Shift, nNumOfPeriods );

}





//

//   t C o u p o n C o m p o u n d e r

//



tCouponCompounder::tCouponCompounder( const tCouponCompounder& Compounder )



{

    init();

    copyFrom( Compounder );

}





//

//   ~ t C o u p o n C o m p o u n d e r

//



tCouponCompounder::~tCouponCompounder()



{

    if( m_pShift != 0 )

        delete m_pShift;

}





//

//   o p e r a t o r =

//



tCouponCompounder& tCouponCompounder::operator=(

    const tCouponCompounder& Compounder )



{

    if( &Compounder != this )

        copyFrom( Compounder );

    return *this;

}





//

//   c l o n e

//



tCompounder* tCouponCompounder::clone() const



{

    return new tCouponCompounder( *this );

}





//

//   s e t

//



void tCouponCompounder::set( tDate Maturity, int nNumOfPeriods )



{

    set( Maturity, Maturity, tIdentityDayShift(), nNumOfPeriods );

}





//

//   s e t

//



void tCouponCompounder::set( tDate Maturity, const tDayShift& Shift,

    int nNumOfPeriods )



{

    set( Maturity, Maturity, Shift, nNumOfPeriods );

}





//

//   s e t

//



void tCouponCompounder::set( tDate Maturity, tDate Coupon, int nNumOfPeriods )



{

    set( Maturity, Coupon, tIdentityDayShift(), nNumOfPeriods );

}





//

//   s e t

//



void tCouponCompounder::set( tDate Maturity, tDate Coupon,

    const tDayShift& Shift, int nNumOfPeriods )



{

    if( m_pShift != 0 )

        delete m_pShift;

    m_pShift = Shift.clone();



    m_Coupon = Coupon;

    m_Maturity = m_pShift->shift( Maturity );

    m_nNumOfPeriods = nNumOfPeriods;



    setDateTemplates();

}





//

//   i s C o m p o u n d i n g D a t e

//



bool tCouponCompounder::isCompoundingDate( tDate Date ) const



{

    if( Date > m_Maturity )

        return false;

    return super::isCompoundingDate( Date );

}





//

//   n u m O f C o m p o u n d i n g D a t e s

//



int tCouponCompounder::numOfCompoundingDates( tDate Start, tDate End ) const



{

    if( End > m_Maturity )

        End = m_Maturity;

    return super::numOfCompoundingDates( Start, End );

}



MTG_END_NAMESPACE





//#define _TEST

#if defined(_TEST)



#if defined(_WIN32)

    #include <conio.h>

#else

    #define getche getchar

#endif



MTG_USING_NAMESPACE



//

//   m a i n

//



void main( int argc, char *argv[] )



{

    printf( "\nTest tCouponCompounder\n\n" );



    char sBuf[256];

    int cCmd, nPeriods;



    tCouponCompounder C1, C2;

    tDate D1, D2, D3;

    bool bCoupon;



    tIdentityDayShift IShift;

    tFollowingDayShift FShift;

    tPrecedingDayShift PShift;

    tModifiedDayShift MShift;



    tDayShift* pShift = &IShift;



    bool bGo = true;



    while( bGo ) { 

        printf( "S<h>ift <S>et Is<D>ate <N>umOfDates <P>eriod E<x>it: " );

        cCmd = getche();

        printf( "\n" );



        switch( cCmd ) {

            case 'h' :

            case 'H' :

                printf( "1=Identity  2=Following\n"

                        "3=Preceding 4=Modified\n"

                        "Choose: " );

                cCmd = getche();

                printf( "\n" );

                switch( cCmd ) {

                    case '1' : pShift = &IShift; break;

                    case '2' : pShift = &FShift; break;

                    case '3' : pShift = &PShift; break;

                    case '4' : pShift = &MShift; break;

                }

                break;



            case 's' :

            case 'S' :

                printf( "Coupon:    " );

                gets( sBuf );

                bCoupon = ( D1.set( sBuf ) != 0 );

                printf( "Maturity: " );

                gets( sBuf );

                if( D2.set( sBuf ) == 0 ) {

                    printf( "Format error\n" );

                }

                else {

                    printf( "Periods:  " );

                    gets( sBuf );

                    sscanf( sBuf, "%d", &nPeriods );

                    if( bCoupon )

                        C2.set( D2, D1, *pShift, nPeriods );

                    else

                        C2.set( D2, *pShift, nPeriods );

                    C1 = C2;

                }

                break;



            case 'd' :

            case 'D' :

                printf( "Date: " );

                gets( sBuf );

                if( D1.set( sBuf ) == 0 ) {

                    printf( "Format error\n" );

                }

                else {

                    if( C1.isCompoundingDate( D1 ) )

                        printf( "Compounding date\n" );

                    else

                        printf( "Not a compounding date\n" );

                }

                break;



            case 'n' :

            case 'N' :

                printf( "Start: " );

                gets( sBuf );

                if( D1.set( sBuf ) == 0 ) {

                    printf( "Format error\n" );

                }

                else {

                    printf( "End:   " );

                    gets( sBuf );

                    if( D2.set( sBuf ) == 0 ) {

                        printf( "Format error\n" );

                    }

                    else {

                        printf( "Number of compounding dates: %d\n",

                            C1.numOfCompoundingDates( D1, D2 ) );

                    }

                }

                break;



            case 'p' :

            case 'P' :

                printf( "Date: " );

                gets( sBuf );

                if( D1.set( sBuf ) == 0 ) {

                    printf( "Format error\n" );

                }

                else {

                    C1.getCompoundingPeriod( D1, D2, D3 );

                    printf( "%02d/%02d/%02d to %02d/%02d/%02d\n",

                        D2.month(), D2.day(), D2.year(),

                        D3.month(), D3.day(), D3.year() );

                }

                break;



            case 'x' :

            case 'X' :

                bGo = false;

                break;

        }

    }



#if ! defined(_WIN32)

    printf( "\n" );

#endif

}



#endif

