// Copyright 1999, 2002 Robert Buff

// Contact: http://robertbuff.com/uvm

//

// This file is part of Mtg-Book.

//

// Mtg-Book is free software; you can redistribute it and/or modify

// it under the terms of the GNU General Public License as published

// by the Free Software Foundation; either version 2 of the License,

// or (at your option) any later version.

//

// Mtg-Book is distributed in the hope that it will be useful,

// but WITHOUT ANY WARRANTY; without even the implied warranty of

// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the

// GNU General Public License for more details.

//

// You should have received a copy of the GNU General Public License

// along with Mtg-Book; if not, write to the 

//

// Free Software Foundation, Inc.

// 59 Temple Place, Suite 330

// Boston, MA 02111-1307

// USA



#include "MtgIncl.h"

#include "MtgCurveContainer.h"

#include "MtgCurve.h"

#include "MtgCurveInstance.h"

#include "MtgImageInstance.h"

#include "MtgParser.h"



MTG_BEGIN_NAMESPACE





//

//   c l e a n u p

//



void tCurveContainer::cleanup()



{

    for( int i = 0; i < m_Item.numOfElems(); ++i ) {

        tItem& I = *m_Item[i];



        I.m_pCurve->downRef();

        if( I.m_pCurve->canDelete() )

            delete I.m_pCurve;



        if( I.m_pInstance != 0 )

            delete I.m_pInstance;



        delete m_Item[i];

    }

    m_Item.reset();

}





//

//   c o p y F r o m

//



void tCurveContainer::copyFrom( const tCurveContainer& Container )



{

    if( &Container == this )

        return;



    cleanup();



        // note that we're not copying the curve instances



    for( int i = 0; i < Container.m_Item.numOfElems(); ++i ) {

        tItem& I = *Container.m_Item[i];

        addCurve( I.m_pCurve, &I.m_OutFile, I.m_pImage, I.m_nTag );

    }

}





//

//   a d d C u r v e

//



void tCurveContainer::addCurve( tCurve* pCurve, const tFileName* pOutFile,

    tImageInstance* pImage, int nTag )



{

    MTG_ASSERT( pCurve != 0 );



    tItem* p = new tItem;



    p->m_pCurve = pCurve;

    p->m_pCurve->upRef();



    if( pOutFile != 0 )

        p->m_OutFile = *pOutFile;



    if( pImage != 0 )

        p->m_pImage = pImage;

    else

        p->m_pImage = 0;



    p->m_nTag = nTag;

    p->m_pInstance = 0;

    p->m_pStealFrom = 0;

    p->m_nRefCount = 0;

    p->m_gBadWeight = 0;



    m_Item.append( p );

}





//

//   t C u r v e C o n t a i n e r

//



tCurveContainer::tCurveContainer()



{

}





//

//   t C u r v e C o n t a i n e r

//



tCurveContainer::tCurveContainer( const tCurveContainer& Container )



{

    copyFrom( Container );

}





//

//   ~ t C u r v e C o n t a i n e r

//



tCurveContainer::~tCurveContainer()



{

    cleanup();

}





//

//   o p e r a t o r =

//



tCurveContainer& tCurveContainer::operator=(

    const tCurveContainer& Container )



{

    if( &Container != this )

        copyFrom( Container );

    return *this;

}





//

//   a d d C u r v e

//



void tCurveContainer::addCurve( tCurve* pCurve, const tFileName* pOutFile )



{

    MTG_ASSERT( pCurve != 0 );

    addCurve( pCurve, pOutFile, 0, 0 );

}





//

//   a d d C u r v e

//



void tCurveContainer::addCurve( tCurve* pCurve, tImageInstance* pImage,

    int nTag )



{

    MTG_ASSERT( pCurve != 0 );

    addCurve( pCurve, 0, pImage, nTag );

}





//

//   m e r g e

//



void tCurveContainer::merge( const tCurveContainer& Container )



{

    if( &Container == this )

        return;



        // simply append, without checking for duplicates



    for( int i = 0; i < Container.m_Item.numOfElems(); ++i ) {

        tItem& I = *Container.m_Item[i];

        addCurve( I.m_pCurve, &I.m_OutFile, I.m_pImage, I.m_nTag );

    }

}





//

//   p r e p a r e

//



tRetCode tCurveContainer::prepare( tEngine& Engine )



{

    tRetCode nRet;



    for( int i = 0; i < m_Item.numOfElems(); ++i ) {

        tItem& I = *m_Item[i];



            // first create curve instance



        if( I.m_pInstance != 0 ) {

            delete I.m_pInstance;

            I.m_pInstance = 0;

        }



        if( ( nRet = I.m_pCurve->createInstance(

                        Engine, I.m_pInstance ) ) != OK ) {

            goto error;

        }

            

        I.m_pStealFrom = 0;



            // now check if speedup is possible



        for( int j = 0; j < i; ++j ) {

            tItem& J = *m_Item[j];



            if( J.m_pStealFrom == 0 &&

                I.m_pInstance->containsEvaluationOf( *J.m_pInstance ) ) {

                tItem* p = &I;



                J.m_pStealFrom = p;

                while( p != 0 ) {

                    ++p->m_nRefCount;

                    p = p->m_pStealFrom;

                }

                break;

            }



            if( J.m_pInstance->containsEvaluationOf( *I.m_pInstance ) ) {

                tItem* p = &J;



                I.m_pStealFrom = p;

                while( p != 0 ) {

                    ++p->m_nRefCount;

                    p = p->m_pStealFrom;

                }

                break;

            }

        }

    }



        // now sort topologically



    for( MTG_FOR_INIT( int ) i = 1; i < m_Item.numOfElems(); ++i ) {

        tItem*p = m_Item[i];

        int j = i;



        while( j > 0 && m_Item[j - 1]->m_nRefCount < p->m_nRefCount ) {

            m_Item[j] = m_Item[j - 1];

            --j;

        }



        m_Item[j] = p;

    }



        // now prepare for evaluation



    for( MTG_FOR_INIT( int ) i = 0; i < m_Item.numOfElems(); ++i ) {

        tItem& I = *m_Item[i];



        if( I.m_pStealFrom == 0 ) {

            if( ( nRet = I.m_pInstance->prepare() ) != OK )

                goto error;

        }

    }



    return OK;



error:



    for( MTG_FOR_INIT( int ) i = 0; i < m_Item.numOfElems(); ++i ) {

        if( m_Item[i]->m_pInstance != 0 ) {

            delete m_Item[i]->m_pInstance;

            m_Item[i]->m_pInstance = 0;

        }

    }



    return nRet;

}





//

//   e v a l u a t e

//



tRetCode tCurveContainer::evaluate( double gWeight )



{

    tRetCode nRet;



    for( int i = 0; i < m_Item.numOfElems(); ++i ) {

        tItem& I = *m_Item[i];



        MTG_ASSERT( I.m_pInstance != 0 );



        if( I.m_pStealFrom == 0 ) {

            if( ( nRet = I.m_pInstance->evaluate( gWeight ) ) != OK )

                return nRet;

        }

    }



    return OK;

}





//

//   p o s t P r o c e s s

//



tRetCode tCurveContainer::postProcess( double gWeight )



{

    tRetCode nRet;



        // first propagate intermediate results among curves



    for( int i = 0; i < m_Item.numOfElems(); ++i ) {

        tItem& I = *m_Item[i];

        MTG_ASSERT( I.m_pInstance != 0 );



        if( I.m_pStealFrom != 0 ) {

            if( ( nRet = I.m_pInstance->stealEvaluationFrom( 

                    *I.m_pStealFrom->m_pInstance ) ) != OK ) {

                return nRet;

            }

        }

    }



        // then finalize results individually



    for( MTG_FOR_INIT( int ) i = 0; i < m_Item.numOfElems(); ++i ) {

        if( ( nRet = m_Item[i]->m_pInstance->postProcess( gWeight ) ) != OK ) {

            if( gWeight >= 0 &&

                ( nRet == OUT_OF_RANGE || nRet == NO_CONVERGENCE ) ) {

                m_Item[i]->m_gBadWeight += gWeight;

                m_Item[i]->m_pInstance->zero();

            }

            else {

                return nRet;

            }

        }

    }



    return OK;

}





//

//   p r o p a g a t e

//



tRetCode tCurveContainer::propagate()



{

    for( int i = 0; i < m_Item.numOfElems(); ++i ) {

        tItem& I = *m_Item[i];



        I.m_pInstance->reweigh( I.m_gBadWeight );

        I.m_gBadWeight = 0;



        if( I.m_pImage != 0 )

            I.m_pImage->registerData( I.m_pInstance, I.m_nTag );

    }



    return OK;

}





//

//   s a v e

//



tRetCode tCurveContainer::save() const



{

    tRetCode nRet;



    for( int i = 0; i < m_Item.numOfElems(); ++i ) {

        tItem& I = *m_Item[i];



        MTG_ASSERT( I.m_pInstance != 0 );



        if( ( nRet = I.m_pCurve->save( I.m_OutFile,

                        *I.m_pInstance ) ) != OK ) {

            return nRet;

        }

    }



    return OK;

}





//

//   p a r s e

//



tRetCode tCurveContainer::parse( tParser& Parser )



{

    tRetCode nRet;

    tObject* pObj;

    tCurve* pCurve;

    tFileName OutFile;



    switch( Parser.curToken() ) {

        case xTokCurve :

            if( ( nRet = Parser.readToken() ) != OK )

                return nRet;

            // fall through



        case xTokId :

            if( Parser.curToken() != xTokId )

                return Parser.setError( INVALID_KEYWORD );

            if( ( pObj = Parser.findObject() ) == 0 )

                return Parser.setError( NOT_FOUND );

            if( ( pCurve = dynamic_cast<tCurve*>( pObj ) ) == 0 ) 

                return Parser.setError( "curve expected" );

            if( ( nRet = Parser.readToken() ) != OK )

                return nRet;

            if( Parser.curToken() == xTokString ) {

                if( ( nRet = OutFile.parseWrite( Parser ) ) != OK )

                    return nRet;

            }

            break;



        default :

            return GO;

    }



    addCurve( pCurve, &OutFile );



    return OK;

}



MTG_END_NAMESPACE



