// Copyright 1999, 2002 Robert Buff

// Contact: http://robertbuff.com/uvm

//

// This file is part of Mtg-Book.

//

// Mtg-Book is free software; you can redistribute it and/or modify

// it under the terms of the GNU General Public License as published

// by the Free Software Foundation; either version 2 of the License,

// or (at your option) any later version.

//

// Mtg-Book is distributed in the hope that it will be useful,

// but WITHOUT ANY WARRANTY; without even the implied warranty of

// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the

// GNU General Public License for more details.

//

// You should have received a copy of the GNU General Public License

// along with Mtg-Book; if not, write to the 

//

// Free Software Foundation, Inc.

// 59 Temple Place, Suite 330

// Boston, MA 02111-1307

// USA



#include "MtgIncl.h"

#include "MtgCustomClaim.h"

#include "MtgEvGenerator.h"

#include "MtgExPolicyExpr.h"

#include "MtgFactor.h"

#include "MtgFDEngine.h"

#include "MtgInterestSpline.h"

#include "MtgNumericalExpr.h"

#include "MtgParser.h"



MTG_BEGIN_NAMESPACE





//

//   t C u s t o m C a s h f l o w

//



tCustomClaim::tCustomCashflow::tCustomCashflow( tSystem& System, int nDay,

    tNumericalExpr* pExpr ) :

    tCashflow( System, nDay )



{

    MTG_ASSERT( pExpr != 0 );

    m_pExpr = pExpr;

}



    

//

//   t C u s t o m C a s h f l o w

//

    

tCustomClaim::tCustomCashflow::tCustomCashflow( tSystem& System, tDate Date,

    tNumericalExpr* pExpr ) :

    tCashflow( System, Date )



{

    MTG_ASSERT( pExpr != 0 );

    m_pExpr = pExpr;

}



    

//

//   t C u s t o m C a s h f l o w

//

    

tCustomClaim::tCustomCashflow::tCustomCashflow(

    const tCustomCashflow& CC ) :

    tCashflow( CC )



{

    m_pExpr = new tNumericalExpr( *CC.m_pExpr );

}





//

//   ~ t C u s t o m C a s h f l o w

//

    

tCustomClaim::tCustomCashflow::~tCustomCashflow()



{

    if( m_pExpr != 0 )

        delete m_pExpr;

}





//

//   c l o n e

//



tCashflow* tCustomClaim::tCustomCashflow::clone() const



{

    return new tCustomCashflow( *this );

}





//

//   g e n e r a t e

//



double tCustomClaim::tCustomCashflow::generate( tEngine& Engine,

    const tClaim& Claim )



{

    MTG_ASSERT( m_pExpr != 0 );

    return m_pExpr->apply( Engine );

}





//

//   i n i t

//



void tCustomClaim::init()



{

    m_pPayoff = 0;

    m_pKnockoutPayoff = 0;

    m_pExercisePayoff = 0;

    m_pUpBarrier = 0;

    m_pDownBarrier = 0;

    m_pMonitor = 0;

}





//

//   r e g C a s h f l o w

//



void tCustomClaim::regCashflow( int nDay, tNumericalExpr *pCashflow )



{

    MTG_ASSERT( nDay >= 0 && pCashflow != 0 );



    tCustomCashflow *pCF =

        new tCustomCashflow( system(), nDay, pCashflow );



    super::regCashflow( pCF );

    touch();

}





//

//   r e g C a s h f l o w

//



void tCustomClaim::regCashflow( tDate Date, tNumericalExpr *pCashflow )



{

    MTG_ASSERT( Date >= system().base() && pCashflow != 0 );



    tCustomCashflow *pCF =

        new tCustomCashflow( system(), Date, pCashflow );



    super::regCashflow( pCF );

    touch();

}





//

//   g e t E v e n t s

//



void tCustomClaim::getEvents( tSeqEvGenerator& EvGen ) const



{

    super::getEvents( EvGen );



    if( m_pUpBarrier != 0 )

        m_pUpBarrier->getEvents( EvGen );

    if( m_pDownBarrier != 0 )

        m_pDownBarrier->getEvents( EvGen );

    if( m_pMonitor != 0 )

        m_pMonitor->getEvents( EvGen );

}





//

//   g e t B a r r i e r s

//



void tCustomClaim::getBarriers( const tFactor* pFactor, bool bMain,

    tHeap<double>& Barrier ) const



{

    super::getBarriers( pFactor, bMain, Barrier );



    if( m_pUpBarrier != 0 )

        m_pUpBarrier->getConstants( Barrier );

    if( m_pDownBarrier != 0 )

        m_pDownBarrier->getConstants( Barrier );

    if( m_pMonitor != 0 )

        m_pMonitor->getBarriers( pFactor, Barrier );

}





//

//   p a y o f f

//



double tCustomClaim::payoff( tEngine& Engine )



{

    MTG_ASSERT( isResolved() );



    if( m_pPayoff != 0 )

        return m_pPayoff->apply( Engine );

    return 0;

}





//

//   k n o c k o u t P a y o f f

//



double tCustomClaim::knockoutPayoff( tEngine& Engine )



{

    MTG_ASSERT( isResolved() );



    tFDEngine* pFD = dynamic_cast<tFDEngine*>( &Engine );



    if( pFD == 0 )

        throw tException( NOT_AVAILABLE );



    if( m_pKnockoutPayoff != 0 )

        return m_pKnockoutPayoff->apply( Engine );

    return 0;

}





//

//   e x e r c i s e P a y o f f

//



double tCustomClaim::exercisePayoff( tEngine& Engine )



{

    MTG_ASSERT( isResolved() );



    tFDEngine* pFD = dynamic_cast<tFDEngine*>( &Engine );



    if( pFD == 0 )

        throw tException( NOT_AVAILABLE );



    if( m_pExercisePayoff != 0 )

        return m_pExercisePayoff->apply( Engine );

    if( m_pPayoff != 0 )

        return m_pPayoff->apply( Engine );

    return 0;

}





//

//   u p B a r r i e r

//



bool tCustomClaim::upBarrier( tEngine& Engine, double& gBarrier )



{

    if( m_pUpBarrier != 0 ) {

        MTG_ASSERT( isResolved() && hasUpBarrier() );



        bool bValid;



        gBarrier = m_pUpBarrier->apply( Engine, bValid );

        return bValid;

    }

    return super::upBarrier( Engine, gBarrier );

}





//

//   d o w n B a r r i e r

//



bool tCustomClaim::downBarrier( tEngine& Engine, double& gBarrier )



{

    if( m_pDownBarrier != 0 ) {

        MTG_ASSERT( isResolved() && hasDownBarrier() );



        bool bValid;



        gBarrier = m_pDownBarrier->apply( Engine, bValid );

        return bValid;                            

    }

    return super::downBarrier( Engine, gBarrier );

}





//

//   m o n i t o r

//



tExPolicy tCustomClaim::monitor( tEngine& Engine, double gUnitValue )



{

    MTG_ASSERT( isResolved() && isMonitored() );



    tFDEngine* pFD = dynamic_cast<tFDEngine*>( &Engine );



    if( pFD == 0 )

        throw tException( NOT_AVAILABLE );



    if( m_pMonitor != 0 ) {

        bool bValid;

        return m_pMonitor->apply( Engine, gUnitValue, bValid );                        

    }

    return xDontExercise;

}





//

//   p a r s e B a r r i e r

//



tRetCode tCustomClaim::parseBarrier( tParser& Parser )

    

{

    MTG_ASSERT( Parser.curToken() == xTokUpAndOut ||

                Parser.curToken() == xTokDownAndOut );



    tRetCode nRet;

    double gBarrier;

    tNumericalExpr* pBarrier;



    switch( Parser.curToken() ) {

        case xTokUpAndOut :

            if( hasUpBarrier() ) {

                return Parser.setError( "'up_and_out' barrier "

                                    "defined more than once" );

            }

            if( ( nRet = Parser.readToken() ) != OK )

                return nRet;



            MTG_ASSERT( m_pUpBarrier == 0 );



            pBarrier = new tNumericalExpr;

            if( ( nRet = pBarrier->parse( Parser,

                    tExpression::ALLOW_FDMC ) ) != OK ) {

                delete pBarrier;

                return nRet;

            }

            if( pBarrier->isNumber( gBarrier ) ) {

                delete pBarrier;

                setUpBarrier( gBarrier );

            }

            else {

                m_pUpBarrier = pBarrier;

                setUpBarrier( -1 );

            }

            break;



        case xTokDownAndOut :

            if( hasDownBarrier() ) {

                return Parser.setError( "'down_and_out' barrier "

                                    "defined more than once" );

            }

            if( ( nRet = Parser.readToken() ) != OK )

                return nRet;



            MTG_ASSERT( m_pDownBarrier == 0 );



            pBarrier = new tNumericalExpr;

            if( ( nRet = pBarrier->parse( Parser,

                    tExpression::ALLOW_FDMC ) ) != OK ) {

                delete pBarrier;

                return nRet;

            }

            if( pBarrier->isNumber( gBarrier ) ) {

                delete pBarrier;

                setDownBarrier( gBarrier );

            }

            else {

                m_pDownBarrier = pBarrier;

                setDownBarrier( -1 );

            }

            break;



        default :

            break;

    }



    return OK;

}





//

//   p a r s e P a r a m

//



tRetCode tCustomClaim::parseParam( tParser& Parser, tParseInfoStub& Info )



{

    tRetCode nRet;

    tDate Date;

    tNumericalExpr* pNumerical;

    tExPolicyExpr* pExPolicy;



restart:



        // "restart" is an option that helps if the user

        // enters "payoff_* { .. } monitor { .. }" etc. in the

        // user interface without putting the required

        // comma after the payoff construct. In this case

        // (and only in this case) we're lenient and don't

        // report a syntax error.



    switch( Parser.curToken() ) {

        case xTokPayoff :

            if( m_pPayoff != 0 )

                return Parser.setError( "'payoff' defined more than once" );

            pNumerical = new tNumericalExpr;

            if( ( nRet = Parser.readToken() ) != OK ||

                ( nRet = pNumerical->parse( Parser,

                            tExpression::ALLOW_ALL &

                            ~tNumericalExpr::ALLOW_UNITVALUE ) ) != OK ) {

                delete pNumerical;

                return nRet;

            }

            m_pPayoff = pNumerical;

            if( Parser.curToken() == xTokMonitor ||

                Parser.curToken() == xTokKnockoutPayoff ||

                Parser.curToken() == xTokExercisePayoff ) {

                goto restart;

            }

            break;



        case xTokKnockoutPayoff :

            if( m_pKnockoutPayoff != 0 ) {

                return Parser.setError( "'knockout_payoff' defined "

                                            "more than once" );

            }

            pNumerical = new tNumericalExpr;

            if( ( nRet = Parser.readToken() ) != OK ||

                ( nRet = pNumerical->parse( Parser,

                            tExpression::ALLOW_ALL &

                            ~tNumericalExpr::ALLOW_UNITVALUE ) ) != OK ) {

                delete pNumerical;

                return nRet;

            }

            m_pKnockoutPayoff = pNumerical;

            if( Parser.curToken() == xTokMonitor ||

                Parser.curToken() == xTokPayoff ||

                Parser.curToken() == xTokExercisePayoff ) {

                goto restart;

            }

            break;



        case xTokExercisePayoff :

            if( m_pExercisePayoff != 0 ) {

                return Parser.setError( "'exercise_payoff' defined "

                                            "more than once" );

            }

            pNumerical = new tNumericalExpr;

            if( ( nRet = Parser.readToken() ) != OK ||

                ( nRet = pNumerical->parse( Parser,

                            tExpression::ALLOW_ALL &

                            ~tNumericalExpr::ALLOW_UNITVALUE ) ) != OK ) {

                delete pNumerical;

                return nRet;

            }

            m_pExercisePayoff = pNumerical;

            if( Parser.curToken() == xTokMonitor ||

                Parser.curToken() == xTokPayoff ||

                Parser.curToken() == xTokKnockoutPayoff ) {

                goto restart;

            }

            break;



        case xTokMonitor :

            if( m_pMonitor != 0 )

                return Parser.setError( "'monitor' defined more than once" );

            pExPolicy = new tExPolicyExpr;

            if( ( nRet = Parser.readToken() ) != OK ||

                ( nRet = pExPolicy->parse( Parser ) ) != OK ) {

                delete pExPolicy;

                return nRet;

            }

            m_pMonitor = pExPolicy;

            setMonitor();

            if( Parser.curToken() == xTokPayoff ||

                Parser.curToken() == xTokKnockoutPayoff ||

                Parser.curToken() == xTokExercisePayoff ) {

                goto restart;

            }

            break;



        case xTokCashflow :

            if( ( nRet = Parser.readToken() ) != OK ) 

                return nRet;

            if( Parser.tryDate( Date ) ) {

                if( Date < system().base() )

                    return INVALID_DATE;

                if( ( nRet = Parser.readToken() ) != OK )

                    return nRet;

                pNumerical = new tNumericalExpr;

                if( ( nRet = pNumerical->parse( Parser,

                                tExpression::ALLOW_ALL &

                                ~tNumericalExpr::ALLOW_UNITVALUE ) ) != OK ) {

                    delete pNumerical;

                    return nRet;

                }

                regCashflow( Date, pNumerical );

            }

            else {

                int nDay;



                if( ( nRet = Parser.scanInteger( nDay, 0 ) ) != OK )

                    return nRet;

                pNumerical = new tNumericalExpr;

                if( ( nRet = pNumerical->parse( Parser,

                                tExpression::ALLOW_ALL &

                                ~tNumericalExpr::ALLOW_UNITVALUE ) ) != OK ) {

                    delete pNumerical;

                    return nRet;

                }

                regCashflow( nDay, pNumerical );

            }

            break;



        case xTokUpAndOut :

        case xTokDownAndOut :

                // This overrides the handling of the barriers

                // by the super type.

            if( ( nRet = parseBarrier( Parser ) ) != OK )

                return nRet;

            break;



        default :

            return super::parseParam( Parser, Info );

    }



    return OK;

}





//

//   t C u s t o m C l a i m

//



tCustomClaim::tCustomClaim()



{

    init();

}





//

//   t C u s t o m C l a i m

//



tCustomClaim::tCustomClaim( const tCustomClaim& Claim )



{

    init();

    copyFrom( Claim );

}





//

//   ~ t C u s t o m C l a i m

//



tCustomClaim::~tCustomClaim()



{

    if( m_pPayoff != 0 )

        delete m_pPayoff;



    if( m_pKnockoutPayoff != 0 )

        delete m_pKnockoutPayoff;



    if( m_pExercisePayoff != 0 )

        delete m_pExercisePayoff;



    if( m_pUpBarrier != 0 )

        delete m_pUpBarrier;



    if( m_pDownBarrier != 0 )

        delete m_pDownBarrier;



    if( m_pMonitor != 0 )

        delete m_pMonitor;

}





//

//   o p e r a t o r =

//



tCustomClaim& tCustomClaim::operator=( const tCustomClaim& Claim )



{

    if( &Claim != this )

        copyFrom( Claim );

    return *this;

}





//

//   c o p y F r o m

//



void tCustomClaim::copyFrom( const tCustomClaim& Claim )



{

    if( &Claim == this )

        return;



    if( m_pPayoff != 0 ) {

        delete m_pPayoff;

        m_pPayoff = 0;

    }

    if( m_pKnockoutPayoff != 0 ) {

        delete m_pKnockoutPayoff;

        m_pKnockoutPayoff = 0;

    }

    if( m_pExercisePayoff != 0 ) {

        delete m_pExercisePayoff;

        m_pExercisePayoff = 0;

    }

    if( m_pUpBarrier != 0 ) {

        delete m_pUpBarrier;

        m_pUpBarrier = 0;

    }

    if( m_pDownBarrier != 0 ) {

        delete m_pDownBarrier;

        m_pDownBarrier = 0;

    }

    if( m_pMonitor != 0 ) {

        delete m_pMonitor;

        m_pMonitor = 0;

    }



    if( Claim.m_pPayoff != 0 )

        m_pPayoff = new tNumericalExpr( *Claim.m_pPayoff );

    if( Claim.m_pKnockoutPayoff != 0 )

        m_pKnockoutPayoff = new tNumericalExpr( *Claim.m_pKnockoutPayoff );

    if( Claim.m_pExercisePayoff != 0 )

        m_pExercisePayoff = new tNumericalExpr( *Claim.m_pExercisePayoff );

    if( Claim.m_pUpBarrier != 0 ) 

        m_pUpBarrier = new tNumericalExpr( *Claim.m_pUpBarrier );

    if( Claim.m_pDownBarrier != 0 ) 

        m_pDownBarrier = new tNumericalExpr( *Claim.m_pDownBarrier );

    if( Claim.m_pMonitor != 0 ) 

        m_pMonitor = new tExPolicyExpr( *Claim.m_pMonitor );



    super::copyFrom( Claim );

}





//

//   c l o n e

//



tObject* tCustomClaim::clone() const



{

    return new tCustomClaim( *this );

}



    

//

//   f i n a l i z e

//



tRetCode tCustomClaim::finalize()



{

    if( isFinalized() )

        return OK;



    return super::finalize();

}





//

//   r e s o l v e

//



tRetCode tCustomClaim::resolve( tLookup<int>& Claim,

    tLookup<int>& Factor, tHeap<tFactor*>& FactorPtr, tPortfolio& Pf )



{

    tRetCode nRet;



    if( isResolved() )

        return OK;



    for( int k = 0; k < m_Cashflow.numOfElems(); ++k ) {

        tCustomCashflow* p = static_cast<tCustomCashflow*>( m_Cashflow[k] );

        if( ( nRet = p->m_pExpr->resolve( id(), Claim, Factor,

                FactorPtr, Pf, tNumericalExpr::ALLOW_FACTOR ) ) != OK ) {

            return nRet;

        }

    }



    if( m_pPayoff != 0 ) {

        if( ( nRet = m_pPayoff->resolve( id(), Claim, Factor,

                FactorPtr, Pf, tNumericalExpr::ALLOW_FACTOR ) ) != OK ) {

            return nRet;

        }

    }



    if( m_pKnockoutPayoff != 0 ) {

        if( ( nRet = m_pKnockoutPayoff->resolve( id(), Claim, Factor,

                FactorPtr, Pf, tNumericalExpr::ALLOW_FACTOR ) ) != OK ) {

            return nRet;

        }

    }



    if( m_pExercisePayoff != 0 ) {

        if( ( nRet = m_pExercisePayoff->resolve( id(), Claim, Factor,

                FactorPtr, Pf, tNumericalExpr::ALLOW_FACTOR ) ) != OK ) {

            return nRet;

        }

    }



    if( m_pUpBarrier != 0 ) {

        if( ( nRet = m_pUpBarrier->resolve( id(), Claim, Factor,

                FactorPtr, Pf, 0 ) ) != OK ) {

            return nRet;

        }

    }



    if( m_pDownBarrier != 0 ) {

        if( ( nRet = m_pDownBarrier->resolve( id(), Claim, Factor,

                FactorPtr, Pf, 0 ) ) != OK ) {

            return nRet;

        }

    }



    if( m_pMonitor != 0 ) {

        if( ( nRet = m_pMonitor->resolve( id(), Claim, Factor,

                FactorPtr, Pf ) ) != OK ) {

            return nRet;

        }

    }



    return super::resolve( Claim, Factor, FactorPtr, Pf );

}





//

//   a d d P a y m e n t S t r e a m

//



bool tCustomClaim::addPaymentStream( tInterestSpline& Spline ) const



{

    if( ! isPriced() )

        return false;



    tHeap<tPayment> Payment;



    for( int i = 0; i < m_Cashflow.numOfElems(); ++i ) {

        tCustomCashflow* p =

            dynamic_cast<tCustomCashflow*>( m_Cashflow[i] );



        MTG_ASSERT( p != 0 );                             



        double gValue;



        if( p->m_pExpr->isNumber( gValue ) )

            Payment.append( tPayment( p->date(), gValue ) );

        else

            return false;

    }



    if( m_pPayoff != 0 ) {

        double gValue;



        if( m_pPayoff->isNumber( gValue ) )

            Payment.append( tPayment( maturityDate(), gValue ) );

        else

            return false;

    }



    tDate Base = const_cast<tCustomClaim*>( this )->system().base();



    Spline.addPayment( Base,

        ( unitAskPrice() + unitBidPrice() ) / 2, Payment );



    return true;

}



MTG_END_NAMESPACE