// Copyright 1999, 2002 Robert Buff

// Contact: http://robertbuff.com/uvm

//

// This file is part of Mtg-Book.

//

// Mtg-Book is free software; you can redistribute it and/or modify

// it under the terms of the GNU General Public License as published

// by the Free Software Foundation; either version 2 of the License,

// or (at your option) any later version.

//

// Mtg-Book is distributed in the hope that it will be useful,

// but WITHOUT ANY WARRANTY; without even the implied warranty of

// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the

// GNU General Public License for more details.

//

// You should have received a copy of the GNU General Public License

// along with Mtg-Book; if not, write to the 

//

// Free Software Foundation, Inc.

// 59 Temple Place, Suite 330

// Boston, MA 02111-1307

// USA



#include "MtgIncl.h"

#include "MtgDataScanner.h"



MTG_BEGIN_NAMESPACE





//

//   t D a t a S c a n n e r

//



tDataScanner::tDataScanner()



{

    m_pAutoSrc = 0;

}





//

//   t D a t a S c a n n e r

//



tDataScanner::tDataScanner( tSource& Src )



{

    m_pAutoSrc = 0;

    setSource( Src );

}





//

//   t D a t a S c a n n e r

//



tDataScanner::tDataScanner( tSource* pSrc )



{

    m_pAutoSrc = 0;

    setSource( pSrc );

}





//

//   t D a t a S c a n n e r

//



tDataScanner::tDataScanner( const tFileName& FName )



{

    tRetCode nRet;



    if( ! FName.isValid() ) {

        m_pAutoSrc = 0;

        if( FName.isTry() )

            throw tException( GO );

        throw tException( OPEN_ERROR );

    }



    m_pAutoSrc = new tFileSource( FName.name() );



    if( ( nRet = setSource( m_pAutoSrc ) ) != OK ) {

        delete m_pAutoSrc;

        m_pAutoSrc = 0;

        if( nRet == OPEN_ERROR && FName.isTry() )

            throw tException( GO );

        throw tException( nRet );

    }

}





//

//   s e t S o u r c e

//



tRetCode tDataScanner::setSource( const tFileName* pFName )



{

    MTG_ASSERT( pFName != 0 );



    tRetCode nRet;



    if( m_pAutoSrc != 0 ) {

        delete m_pAutoSrc;

        m_pAutoSrc = 0;

    }



    if( ! pFName->isValid() ) {

        if( pFName->isTry() )

            return GO;

        return OPEN_ERROR;

    }



    m_pAutoSrc = new tFileSource( pFName->name() );



    if( ( nRet = setSource( m_pAutoSrc ) ) != OK ) {

        delete m_pAutoSrc;

        m_pAutoSrc = 0;

        if( nRet == OPEN_ERROR && pFName->isTry() )

            return GO;

        return nRet;

    }



    return OK;

}





//

//   ~ t D a t a S c a n n e r

//



tDataScanner::~tDataScanner()



{

    cleanup();

    if( m_pAutoSrc != 0 )

        delete m_pAutoSrc;

}





//

//   s c a n S t r i n g

//



tRetCode tDataScanner::scanString( char*& sString )



{

    tRetCode nRet;



    if( curToken() == xTokString )

        return super::scanString( sString );



        // take anything as string

    char* s = StrCopy( curText() );



    if( ( nRet = readToken() ) != OK ) {

        delete s;

        return setError( nRet );

    }

    sString = s;

    return OK;

}



#if defined(_MTG_WITH_TCL)





//

//   s c a n O b j e c t

//



tRetCode tDataScanner::scanObject( Tcl_Obj*& pObj )



{

    tRetCode nRet;



        // notice that the way we treat numbers makes   

        // it impossible to have single '-' and '+'

        // in the data stream; they have to be enclosed

        // by quotes



    if( curToken() == xTokEOF )

        return setError( END_OF_FILE );



    if( beginOfNumber() ) {

        double gValue;



        if( ( nRet = scanDouble( gValue ) ) != OK )

            return nRet;

        if( gValue == (int) gValue )

            pObj = Tcl_NewIntObj( (int) gValue );

        else

            pObj = Tcl_NewDoubleObj( gValue );

        return OK;

    }



    Tcl_Obj* p = Tcl_NewStringObj( const_cast<char*>( curText() ), -1 );



    if( ( nRet = readToken() ) != OK ) {

        delete p;

        return setError( nRet );

    }



    pObj = p;

    return OK;

}



#endif





//

//   a p p e n d

//



tRetCode tDataScanner::append( tHeap<double>& Data )



{

    tRetCode nRet;

    double gValue;



    while( beginOfNumber() ) {

        if( ( nRet = scanDouble( gValue ) ) != OK )

            return nRet;

        Data.append( gValue );

    }



    return OK;

}





//

//   a p p e n d

//



tRetCode tDataScanner::append( tHeap<double>& Data, double gLoBound,

    double gHiBound )



{

    tRetCode nRet;

    double gValue;



    while( beginOfNumber() ) {

        if( ( nRet = scanDouble( gValue ) ) != OK )

            return nRet;

        if( gValue < gLoBound || gValue > gHiBound )

            return setError( OUT_OF_RANGE );

        Data.append( gValue );

    }



    return OK;

}



#if defined(_MTG_WITH_TCL)





//

//   a p p e n d

//



tRetCode tDataScanner::append( tHeap<Tcl_Obj*>& Data )



{

    tRetCode nRet;

    Tcl_Obj* pObj;



    if( atEOF() )

        return OK;



    while( ! atEOF() && ( nRet = scanObject( pObj ) ) == OK )

        Data.append( pObj );



    return nRet;

}



#endif





//

//   f i l l

//



tRetCode tDataScanner::fill( tHeap<double>& Data )



{

    tRetCode nRet;



    for( int k = 0; k < Data.numOfElems(); ++k ) {

        if( ( nRet = scanDouble( Data[k] ) ) != OK )

            return nRet;

    }



    return OK;

}





//

//   f i l l

//



tRetCode tDataScanner::fill( tHeap<double>& Data, double gLoBound,

    double gHiBound )



{

    tRetCode nRet;



    for( int k = 0; k < Data.numOfElems(); ++k ) {

        if( ( nRet = scanDouble( Data[k] ) ) != OK )

            return nRet;

        if( Data[k] < gLoBound || Data[k] > gHiBound )

            return setError( OUT_OF_RANGE );

    }



    return OK;

}





//

//   a t E O F

//



bool tDataScanner::atEOF() const



{

    return curToken() == xTokEOF;

}



MTG_END_NAMESPACE



