// Copyright 1999, 2002 Robert Buff

// Contact: http://robertbuff.com/uvm

//

// This file is part of Mtg-Book.

//

// Mtg-Book is free software; you can redistribute it and/or modify

// it under the terms of the GNU General Public License as published

// by the Free Software Foundation; either version 2 of the License,

// or (at your option) any later version.

//

// Mtg-Book is distributed in the hope that it will be useful,

// but WITHOUT ANY WARRANTY; without even the implied warranty of

// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the

// GNU General Public License for more details.

//

// You should have received a copy of the GNU General Public License

// along with Mtg-Book; if not, write to the 

//

// Free Software Foundation, Inc.

// 59 Temple Place, Suite 330

// Boston, MA 02111-1307

// USA



#if ! defined(_MTG_DATE_BASE_)

#define _MTG_DATE_BASE_



#include "MtgDayCount.h"



MTG_BEGIN_NAMESPACE



class tParse;





//

//   t D a t e B a s e

//



class tDateBase {



    tScale m_nScale;        // default is daily

    int m_nNumOfPeriods;    // number of periods, equally sized,

                            // if the scale is "year"



    tDayCount m_DayCount;   // default is ACT/365



    double m_gDtTU;         // implied (mathematical) time steps of

    double m_gDtSI;         // time units, sub intervals and days.

    double m_gDtDay;



        // (If the scale is days, m_gDt can be a little tricky -

        // by (our arbitrary) definition, we take the time step to be the

        // day count fraction between 1/1/97 and 1/2/97, a non-leap year.)



    tDate m_Base;           // default is tDate::today()



    void init( tScale nScale, int nNumOfPeriods,

        const tDayCount& DayCount, tDate Base );



    void deriveDt();



protected:



    void copyFrom( const tDateBase& TS );



        // Parse ONE parameter in:



    tRetCode parse( tParser& Parser );



        // This is the basic service rendered by the object:

        // translate from a concrete date to the proper time

        // unit, or subinterval of time unit, using the day

        // count convention and the base date.



        // In the getTU() functions, the return value will

        // be the time unit in days or fractions of years.



    void getTU( tDate Date, int& nUnit ) const;

    void getTU( tDate Date, long& nUnit ) const;

    void getTU( tDate Date, double& gUnit ) const;



    double getTU( tDate Date ) const {

        double gUnit;



        getTU( Date, gUnit );

        return gUnit;

    }



        // In the getSI() functions, the return value will

        // be the time unit in days if the scale is xDay,

        // and in multiples of 1/m_numOfPeriods if the

        // scale is xYear.



    void getSI( tDate Date, int& nSub ) const;

    void getSI( tDate Date, long& nSub ) const;

    void getSI( tDate Date, double& gSub ) const;



    double getSI( tDate Date ) const {

        double gSub;



        getSI( Date, gSub );

        return gSub;

    }



public:



    tDateBase();

    tDateBase( tScale nScale );

    tDateBase( tScale nScale, int nNumOfPeriods );

    tDateBase( tScale nScale, const tDayCount& DayCount );

    tDateBase( tScale nScale, int nNumOfPeriods, const tDayCount& DayCount );

    tDateBase( tScale nScale, const tDayCount& DayCount,

        tDate Base );

    tDateBase( tScale nScale, int nNumOfPeriods, const tDayCount& DayCount,

        tDate Base );



    tDateBase( const tDateBase& DB );



    ~tDateBase();



    tDateBase& operator=( const tDateBase& DB );



        // Set parameters individually:



    void setScale( tScale nScale );

    void setScale( tScale nScale, int nNumOfPeriods );

    void setDayCount( const tDayCount& DayCount );



    void setBase( tDate Base );



        // Retrieve parameters individually:



    tScale scale() const {

        return m_nScale;

    }



    int numOfPeriods() const {

        return m_nNumOfPeriods;

    }



    const tDayCount& dayCount() const {

        return m_DayCount;

    }



    tDate base() const {

        return m_Base;

    }



    double dtTU() const {

        return m_gDtTU;

    }



    double dtSI() const {

        return m_gDtSI;

    }



    double dtDay() const {

        return m_gDtDay;

    }

    

        // Run some tests.



    bool sameDateBaseAs( const tDateBase& Base ) const;

    bool compatibleDateBaseWith( const tDateBase& Base ) const;



        // Is the date between the base date end the end date,

        // inclusive?



    bool isBetween( tDate Date, tDate End ) const;



        // Find the number of steps from the base date to the

        // maturity date.



    int getNumOfSteps( tDate Maturity, int nExtraSteps = 0 ) const;



        // Find alpha and beta such that a subinterval S in the

        // current date base is subinterval alpha * S + beta

        // in date base DB. If the scales are different, this

        // uses the dt value stored in m_gDt. The day count

        // conventions must be compatible!



    tRetCode getTranslationSI( const tDateBase& DB, double& gAlpha,

        double& gBeta ) const;



        // Find the date closest to the subinterval gSub, by

        // using m_gDt to invert. Note that this is approximate.



    tDate getClosestDateSI( double gSub ) const;



        // Convert the sub interval into the day count fraction,

        // under the registered day count convention, or under the

        // one passed as argument.



    double getFractionSI( double gSub ) const;

    double getFractionSI( double gSub, const tDayCount& DayCount ) const;

};





//

//   f u n c t i o n s

//



    // Output to stream:



ostream& operator<<( ostream& Out, const tDateBase& DB );



MTG_END_NAMESPACE



#endif

