// Copyright 1999, 2002 Robert Buff

// Contact: http://robertbuff.com/uvm

//

// This file is part of Mtg-Book.

//

// Mtg-Book is free software; you can redistribute it and/or modify

// it under the terms of the GNU General Public License as published

// by the Free Software Foundation; either version 2 of the License,

// or (at your option) any later version.

//

// Mtg-Book is distributed in the hope that it will be useful,

// but WITHOUT ANY WARRANTY; without even the implied warranty of

// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the

// GNU General Public License for more details.

//

// You should have received a copy of the GNU General Public License

// along with Mtg-Book; if not, write to the 

//

// Free Software Foundation, Inc.

// 59 Temple Place, Suite 330

// Boston, MA 02111-1307

// USA



#include "MtgIncl.h"

#include "MtgDayCount.h"

#include "MtgParser.h"



MTG_BEGIN_NAMESPACE





//

//    o b j e c t s

//



tDayCount DayCountACT_ACT( tDayCount::xACT_ACT );

tDayCount DayCountACT_365( tDayCount::xACT_365 );

tDayCount DayCountACT_365_25( tDayCount::xACT_365_25 );

tDayCount DayCountACT_365_NL( tDayCount::xACT_365_NL );

tDayCount DayCountACT_365_ISDA( tDayCount::xACT_365_ISDA );

tDayCount DayCountACT_360( tDayCount::xACT_360 );

tDayCount DayCount30_360_ISDA( tDayCount::x30_360_ISDA );

tDayCount DayCount30_360_PSA( tDayCount::x30_360_PSA );

tDayCount DayCount30_360_SIA( tDayCount::x30_360_SIA );

tDayCount DayCount30E_360( tDayCount::x30E_360 );





//

//   n u m O f D a y s A C T _ A C T

//



long tDayCount::numOfDaysACT_ACT( tDate Start, tDate End,

    bool bCouponOnLastDayOfFeb ) const



{

    return End - Start;

}





//

//   n u m O f D a y s A C T _ 3 6 5

//



long tDayCount::numOfDaysACT_365( tDate Start, tDate End,

    bool bCouponOnLastDayOfFeb ) const



{

    return End - Start;

}





//

//   n u m O f D a y s A C T _ 3 6 5 _ 2 5

//



long tDayCount::numOfDaysACT_365_25( tDate Start, tDate End,

    bool bCouponOnLastDayOfFeb ) const



{

    return End - Start;

}





//

//   n u m O f D a y s A C T _ 3 6 5 _ N L

//



long tDayCount::numOfDaysACT_365_NL( tDate Start, tDate End,

    bool bCouponOnLastDayOfFeb ) const



{

        // Exclude leap days in the numerator.

    return End - Start - ( Start + 1 ).numOfLeapDaysTo( End );

}





//

//   n u m O f D a y s A C T _ 3 6 5 _ I S D A

//



long tDayCount::numOfDaysACT_365_ISDA( tDate Start, tDate End,

    bool bCouponOnLastDayOfFeb ) const



{

    return End - Start;

}





//

//   n u m O f D a y s A C T _ 3 6 0

//



long tDayCount::numOfDaysACT_360( tDate Start, tDate End,

    bool bCouponOnLastDayOfFeb ) const



{

    return End - Start;

}





//

//   n u m O f D a y s 3 0 _ 3 6 0 _ I S D A

//



long tDayCount::numOfDays30_360_ISDA( tDate Start, tDate End,

    bool bCouponOnLastDayOfFeb ) const



{

    int nDay1, nDay2;



    if( Start.day() == 31 )

        nDay1 = 30;

    else

        nDay1 = Start.day();



    if( End.day() == 31 && nDay1 == 30 )

        nDay2 = 30;

    else

        nDay2 = End.day();



    return 360L * ( End.year() - Start.year() ) +

            30L * ( End.month() - Start.month() ) +

                  ( nDay2 - nDay1 );

}





//

//   n u m O f D a y s 3 0 _ 3 6 0 _ P S A

//



long tDayCount::numOfDays30_360_PSA( tDate Start, tDate End,

    bool bCouponOnLastDayOfFeb ) const



{

    int nDay1, nDay2;



    if( Start.day() == 31 || Start.isLastDayInFebruary() )

        nDay1 = 30;

    else

        nDay1 = Start.day();



    if( End.day() == 31 && nDay1 == 30 )

        nDay2 = 30;

    else

        nDay2 = End.day();



    return 360L * ( End.year() - Start.year() ) +

            30L * ( End.month() - Start.month() ) +

                  ( nDay2 - nDay1 );

}





//

//   n u m O f D a y s 3 0 _ 3 6 0 _ S I A

//



long tDayCount::numOfDays30_360_SIA( tDate Start, tDate End,

    bool bCouponOnLastDayOfFeb ) const



{

    int nDay1, nDay2;



    if( Start.day() == 31 || 

        Start.isLastDayInFebruary() && bCouponOnLastDayOfFeb ) {

        nDay1 = 30;

    }

    else {

        nDay1 = Start.day();

    }



    if( End.day() == 31 && nDay1 == 30 )

        nDay2 = 30;

    else

        nDay2 = End.day();



    return 360L * ( End.year() - Start.year() ) +

            30L * ( End.month() - Start.month() ) +

                  ( nDay2 - nDay1 );

}





//

//   n u m O f D a y s 3 0 E _ 3 6 0

//



long tDayCount::numOfDays30E_360( tDate Start, tDate End,

    bool bCouponOnLastDayOfFeb ) const



{

    int nDay1, nDay2;



    if( Start.day() == 31 )

        nDay1 = 30;

    else

        nDay1 = Start.day();



    if( End.day() == 31 )

        nDay2 = 30;

    else

        nDay2 = End.day();



    return 360L * ( End.year() - Start.year() ) +

            30L * ( End.month() - Start.month() ) +

                  ( nDay2 - nDay1 );

}





//

//   f r a c t i o n A C T _ A C T

//



double tDayCount::fractionACT_ACT( tDate Start, tDate End,

    tDate PeriodStart, tDate PeriodEnd,

    bool bCouponOnLastDayOfFeb ) const



{

    return numOfDaysACT_ACT( Start, End, bCouponOnLastDayOfFeb ) /

           (double) ( PeriodEnd - PeriodStart );

}





//

//   f r a c t i o n A C T _ 3 6 5

//



double tDayCount::fractionACT_365( tDate Start, tDate End,

    bool bCouponOnLastDayOfFeb ) const



{

    return numOfDaysACT_365( Start, End, bCouponOnLastDayOfFeb ) / 365.0;

}





//

//   f r a c t i o n A C T _ 3 6 5 _ 2 5

//



double tDayCount::fractionACT_365_25( tDate Start, tDate End,

    bool bCouponOnLastDayOfFeb ) const



{

    return numOfDaysACT_365_25( Start, End, bCouponOnLastDayOfFeb ) / 365.25;

}





//

//   f r a c t i o n A C T _ 3 6 5 _ N L

//



double tDayCount::fractionACT_365_NL( tDate Start, tDate End,

    bool bCouponOnLastDayOfFeb ) const



{

        // Exclude leap days in the numerator.

    return numOfDaysACT_365_NL( Start, End, bCouponOnLastDayOfFeb ) / 365.0;

}





//

//   f r a c t i o n A C T _ 3 6 5 _ I S D A

//



double tDayCount::fractionACT_365_ISDA( tDate Start, tDate End,

    bool bCouponOnLastDayOfFeb ) const



{

        // The counting functions include the first day.

    tDate d = Start + 1;



    int k = d.numOfDaysInLeapYearsTo( End );

    int l = d.numOfDaysInNonLeapYearsTo( End );



    MTG_ASSERT( k + l == End - Start );



    return  k / 366.0 + l / 365.0;

}





//

//   f r a c t i o n A C T _ 3 6 0

//



double tDayCount::fractionACT_360( tDate Start, tDate End,

    bool bCouponOnLastDayOfFeb ) const



{

    return numOfDaysACT_360( Start, End, bCouponOnLastDayOfFeb ) / 360.0;

}





//

//   f r a c t i o n 3 0 _ 3 6 0 _ I S D A

//



double tDayCount::fraction30_360_ISDA( tDate Start, tDate End,

    bool bCouponOnLastDayOfFeb ) const



{

    return numOfDays30_360_ISDA( Start, End, bCouponOnLastDayOfFeb ) / 360.0;

}





//

//   f r a c t i o n 3 0 _ 3 6 0 _ P S A

//



double tDayCount::fraction30_360_PSA( tDate Start, tDate End,

    bool bCouponOnLastDayOfFeb ) const



{

    return numOfDays30_360_PSA( Start, End, bCouponOnLastDayOfFeb ) / 360.0;

}





//

//   f r a c t i o n 3 0 _ 3 6 0 _ S I A

//



double tDayCount::fraction30_360_SIA( tDate Start, tDate End,

    bool bCouponOnLastDayOfFeb ) const



{

    return numOfDays30_360_SIA( Start, End, bCouponOnLastDayOfFeb ) / 360.0;

}





//

//   f r a c t i o n 3 0 E _ 3 6 0

//



double tDayCount::fraction30E_360( tDate Start, tDate End,

    bool bCouponOnLastDayOfFeb ) const



{

    return numOfDays30E_360( Start, End, bCouponOnLastDayOfFeb ) / 360.0;

}





//

//   t D a y C o u n t

//



tDayCount::tDayCount()



{

    m_nPolicy = xACT_365;;

}





//

//   t D a y C o u n t

//



tDayCount::tDayCount( tPolicy nPolicy )



{

    m_nPolicy = nPolicy;

}





//

//   t D a y C o u n t

//



tDayCount::tDayCount( const tDayCount& Count )



{

    m_nPolicy = Count.m_nPolicy;

}





//

//   o p e r a t o r =

//



tDayCount& tDayCount::operator=( tDayCount Count )



{

    if( &Count != this )

        m_nPolicy = Count.m_nPolicy;

    return *this;

}





//

//   o p e r a t o r =

//



tDayCount& tDayCount::operator=( tPolicy nPolicy )



{

    m_nPolicy = nPolicy;

    return *this;

}





//

//   f r a c t i o n

//



double tDayCount::fraction( tDate Start, tDate End,

    tDate PeriodStart, tDate PeriodEnd,

    bool bCouponOnLastDayOfFeb ) const



{

    double f;

    int nSign = 1;



    if( Start > End ) {

        tDate T = Start;

        Start = End;

        End = T;

        nSign = -1;

    }



    switch( m_nPolicy ) {

        case xACT_ACT:

            MTG_ASSERT( PeriodStart < PeriodEnd );



            f = fractionACT_ACT( Start, End,

                    PeriodStart, PeriodEnd, bCouponOnLastDayOfFeb );

            break;



        case xACT_365:

            f = fractionACT_365( Start, End, bCouponOnLastDayOfFeb );

            break;



        case xACT_365_25:

            f = fractionACT_365_25( Start, End, bCouponOnLastDayOfFeb );

            break;



        case xACT_365_NL:

            f = fractionACT_365_NL( Start, End, bCouponOnLastDayOfFeb );

            break;



        case xACT_365_ISDA:

            f = fractionACT_365_ISDA( Start, End, bCouponOnLastDayOfFeb );

            break;



        case xACT_360:

            f = fractionACT_360( Start, End, bCouponOnLastDayOfFeb );

            break;



        case x30_360_ISDA:

            f = fraction30_360_ISDA( Start, End, bCouponOnLastDayOfFeb );

            break;



        case x30_360_PSA:

            f = fraction30_360_PSA( Start, End, bCouponOnLastDayOfFeb );

            break;



        case x30_360_SIA:

            f = fraction30_360_SIA( Start, End, bCouponOnLastDayOfFeb );

            break;



        case x30E_360:

            f = fraction30E_360( Start, End, bCouponOnLastDayOfFeb );

            break;



        default:

            throw tException( INTERNAL_ERROR );

    }



    return f * nSign;

}





//

//   f r a c t i o n

//



double tDayCount::fraction( tDate Start, tDate End,

    bool bCouponOnLastDayOfFeb ) const



{

        // To advance the start date by one year works

        // for ACT/ACT, according to Stigum, p. 49. For

        // all other policies, it's irrelevant.



    tDate d;

    int nSign = 1;



    if( Start > End ) {

        tDate T = Start;

        Start = End;

        End = T;

        nSign = -1;

    }



    if( Start.month() == 2 && Start.day() == 29 )

        d.set( 2, 28, Start.year() + 1 );

    else

        d.set( Start.day(), Start.month(), Start.year() + 1 );



    return nSign * fraction( Start, End, Start, d, bCouponOnLastDayOfFeb );

}





//

//   n u m O f D a y s

//



long tDayCount::numOfDays( tDate Start, tDate End,

    bool bCouponOnLastDayOfFeb ) const



{

    long n;

    int nSign = 1;



    if( Start > End ) {

        tDate T = Start;

        Start = End;

        End = T;

        nSign = -1;

    }



    switch( m_nPolicy ) {

        case xACT_ACT:

            n = numOfDaysACT_ACT( Start, End, bCouponOnLastDayOfFeb );

            break;



        case xACT_365:

            n = numOfDaysACT_365( Start, End, bCouponOnLastDayOfFeb );

            break;



        case xACT_365_25:

            n = numOfDaysACT_365_25( Start, End, bCouponOnLastDayOfFeb );

            break;



        case xACT_365_NL:

            n = numOfDaysACT_365_NL( Start, End, bCouponOnLastDayOfFeb );

            break;



        case xACT_365_ISDA:

            n = numOfDaysACT_365_ISDA( Start, End, bCouponOnLastDayOfFeb );

            break;



        case xACT_360:

            n = numOfDaysACT_360( Start, End, bCouponOnLastDayOfFeb );

            break;



        case x30_360_ISDA:

            n = numOfDays30_360_ISDA( Start, End, bCouponOnLastDayOfFeb );

            break;



        case x30_360_PSA:

            n = numOfDays30_360_PSA( Start, End, bCouponOnLastDayOfFeb );

            break;



        case x30_360_SIA:

            n = numOfDays30_360_SIA( Start, End, bCouponOnLastDayOfFeb );

            break;



        case x30E_360:

            n = numOfDays30E_360( Start, End, bCouponOnLastDayOfFeb );

            break;



        default:

            throw tException( INTERNAL_ERROR );

    }



    return n * nSign;

}





//

//   o p e r a t o r = =

//



bool tDayCount::operator==( tDayCount Count ) const



{

    return m_nPolicy == Count.m_nPolicy;

}





//

//   o p e r a t o r = =

//



bool tDayCount::operator==( tPolicy nPolicy ) const



{

    return m_nPolicy == nPolicy;

}





//

//   p a r s e

//



tRetCode tDayCount::parse( tParser& Parser )



{

    tRetCode nRet;



    switch( Parser.curToken() ) {

        case xTokDayCount :

        case xTokACT_ACT :

        case xTokACT_365 :

        case xTokACT_365_25 :

        case xTokACT_365_ISDA :

        case xTokACT_365_NL :

        case xTokACT_360 :

        case xTok30_360_ISDA :

        case xTok30_360_PSA :

        case xTok30_360_SIA :

        case xTok30E_360 :

            if( Parser.curToken() == xTokDayCount &&

                ( nRet = Parser.readToken() ) != OK ) {

                return nRet;

            }



            switch( Parser.curToken() ) {

                case xTokACT_ACT :

                    *this = DayCountACT_ACT;      break;

                case xTokACT_365 :

                    *this = DayCountACT_365;      break;

                case xTokACT_365_25 :

                    *this = DayCountACT_365_25;   break;

                case xTokACT_365_ISDA :

                    *this = DayCountACT_365_ISDA; break;

                case xTokACT_365_NL :

                    *this = DayCountACT_365_NL;   break;

                case xTokACT_360 :

                    *this = DayCountACT_360;      break;

                case xTok30_360_ISDA :

                    *this = DayCount30_360_ISDA;  break;

                case xTok30_360_PSA :

                    *this = DayCount30_360_PSA;   break;

                case xTok30_360_SIA :

                    *this = DayCount30_360_SIA;   break;         

                case xTok30E_360 :

                    *this = DayCount30E_360;      break;



                default :

                    return Parser.setError( INVALID_KEYWORD );

            }



            if( ( nRet = Parser.readToken() ) != OK )

                return nRet;

            break;



        default :

            return GO;

    }

  

    return OK;

}





//

//   o p e r a t o r < <

//



ostream& operator<<( ostream& Out, const tDayCount& DayCount )



{

    Out << Token2Text( xTokDayCount ) << " ";



    switch( DayCount.m_nPolicy ) {

        case tDayCount::xACT_ACT:

            Out << Token2Text( xTokACT_ACT );

            break;



        case tDayCount::xACT_365:

            Out << Token2Text( xTokACT_365 );

            break;



        case tDayCount::xACT_365_25:

            Out << Token2Text( xTokACT_365_25 );

            break;



        case tDayCount::xACT_365_NL:

            Out << Token2Text( xTokACT_365_NL );

            break;



        case tDayCount::xACT_365_ISDA:

            Out << Token2Text( xTokACT_365_ISDA );

            break;



        case tDayCount::xACT_360:

            Out << Token2Text( xTokACT_360 );

            break;



        case tDayCount::x30_360_ISDA:

            Out << Token2Text( xTok30_360_ISDA );

            break;



        case tDayCount::x30_360_PSA:

            Out << Token2Text( xTok30_360_PSA );

            break;



        case tDayCount::x30_360_SIA:

            Out << Token2Text( xTok30_360_SIA );

            break;



        case tDayCount::x30E_360:

            Out << Token2Text( xTok30E_360 );

            break;



        default:

            throw tException( INTERNAL_ERROR );

    }



    return Out;

}



MTG_END_NAMESPACE





//#define _TEST

#if defined(_TEST)



#if defined(_WIN32)

    #include <conio.h>

#else

    #define getche getchar

#endif



MTG_USING_NAMESPACE



//

//   m a i n

//



void main( int argc, char *argv[] )



{

    printf( "\nTest tDayCount\n\n" );



    char sBuf[256];

    int cCmd;



    tDate D1, D2;



    bool bGo = true;



    while( bGo ) { 

        printf( "Date<1> Date<2> <C>ount E<x>it: " );

        cCmd = getche();

        printf( "\n" );



        switch( cCmd ) {

            case '1' :

                printf( "Date1: " );

                gets( sBuf );

                if( D1.set( sBuf ) == 0 )

                    printf( "Format error\n" );

                break;



            case '2' :

                printf( "Date2: " );

                gets( sBuf );

                if( D2.set( sBuf ) == 0 )

                    printf( "Format error\n" );

                break;



            case 'c' :

            case 'C' :

                printf( "ACT/ACT     : %ld\n",

                    DayCountACT_ACT.numOfDays( D1, D2 ) );

                printf( "ACT/365     : %ld\n",

                    DayCountACT_365.numOfDays( D1, D2 ) );

                printf( "ACT/365.25  : %ld\n",

                    DayCountACT_365_25.numOfDays( D1, D2 ) );

                printf( "ACT/365 NL  : %ld\n",

                    DayCountACT_365_NL.numOfDays( D1, D2 ) );

                printf( "ACT/365 ISDA: %ld\n",

                    DayCountACT_365_ISDA.numOfDays( D1, D2 ) );

                printf( "ACT/360     : %ld\n",

                    DayCountACT_360.numOfDays( D1, D2 ) );

                printf( "30/360 ISDA : %ld\n",

                    DayCount30_360_ISDA.numOfDays( D1, D2 ) );

                printf( "30/360 PSA  : %ld\n",

                    DayCount30_360_PSA.numOfDays( D1, D2 ) );

                printf( "30/360 SIA  : %ld\n",

                    DayCount30_360_SIA.numOfDays( D1, D2 ) );

                printf( "30E/360     : %ld\n",

                    DayCount30E_360.numOfDays( D1, D2 ) );

                break;



            case 'x' :

            case 'X' :

                bGo = false;

                break;

        }

    }



#if ! defined(_WIN32)

    printf( "\n" );

#endif

}



#endif