// Copyright 1999, 2002 Robert Buff

// Contact: http://robertbuff.com/uvm

//

// This file is part of Mtg-Book.

//

// Mtg-Book is free software; you can redistribute it and/or modify

// it under the terms of the GNU General Public License as published

// by the Free Software Foundation; either version 2 of the License,

// or (at your option) any later version.

//

// Mtg-Book is distributed in the hope that it will be useful,

// but WITHOUT ANY WARRANTY; without even the implied warranty of

// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the

// GNU General Public License for more details.

//

// You should have received a copy of the GNU General Public License

// along with Mtg-Book; if not, write to the 

//

// Free Software Foundation, Inc.

// 59 Temple Place, Suite 330

// Boston, MA 02111-1307

// USA



#include "MtgIncl.h"

#include "MtgEvaluate.h"

#include "MtgFDEngine.h"

#include "MtgLattice.h"

#include "MtgMCEngine.h"

#include "MtgModel.h"

#include "MtgOptimizer.h"

#include "MtgParser.h"

#include "MtgPathSpace.h"

#include "MtgPortfolio.h"

#include "MtgScenario.h"



MTG_BEGIN_NAMESPACE





//

//   p a r s e P a r a m

//



tRetCode tEvaluate::parseParam( tParser& Parser, tParseInfoStub& Info )



{

    tRetCode nRet;

    tObject* pObj;



    tParseInfo& I = static_cast<tParseInfo&>( Info );



    switch( Parser.curToken() ) {

        case xTokLattice :

            if( m_pLattice != 0 )

                return Parser.setError( "lattice already defined" );

            if( m_pPathSpace != 0 ) {

                return Parser.setError(

                    "only lattice OR path space can be defined" );

            }

            if( ( nRet = Parser.readToken() ) != OK )

                return nRet;

            if( Parser.curToken() != xTokId )

                return Parser.setError( INVALID_KEYWORD );

            if( ( pObj = Parser.findObject() ) == 0 ) 

                return Parser.setError( NOT_FOUND );

            if( ( m_pLattice = dynamic_cast<tLattice*>( pObj ) ) == 0 )

                return Parser.setError( "lattice expected" );

            if( ( nRet = Parser.readToken() ) != OK )

                return nRet;

            break;



        case xTokPathSpace :

            if( m_pPathSpace != 0 )

                return Parser.setError( "path space already defined" );

            if( m_pLattice != 0 ) {

                return Parser.setError(

                    "only lattice OR path space can be defined" );

            }

            if( ( nRet = Parser.readToken() ) != OK )

                return nRet;

            if( Parser.curToken() != xTokId )

                return Parser.setError( INVALID_ID );

            if( ( pObj = Parser.findObject() ) == 0 ) 

                return Parser.setError( NOT_FOUND );

            if( ( m_pPathSpace = dynamic_cast<tPathSpace*>( pObj ) ) == 0 )

                return Parser.setError( "path space expected" );

            if( ( nRet = Parser.readToken() ) != OK )

                return nRet;

            break;



        case xTokModel :

            if( m_pModel != 0 )

                return Parser.setError( "model already defined" );

            if( ( nRet = Parser.readToken() ) != OK )

                return nRet;

            if( Parser.curToken() != xTokId )

                return Parser.setError( INVALID_ID );

            if( ( pObj = Parser.findObject() ) == 0 ) 

                return Parser.setError( NOT_FOUND );

            if( ( m_pModel = dynamic_cast<tModel*>( pObj ) ) == 0 )

                return Parser.setError( "model expected" );

            if( ( nRet = Parser.readToken() ) != OK )

                return nRet;

            break;



        case xTokPortfolio :

            if( m_pPortfolio != 0 )

                return Parser.setError( "portfolio already defined" );

            if( ( nRet = Parser.readToken() ) != OK )

                return nRet;

            if( Parser.curToken() != xTokId )

                return Parser.setError( INVALID_ID );

            if( ( pObj = Parser.findObject() ) == 0 ) 

                return Parser.setError( NOT_FOUND );

            if( ( m_pPortfolio = dynamic_cast<tPortfolio*>( pObj ) ) == 0 )

                return Parser.setError( "portfolio expected" );

            if( ( nRet = Parser.readToken() ) != OK )

                return nRet;

            break;



        case xTokScenario :

            if( m_pScenario != 0 )

                return Parser.setError( "scenario already defined" );

            if( ( nRet = Parser.readToken() ) != OK )

                return nRet;

            if( Parser.curToken() != xTokId )

                return Parser.setError( INVALID_ID );

            if( ( pObj = Parser.findObject() ) == 0 ) 

                return Parser.setError( NOT_FOUND );

            if( ( m_pScenario = dynamic_cast<tScenario*>( pObj ) ) == 0 )

                return Parser.setError( "scenario expected" );

            if( ( nRet = Parser.readToken() ) != OK )

                return nRet;

            break;



        case xTokOptimizer :

            if( m_pOptimizer != 0 )

                return Parser.setError( "optimizer already defined" );

            if( ( nRet = Parser.readToken() ) != OK )

                return nRet;

            if( Parser.curToken() != xTokId )

                return Parser.setError( INVALID_ID );

            if( ( pObj = Parser.findObject() ) == 0 ) 

                return Parser.setError( NOT_FOUND );

            if( ( m_pOptimizer = dynamic_cast<tOptimizer*>( pObj ) ) == 0 )

                return Parser.setError( "optimizer expected" );

            if( ( nRet = Parser.readToken() ) != OK )

                return nRet;

            break;



        case xTokCurve :

            if( ( nRet = Parser.readToken() ) != OK ||

                ( nRet = m_CurveContainer.parse( Parser ) ) != OK ) {

                return nRet;

            }

            break;



        case xTokImage :

            if( ( nRet = Parser.readToken() ) != OK ||

                ( nRet = m_ImageContainer.parse( Parser ) ) != OK ) {

                return nRet;

            }

            break;



        case xTokAccuracy :

            if( I.m_bAccuracy )

                return Parser.setError( "accuracy already defined" );

            if( ( nRet = Parser.readToken() ) != OK )

                return nRet;

            switch( Parser.curToken() ) {

                case xTokExact :  m_nAccuracy = xExact;  break;

                case xTokHigh :   m_nAccuracy = xHigh;   break;

                case xTokMedium : m_nAccuracy = xMedium; break;

                case xTokLow :    m_nAccuracy = xLow;    break;

                default :

                    return Parser.setError( INVALID_ID );

            }

            if( ( nRet = Parser.readToken() ) != OK )

                return nRet;

            break;



        default :

            return super::parseParam( Parser, Info );

    }



    return OK;

}





//

//   p a r s e P o s t f i x

//



tRetCode tEvaluate::parsePostfix( tParser& Parser, tParseInfoStub& Info )



{

    tRetCode nRet;



    tParseInfo& I = static_cast<tParseInfo&>( Info );



    if( m_pLattice == 0 && m_pPathSpace == 0 )

        return Parser.setError( MISSING_LATTICE );

    if( m_pPortfolio == 0 )

        return Parser.setError( MISSING_PORTFOLIO );



    if( m_pLattice != 0 ) {

        if( m_pModel == 0 )

            return Parser.setError( MISSING_MODEL );

        if( m_pScenario == 0 )

            return Parser.setError( MISSING_SCENARIO );



        if( ( nRet = m_pModel->createEngine( m_pScenario, 

                        m_pFDEngine, m_nAccuracy ) ) != OK ) {

            return Parser.setError( nRet );

        }



        m_pFDEngine->setLattice( m_pLattice );

        m_pFDEngine->setScenario( m_pScenario );

        m_pFDEngine->setPortfolio( m_pPortfolio );

        

        if( m_pOptimizer != 0 )

            m_pFDEngine->setOptimizer( m_pOptimizer );



        m_pFDEngine->setCurveContainer( m_CurveContainer );

        m_pFDEngine->setImageContainer( m_ImageContainer );

    }

    else {

        MTG_ASSERT( m_pPathSpace != 0 );



        if( I.m_bAccuracy ) 

            return Parser.setError( "accuracy not supported in Monte Carlo" );

        if( m_pScenario != 0 )

            return Parser.setError( "scenarios not supported in Monte Carlo" );



        if( m_pModel == 0 ) {

            if( ! m_pPathSpace->hasModel() )

                return Parser.setError( MISSING_MODEL );

            m_pModel = &m_pPathSpace->model();

        }

        else {

            if( ! m_pPathSpace->modelMatches( *m_pModel ) )

                return Parser.setError( "model does not match path space" );

        }



        if( ! m_pPathSpace->isCovered( m_pPortfolio->maturityDate() ) ||

            ! m_pPathSpace->isCovered( m_pPortfolio->settlement() ) ) {

            return Parser.setError( OUT_OF_RANGE );

        }

            

            // sometimes the initial dates must match; check that later



        if( ( nRet = m_pModel->createEngine( m_pMCEngine ) ) != OK )

            return Parser.setError( nRet );



        m_pMCEngine->setPathSpace( m_pPathSpace );

        m_pMCEngine->setPortfolio( m_pPortfolio );



        if( m_pOptimizer != 0 )

            m_pMCEngine->setOptimizer( m_pOptimizer );



        m_pMCEngine->setCurveContainer( m_CurveContainer );

        m_pMCEngine->setImageContainer( m_ImageContainer );

    }

    

    return super::parsePostfix( Parser, Info );

}





//

//   t E v a l u a t e

//



tEvaluate::tEvaluate()



{

    m_pLattice = 0;

    m_pPathSpace = 0;

    m_pModel = 0;

    m_pPortfolio = 0;

    m_pScenario = 0;

    m_pOptimizer = 0;

    m_pFDEngine = 0;

    m_pMCEngine = 0;

    m_nAccuracy = xExact;

    setProfile();

}





//

//   ~ t F a c t o r

//



tEvaluate::~tEvaluate()



{

    if( m_pFDEngine != 0 )

        delete m_pFDEngine;

    if( m_pMCEngine != 0 )

        delete m_pMCEngine;

}





//

//   f i n a l i z e

//



tRetCode tEvaluate::finalize()



{

    return super::finalize();

}





//

//   r u n

//



tRetCode tEvaluate::run()



{

    MTG_ASSERT( isFinalized() );



    if( m_pFDEngine != 0 ) {

        tWithProfile::copyTo( m_pFDEngine );

        return m_pFDEngine->run();

    }



    MTG_ASSERT( m_pMCEngine != 0 );

    return m_pMCEngine->run();

}





//

//   d i r t y T o t a l

//



double tEvaluate::dirtyTotal() const



{

    if( m_pFDEngine == 0 ) {

        MTG_ASSERT( m_pMCEngine != 0 );

        return m_pMCEngine->dirtyTotal();

    }

    return m_pFDEngine->total();

}





//

//   c l e a n T o t a l

//



double tEvaluate::cleanTotal() const



{

    if( m_pFDEngine == 0 ) {

        MTG_ASSERT( m_pMCEngine != 0 );

        return m_pMCEngine->cleanTotal();

    }

    return m_pFDEngine->total();

}





//

//   d e l t a

//



double tEvaluate::delta() const



{

    if( m_pFDEngine == 0 )

        throw tException( NOT_AVAILABLE );

    return m_pFDEngine->delta();

}





//

//   g a m m a

//



double tEvaluate::gamma() const



{

    if( m_pFDEngine == 0 )

        throw tException( NOT_AVAILABLE );

    return m_pFDEngine->gamma();

}





//

//   g r a d i e n t

//



void tEvaluate::gradient( tHeap<double>& Gradient ) const



{

    if( m_pFDEngine == 0 )

        throw tException( NOT_AVAILABLE );

    m_pFDEngine->gradient( Gradient );

}





//

//   f r o n t

//



void tEvaluate::front( tHeap2<double>& Front ) const



{

    if( m_pFDEngine == 0 )

        throw tException( NOT_AVAILABLE );

    m_pFDEngine->front( Front );

}





//

//   n u m O f T a s k s

//



int tEvaluate::numOfTasks() const



{

    if( m_pFDEngine == 0 )

        throw tException( NOT_AVAILABLE );

    return m_pFDEngine->numOfTasks();

}





//

//   p a r s e

//



tRetCode tEvaluate::parse( tParser& Parser, tSystem& System, tObject*& pObj )



{

    tRetCode nRet;

    tEvaluate* pEvaluate;

    tParseInfo Info;



    Info.m_bAccuracy = false;



    if( ( nRet = Parser.scanBeginOfObj() ) != OK )

        return nRet;



    pEvaluate = new tEvaluate;

    pEvaluate->setSystem( System );



    if( ( nRet = pEvaluate->super::parse( Parser, &Info ) ) != OK ) {

        delete pEvaluate;

        return nRet;

    }



    pObj = pEvaluate;

    return OK;

}



MTG_END_NAMESPACE